package ru.yandex.direct.grid.processing.service.campaign;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.CampaignFetchedFieldsResolver;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdCampaignAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdAbSegmentRetargetingConditionIds;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStatus;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStatusCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignAgencyInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignManagerInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignMeasurer;
import ru.yandex.direct.grid.model.campaign.GdCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdCampaignStatus;
import ru.yandex.direct.grid.model.campaign.GdCpmBannerCampaign;
import ru.yandex.direct.grid.model.campaign.GdDynamicCampaign;
import ru.yandex.direct.grid.model.campaign.GdInternalDistribCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdInternalFreeCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaign;
import ru.yandex.direct.grid.model.campaign.GdPriceCampaign;
import ru.yandex.direct.grid.model.campaign.GdSmartCampaign;
import ru.yandex.direct.grid.model.campaign.GdTextCampaign;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.facelift.GdCampaignAdditionalData;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.model.entity.campaign.GdCampWithAgencyInfo;
import ru.yandex.direct.grid.model.entity.campaign.GdCampWithManagerInfo;
import ru.yandex.direct.grid.model.entity.promoextension.GdEntityWithPromoExtension;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignFilter;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignSourceAvailability;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsCommonTimeTarget;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsContainer;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignsContext;
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatusData;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmBudgetLimit;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmBudgetLimitPayload;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignBrandSurveyCheckRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignBrandSurveyCheckResponseItem;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmCampaignDayBudgetLimitsRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmPriceCampaignBrandSurveyCheckRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmPriceCampaignGeo;
import ru.yandex.direct.grid.processing.model.campaign.GdMultipleCampaignsBrandSurveyCheckRequest;
import ru.yandex.direct.grid.processing.model.campaign.GdMultipleCampaignsBrandSurveyCheckResponse;
import ru.yandex.direct.grid.processing.model.campaign.GdPayForConversionInfo;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdCampaignVcard;
import ru.yandex.direct.grid.processing.model.placement.GdPiPage;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageForClient;
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtension;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionWithMetrikaFlag;
import ru.yandex.direct.grid.processing.model.tag.GdTag;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.campaign.container.CampaignsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.promoextension.PromoExtensionsDataLoader;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.processing.service.autooverdraft.converter.AutoOverdraftDataConverter.toClientAutoOverdraftInfo;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toCampaignsCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toGdCampaignsContext;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;

/**
 * Сервис, возвращающий данные о кампаниях клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class CampaignsGraphQlService {
    public static final String CAMPAIGNS_RESOLVER_NAME = "campaigns";
    public static final String WALLETS_RESOLVER_NAME = "wallets";
    static final String LAST_CHANGED_VCARD_ID_RESOLVER_NAME = "lastChangedVcardId";
    static final String HAS_RUNNING_UNMODERATED_ADS_RESOLVER_NAME = "hasRunningUnmoderatedAds";
    static final String ADS_COUNT_RESOLVER_NAME = "adsCount";
    static final String GROUPS_COUNT_RESOLVER_NAME = "groupsCount";
    static final String PAY_FOR_CONVERSION_INFO_RESOLVER_NAME = "payForConversionInfo";
    static final String CAMPAIGN_TAGS = "tags";
    static final String CAMPAIGNS_COMMON_TIME_TARGET = "campaignsCommonTimeTarget";
    static final String CUSTOM_GEO_EXPANDED_RESOLVER_NAME = "customGeoExpanded";
    static final String HAS_DEFAULT_AD_GROUP_RESOLVER_NAME = "hasDefaultAdGroup";
    static final String CPM_CAMPAIGN_BUDGET_LIMITS = "cpmCampaignBudgetLimits";
    static final String CURRENT_CPM_CAMPAIGN_BUDGET_LIMITS = "currentCpmCampaignBudgetLimits";
    static final String PRICE_PACKAGE_RESOLVER_NAME = "pricePackage";
    static final String ALLOWED_PAGES_RESOLVER_NAME = "allowedPages";
    static final String CPM_CAMPAIGN_BRAND_SURVEY_CONDITIONS = "cpmCampaignBrandSurveyConditions";
    static final String CPM_PRICE_CAMPAIGN_BRAND_SURVEY_CONDITIONS = "cpmPriceCampaignBrandSurveyConditions";
    static final String MULTIPLE_CAMPAIGNS_BRAND_SURVEY_CONDITIONS = "multipleCampaignsBrandSurveyConditions";
    static final String STRATEGY_LEARNING_STATUS = "strategyLearningStatus";
    static final String CAMPAIGN_HREF_IS_TURBO = "isTurbo";
    static final String CAMPAIGN_HAS_BANNERS_STOPPED_BY_MONITORING = "hasBannersStoppedByMonitoring";

    private final GridCacheService gridCacheService;
    private final CampaignInfoService campaignsInfoService;
    private final GridShortenerService gridShortenerService;
    private final GridValidationService gridValidationService;
    private final CpmPriceCampaignInfoService cpmPriceCampaignInfoService;
    private final PromoExtensionsDataLoader promoExtensionsDataLoader;
    private final FeatureService featureService;

    @Autowired
    public CampaignsGraphQlService(GridCacheService gridCacheService,
                                   CampaignInfoService campaignsInfoService,
                                   GridShortenerService gridShortenerService,
                                   GridValidationService gridValidationService,
                                   CpmPriceCampaignInfoService cpmPriceCampaignInfoService,
                                   FeatureService featureService,
                                   PromoExtensionsDataLoader promoExtensionsDataLoader) {
        this.gridCacheService = gridCacheService;
        this.campaignsInfoService = campaignsInfoService;
        this.gridShortenerService = gridShortenerService;
        this.gridValidationService = gridValidationService;
        this.cpmPriceCampaignInfoService = cpmPriceCampaignInfoService;
        this.featureService = featureService;
        this.promoExtensionsDataLoader = promoExtensionsDataLoader;
    }

    /**
     * GraphQL подзапрос. Получает информацию о кампаниях клиента, полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAMPAIGNS_RESOLVER_NAME)
    public GdCampaignsContext getCampaigns(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignsContainer input) {
        gridValidationService.validateGdCampaignsContainer(input);
        Long clientId = client.getInfo().getId();

        if (input.getFilterKey() != null) {
            GdCampaignFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    ClientId.fromLong(clientId),
                    GdCampaignFilter.class,
                    () -> new GdCampaignFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdStatRequirements statRequirements = normalizeStatRequirements(input.getStatRequirements(),
                context.getInstant(), input.getFilter().getRecommendations());
        input.setStatRequirements(statRequirements);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());

        // пытаемся прочитать из кеша нужный диапазон строк
        CampaignsCacheRecordInfo recordInfo = toCampaignsCacheRecordInfo(clientId, input);
        Optional<GdCampaignsContext> res = gridCacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        // в кеше данные не нашлись, читаем из mysql/YT
        ImmutableList<GdiCampaign> campaigns = campaignsInfoService.getAllCampaigns(ClientId.fromLong(clientId));
        CampaignFetchedFieldsResolver campaignFetchedFieldsResolver = context.getFetchedFieldsReslover().getCampaign();
        List<GdCampaign> rowsetFull = campaignsInfoService
                .getFilteredCampaigns(client.getInfo(), context.getOperator(), campaigns, input,
                        context.getInstant(), campaignFetchedFieldsResolver);


        boolean cpcAndCpmOnOneGridEnabled =
                featureService.isEnabledForClientId(ClientId.fromLong(clientId), CPC_AND_CPM_ON_ONE_GRID_ENABLED);
        GdCampaignsContext campaignsContext = toGdCampaignsContext(rowsetFull, cpcAndCpmOnOneGridEnabled);

        campaignsContext.setFilter(input.getFilter());

        // сохраняем в кеш, если надо, и возвращаем нужный диапазон строк в результате
        return gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, campaignsContext, rowsetFull, range,
                campaignFetchedFieldsResolver.getCacheKey());
    }

    @GraphQLQuery(name = "commonVcard")
    public CompletableFuture<GdCampaignVcard> getCommonVcard(@GraphQLContext GdTextCampaign campaign) {
        return campaignsInfoService.getCommonVcard(campaign);
    }

    @GraphQLQuery(name = "commonVcard")
    public CompletableFuture<GdCampaignVcard> getCommonVcard(@GraphQLContext GdDynamicCampaign campaign) {
        return campaignsInfoService.getCommonVcard(campaign);
    }

    @GraphQLQuery(name = STRATEGY_LEARNING_STATUS)
    public CompletableFuture<GdConversionStrategyLearningStatusData> getStrategyLearningStatus(@GraphQLContext GdMobileContentCampaign campaign) {
        return campaignsInfoService.getStrategyLearningStatus(campaign, List.of());
    }

    @GraphQLQuery(name = STRATEGY_LEARNING_STATUS)
    public CompletableFuture<GdConversionStrategyLearningStatusData> getStrategyLearningStatus(@GraphQLContext GdTextCampaign campaign) {
        return campaignsInfoService.getStrategyLearningStatus(campaign, campaign.getMeaningfulGoals());
    }

    @GraphQLQuery(name = STRATEGY_LEARNING_STATUS)
    public CompletableFuture<GdConversionStrategyLearningStatusData> getStrategyLearningStatus(@GraphQLContext GdDynamicCampaign campaign) {
        return campaignsInfoService.getStrategyLearningStatus(campaign, campaign.getMeaningfulGoals());
    }

    @GraphQLQuery(name = STRATEGY_LEARNING_STATUS)
    public CompletableFuture<GdConversionStrategyLearningStatusData> getStrategyLearningStatus(@GraphQLContext GdSmartCampaign campaign) {
        return campaignsInfoService.getStrategyLearningStatus(campaign, campaign.getMeaningfulGoals());
    }

    @GraphQLQuery(name = "abSegmentRetargetingCondition")
    public CompletableFuture<GdRetargetingConditionWithMetrikaFlag> getRetargetingCondition(
            @GraphQLContext GdAbSegmentRetargetingConditionIds campaign) {
        Long retargetingConditionId = campaign.getAbSegmentRetargetingConditionId();
        if (retargetingConditionId == null) {
            return CompletableFuture.completedFuture(null);
        }

        return campaignsInfoService.getRetargetingCondition(retargetingConditionId);
    }

    @GraphQLQuery(name = "abSegmentStatisticRetargetingCondition")
    public CompletableFuture<GdRetargetingConditionWithMetrikaFlag> getStatisticRetargetingCondition(
            @GraphQLContext GdAbSegmentRetargetingConditionIds campaign) {
        Long retargetingConditionId = campaign.getAbSegmentStatisticRetargetingConditionId();
        if (retargetingConditionId == null) {
            return CompletableFuture.completedFuture(null);
        }
        return campaignsInfoService.getRetargetingCondition(retargetingConditionId);
    }

    @GraphQLQuery(name = "agencyInfo")
    public CompletableFuture<GdCampaignAgencyInfo> getGdCampaignAgencyInfo(
            @GraphQLContext GdCampWithAgencyInfo campaign) {
        return campaignsInfoService.getGdCampaignAgencyInfo(campaign);
    }

    @GraphQLQuery(name = "managerInfo")
    public CompletableFuture<GdCampaignManagerInfo> getGdCampaignManagerInfo(
            @GraphQLContext GdCampWithManagerInfo campaign) {
        return campaignsInfoService.getGdCampaignManagerInfo(campaign);
    }

    @GraphQLQuery(name = "brandSurveyStatus")
    public CompletableFuture<GdBrandSurveyStatus> getGdCampaignBrandSurveyStatus(
            @GraphQLContext GdBrandSurveyStatusCampaign campaign) {
        return campaignsInfoService.getGdBrandSurveyStatus(campaign);
    }

    @GraphQLQuery(name = "promoExtension")
    public CompletableFuture<GdPromoExtension> getCampaignPromoExtension(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdEntityWithPromoExtension campaign) {
        if (campaign.getPromoExtensionId() == null) {
            return null;
        }
        return promoExtensionsDataLoader.getPromoExtensionById(campaign.getPromoExtensionId());
    }

    @GraphQLQuery(name = "bannerGeoLegalFlags")
    public CompletableFuture<Set<String>> getBannerGeoLegalFlagsInAggrStatus(
            @GraphQLContext GdCampaignAggregatedStatusInfo statusInfo) {
        return campaignsInfoService.getBannerGeoLegalFlagsInAggrStatus(statusInfo);
    }

    @GraphQLQuery(name = CURRENT_CPM_CAMPAIGN_BUDGET_LIMITS)
    public GdCpmBudgetLimit getCurrentCpmCampaignBudgetLimits(
            @GraphQLContext GdCpmBannerCampaign campaign) {
        return campaignsInfoService.getCurrentCpmCampaignBudgetLimits(campaign);
    }

    /**
     * GraphQL подзапрос. Получает информацию о кошельках клиента
     *
     * @param client клиент, кошельки которого нужно получить
     */
    @GraphQLNonNull
    @GraphQLQuery(name = WALLETS_RESOLVER_NAME)
    public List<@GraphQLNonNull GdWallet> getWallets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        List<GdiBaseCampaign> campaigns = campaignsInfoService.getAllBaseCampaigns(clientId);

        return campaignsInfoService.extractWalletsList(
                context.getOperator(), client.getInfo(),
                toClientAutoOverdraftInfo(client.getInfo()), campaigns, context.getInstant());
    }

    /**
     * GraphQL подзапрос. Получает последний измененный vcard_id для кампаний
     */
    @GraphQLQuery(name = LAST_CHANGED_VCARD_ID_RESOLVER_NAME)
    public CompletableFuture<Long> getLastChangedVcardId(@GraphQLContext GdTextCampaign gdTextCampaign) {
        return campaignsInfoService.getLastChangedVcardId(gdTextCampaign.getId());
    }

    /**
     * GraphQL подзапрос. Получает последний измененный vcard_id для кампаний
     */
    @GraphQLQuery(name = LAST_CHANGED_VCARD_ID_RESOLVER_NAME)
    public CompletableFuture<Long> getLastChangedVcardId(@GraphQLContext GdDynamicCampaign gdDynamicCampaign) {
        return campaignsInfoService.getLastChangedVcardId(gdDynamicCampaign.getId());
    }

    /**
     * GraphQL подзапрос. Есть ли в кампании активные баннеры в статусе постмодерации
     */
    @GraphQLQuery(name = HAS_RUNNING_UNMODERATED_ADS_RESOLVER_NAME)
    public CompletableFuture<Boolean> getHasRunningUnmoderatedAds(@GraphQLContext GdCampaignStatus gdCampaignStatus) {
        return campaignsInfoService.getHasRunningUnmoderatedAds(gdCampaignStatus.getCampaignId());
    }

    /**
     * GraphQL подзапрос. Получает кол-во объявлений для кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = ADS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getAdsCount(@GraphQLContext GdInternalDistribCampaignTruncated gdInternalDistribCampaign) {
        return campaignsInfoService.getAdsCount(gdInternalDistribCampaign.getId());
    }

    /**
     * GraphQL подзапрос. Получает кол-во объявлений для кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = ADS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getAdsCount(@GraphQLContext GdInternalFreeCampaignTruncated gdInternalFreeCampaign) {
        return campaignsInfoService.getAdsCount(gdInternalFreeCampaign.getId());
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп для кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = GROUPS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Long> getGroupsCount(@GraphQLContext GdCampaign campaign) {
        return campaignsInfoService.getGroupsCount(campaign.getId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = PAY_FOR_CONVERSION_INFO_RESOLVER_NAME)
    public CompletableFuture<GdPayForConversionInfo> getPayForConversionStatInfo(@GraphQLRootContext GridGraphQLContext context,
                                                                                 @GraphQLContext GdCampaign campaign) {

        return campaignsInfoService.getPayForConversionStatInfo(context, campaign.getId(), campaign.getStrategy());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CUSTOM_GEO_EXPANDED_RESOLVER_NAME)
    public GdCpmPriceCampaignGeo getCpmPriceCampaignGeo(@GraphQLContext GdPriceCampaign cpmPriceCampaign) {
        return cpmPriceCampaignInfoService.geoExpandedToGdCpmPriceCampaignGeo(cpmPriceCampaign);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = HAS_DEFAULT_AD_GROUP_RESOLVER_NAME)
    public CompletableFuture<Boolean> getHasDefaultAdGroup(@GraphQLContext GdPriceCampaign cpmPriceCampaign) {
        return cpmPriceCampaignInfoService.getHasDefaultAdGroup(cpmPriceCampaign.getId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = PRICE_PACKAGE_RESOLVER_NAME)
    public CompletableFuture<GdPricePackageForClient> getPricePackage(
            @GraphQLContext GdPriceCampaign cpmPriceCampaign) {
        return cpmPriceCampaignInfoService.getPricePackage(cpmPriceCampaign.getPricePackageId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CPM_CAMPAIGN_BUDGET_LIMITS)
    public GdCpmBudgetLimitPayload getCpmCampaignBudgetLimits(
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCpmCampaignDayBudgetLimitsRequest request) {
        return campaignsInfoService.getBudgetLimits(client.getInfo().getShard(), client.getInfo().getWorkCurrency(),
                request);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CAMPAIGNS_COMMON_TIME_TARGET)
    public GdTimeTarget campaignsCommonTimeTarget(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignsCommonTimeTarget gdCampaignsCommonTimeTarget) {
        return campaignsInfoService.getCommonTimeTarget(context.getSubjectUser().getClientId(),
                gdCampaignsCommonTimeTarget.getCampaignIds());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CAMPAIGN_TAGS)
    public List<@GraphQLNonNull GdTag> getTags(@GraphQLRootContext GridGraphQLContext context,
                                               @GraphQLContext GdCampaign campaigns) {
        return campaignsInfoService.getTags(context.getQueriedClient(), campaigns);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CPM_CAMPAIGN_BRAND_SURVEY_CONDITIONS)
    public List<@GraphQLNonNull GdCpmCampaignBrandSurveyCheckResponseItem> getBrandSurveyConditions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCpmCampaignBrandSurveyCheckRequest request) {
        return campaignsInfoService.getBrandSurveyConditions(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), request);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CPM_PRICE_CAMPAIGN_BRAND_SURVEY_CONDITIONS)
    public List<@GraphQLNonNull GdCpmCampaignBrandSurveyCheckResponseItem> getBrandSurveyConditions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCpmPriceCampaignBrandSurveyCheckRequest request) {
        return campaignsInfoService.getBrandSurveyConditions(context.getSubjectUser().getClientId(), request);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = MULTIPLE_CAMPAIGNS_BRAND_SURVEY_CONDITIONS)
    public GdMultipleCampaignsBrandSurveyCheckResponse getBrandSurveyConditions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMultipleCampaignsBrandSurveyCheckRequest request) {
        return campaignsInfoService.getBrandSurveyConditions(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), request);
    }

    /**
     * GraphQL подзапрос. Возвращает {@code true}, если ссылка на кампании является ссылкой на турбу
     */
    @GraphQLQuery(name = CAMPAIGN_HREF_IS_TURBO)
    public CompletableFuture<Boolean> getCampaignHrefIsTurbo(@GraphQLContext GdCampaignAdditionalData additionalData) {
        return campaignsInfoService.getCampaignHrefIsTurbo(additionalData);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CAMPAIGN_HAS_BANNERS_STOPPED_BY_MONITORING)
    public Boolean getCampaignHasBannersStoppedByMonitoring(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdCampaign campaign) {
        return campaignsInfoService
                .getHasBannersStoppedByMonitoring(context.getQueriedClient().getShard(), campaign.getId());
    }

    @GraphQLQuery(name = "spentToday")
    public BigDecimal getSpentToday(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdCampaign campaign) {
        return campaignsInfoService.getSpentToday(campaign);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "domains")
    public CompletableFuture<List<@GraphQLNonNull String>> getDomains(@GraphQLContext GdTextCampaign gdTextCampaign) {
        return campaignsInfoService.getDomains(gdTextCampaign);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "availableCampaignSources")
    public List<GdCampaignSourceAvailability> getAvailableCampaignSources(
                @GraphQLContext GdClient client) {
        return campaignsInfoService.getAvailableCampaignSources(client);
    }

    @GraphQLQuery(name = "lastRmpCampaignSource")
    public GdCampaignSource getlastRmpCampaignSource(
            @GraphQLContext GdClient client) {
        return campaignsInfoService.getLastRmpCampaignSource(client.getInfo());
    }

    @GraphQLQuery(name = ALLOWED_PAGES_RESOLVER_NAME)
    public List<@GraphQLNonNull GdPiPage> getAllowedPages(@GraphQLRootContext GridGraphQLContext context,
                                                          @GraphQLContext GdCampaign campaign) {
        return campaignsInfoService.getAllowedPages(context.getQueriedClient().getShard(), campaign.getId());
    }

    @GraphQLQuery(name = "measurers")
    public List<@GraphQLNonNull GdCampaignMeasurer> getCampaignMeasurer(@GraphQLRootContext GridGraphQLContext context,
                                                                        @GraphQLContext GdCampaign campaign) {
        return campaignsInfoService.getCampaignMeasurer(context.getQueriedClient().getShard(), campaign.getId());
    }
}
