package ru.yandex.direct.grid.processing.service.campaign

import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignServiceUtils.PERCENT_MULTIPLIER
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatus
import ru.yandex.direct.utils.NumberUtils
import java.math.BigDecimal
import java.math.RoundingMode
import java.time.LocalDate
import java.time.temporal.ChronoUnit

abstract class ConversionStrategyLearningData(val strategyGoalId: Long?,
                                              val stats: GdiCampaignStats,
                                              val lastDayStats: GdiCampaignStats) {

    open val MAX_DAYS_COUNT_FOR_LEARNING = 7
    open val CONVERSIONS_COUNT_FOR_LEARNED_STATUS = 10
    open val MIN_DAYS_FOR_LEARNED_STATUS = 5

    fun lastDayConversionCount(): Long = lastDayStats.conversionCount()

    fun conversionCount(): Long = stats.conversionCount()

    fun hasRevenue(): Boolean = stats.revenue() > 0 || strategyGoalId == CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID

    fun goalsCrr(): BigDecimal? =
        NumberUtils.ifNotZero(stats.revenue()) { revenue ->
            // Расходы и доходы по целям в одной единице исчисления.
            stats.stat.cost
                ?.multiply(BigDecimal.valueOf(PERCENT_MULTIPLIER))
                ?.divide(BigDecimal.valueOf(revenue), 1, RoundingMode.HALF_UP)
        }


    open fun getStatus(now: LocalDate): GdConversionStrategyLearningStatus {
        val daysFromRestart = ChronoUnit.DAYS.between(restartOrStartDate(), now)
        if (isRevenueModelStrategy()) {
            if (daysFromRestart > MAX_DAYS_COUNT_FOR_LEARNING
                && (conversionCount() < CONVERSIONS_COUNT_FOR_LEARNED_STATUS || !hasRevenue())) {
                return GdConversionStrategyLearningStatus.NOT_LEARNED
            }
        } else {
            if (daysFromRestart > MAX_DAYS_COUNT_FOR_LEARNING && lastDayConversionCount() == 0L && conversionCount() < CONVERSIONS_COUNT_FOR_LEARNED_STATUS) {
                return GdConversionStrategyLearningStatus.NOT_LEARNED
            }
        }
        return if (daysFromRestart > MIN_DAYS_FOR_LEARNED_STATUS && isSufficientStatistics()) {
            GdConversionStrategyLearningStatus.LEARNED
        } else GdConversionStrategyLearningStatus.LEARNING
    }


    private fun isSufficientStatistics(): Boolean {
        return conversionCount() >= CONVERSIONS_COUNT_FOR_LEARNED_STATUS && (!isRevenueModelStrategy() || hasRevenue())
    }

    abstract fun isRevenueModelStrategy(): Boolean

    abstract fun restartOrStartDate(): LocalDate

    abstract fun startDate(): LocalDate

    companion object {
        private fun GdiCampaignStats.conversionCount(): Long = this.goalStats?.sumOf { it.goals ?: 0 } ?: 0

        private fun GdiCampaignStats.revenue(): Long = this.goalStats?.sumOf { it.revenue ?: 0 } ?: 0
    }
}
