package ru.yandex.direct.grid.processing.service.campaign;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.dataloader.BatchLoaderEnvironment;
import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiAggregatorGoal;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdCampaignAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdConversionStrategyLearningStatusData;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.time.LocalDate.now;
import static org.apache.commons.lang3.ObjectUtils.max;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;

@Component
@ParametersAreNonnullByDefault
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class ConversionStrategyLearningStatusDataLoader
        extends GridBatchingDataLoader<GdCampaign, GdConversionStrategyLearningStatusData> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConversionStrategyLearningStatusDataLoader.class);

    public static final int MAX_CAMPAIGN_COUNT_FOR_STRATEGY_LEARNING_STATUS = 10;
    public static final int MAX_DAYS_COUNT_FOR_CAMPAIGN_LEARNING = 7;
    static final LocalDate CONVERSION_STRATEGY_LEARNING_START_DATE = LocalDate.parse("2020-12-03");

    private static final Set<GdStrategyType> STRATEGY_TYPES_WITH_LEARNING_STATUS = Set.of(
            GdStrategyType.WEEK_BUDGET,
            GdStrategyType.AVG_CPA,
            GdStrategyType.AVG_CPA_PER_CAMP,
            GdStrategyType.AVG_CPA_PER_FILTER,
            GdStrategyType.OPTIMIZE_CONVERSIONS,
            GdStrategyType.OPTIMIZE_INSTALLS,
            GdStrategyType.CRR
    );

    public ConversionStrategyLearningStatusDataLoader(
            GridContextProvider gridContextProvider,
            GridCampaignService gridCampaignService,
            GdStrategyExtractorFacade gdStrategyExtractorFacade,
            FeatureService featureService) {

        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(gridCampaignService,
                gdStrategyExtractorFacade, featureService));
    }

    private MappedBatchLoaderWithContext<GdCampaign, GdConversionStrategyLearningStatusData> getBatchLoadFunction(
            GridCampaignService gridCampaignService,
            GdStrategyExtractorFacade gdStrategyExtractorFacade,
            FeatureService featureService
    ) {
        return (campaigns, environment) -> {
            GridGraphQLContext context = environment.getContext();
            User subjectUser = context.getSubjectUser();
            checkNotNull(subjectUser, "subjectUser should be set in gridContext");
            checkArgument(campaigns.size() < MAX_CAMPAIGN_COUNT_FOR_STRATEGY_LEARNING_STATUS);
            boolean isEnabledCrrStrategy = featureService.isEnabledForClientId(subjectUser.getClientId(),
                    FeatureName.SUPPORT_CONVERSION_STRATEGY_LEARNING_STATUS_FOR_CRR);
            return CompletableFuture.completedFuture(getStatusDataByCampaign(campaigns, environment, gridCampaignService, gdStrategyExtractorFacade, isEnabledCrrStrategy));
        };
    }

    static Map<GdCampaign, GdConversionStrategyLearningStatusData> getStatusDataByCampaign(
            Set<GdCampaign> campaigns,
            BatchLoaderEnvironment environment,
            GridCampaignService gridCampaignService,
            GdStrategyExtractorFacade gdStrategyExtractorFacade,
            boolean isEnabledCrrStrategy
    ) {
        LOGGER.info("Number of campaigns got to get status data: {}", campaigns.size());

        Set<GdCampaign> filteredCampaigns = StreamEx.of(campaigns)
                //для черновиков и модерируемых кампаний не показываем статус обучения
                .remove(ConversionStrategyLearningStatusDataLoader::needToHideLearningStatus)
                .filter(campaign -> isStrategyWithSupportLearningStatus(gdStrategyExtractorFacade,
                        campaign.getStrategy(), isEnabledCrrStrategy))
                .collect(Collectors.toSet());
        LOGGER.info("Number of filtered campaigns: {}", filteredCampaigns.size());
        if (filteredCampaigns.isEmpty()) {
            LOGGER.info("Calls to gridCampaignService: 0");
            return new HashMap<>();
        }

        Map<Long, Set<Long>> meaningfulGoalsByCampaignId = extractMeaningfulGoalsByCampaignId(environment);

        return getStatusDataByCampaignForFilteredCampaigns(filteredCampaigns, meaningfulGoalsByCampaignId, gridCampaignService, gdStrategyExtractorFacade);
    }

    private static Map<GdCampaign, GdConversionStrategyLearningStatusData> getStatusDataByCampaignForFilteredCampaigns(
            Set<GdCampaign> campaigns,
            Map<Long, Set<Long>> meaningfulGoalsByCampaignId,
            GridCampaignService gridCampaignService,
            GdStrategyExtractorFacade gdStrategyExtractorFacade
    ) {
        Map<Long, Long> strategyGoalIdByCampaignId = StreamEx.of(campaigns)
                .mapToEntry(Function.identity(), GdCampaign::getStrategy)
                .mapValues(gdStrategyExtractorFacade::extractStrategyGoalId)
                .nonNullValues()
                .mapKeys(GdCampaign::getId)
                .toMap();

        LocalDate now = now();
        Map<Long, LocalDateRange> dateRangeByCampaignId = StreamEx.of(campaigns)
                .mapToEntry(GdCampaign::getId, campaign -> calculateStatisticStartDate(
                        gdStrategyExtractorFacade, campaign, now))
                .mapValues(from -> new LocalDateRange().withFromInclusive(from).withToInclusive(now))
                .toMap();

        Map<Long, List<GdiAggregatorGoal>> aggregatorGoalsByCampaignId =
                getAggregatorGoalsByCampaignId(gdStrategyExtractorFacade, campaigns, meaningfulGoalsByCampaignId);
        Map<Long, GdiCampaignStats> campaignStatByCampaignId =
                gridCampaignService.getCampaignGoalStatsWithOptimizationForDifferentDateRanges(
                        dateRangeByCampaignId, strategyGoalIdByCampaignId, aggregatorGoalsByCampaignId);

        Map<Long, LocalDateRange> lastDayDateRangeByCampaignId = listToMap(strategyGoalIdByCampaignId.keySet(),
                Function.identity(),
                campaignId -> new LocalDateRange().withFromInclusive(now.minusDays(1)).withToInclusive(now));
        Map<Long, GdiCampaignStats> campaignLastDayStatByCampaignId = gridCampaignService.
                getCampaignGoalStatsWithOptimizationForDifferentDateRanges(lastDayDateRangeByCampaignId,
                        strategyGoalIdByCampaignId, aggregatorGoalsByCampaignId);

        Map<Long, GdCampaign> campaignById = listToMap(campaigns, GdCampaign::getId);

        return EntryStream.of(campaignStatByCampaignId)
                .mapToValue((campaignId, stat) -> getStatusData(
                        gdStrategyExtractorFacade,
                        stat,
                        campaignById.get(campaignId),
                        campaignLastDayStatByCampaignId.get(campaignId),
                        now))
                .mapKeys(campaignById::get)
                .toMap();
    }

    /**
     * Цель 13 -- все кц, содержит в себе несколько других целей. Чтобы корректно, посчитать статистику по этой цели
     * получаем мапу для каждой кампании список целей состоящих из нескольких целей
     */
    private static Map<Long, List<GdiAggregatorGoal>> getAggregatorGoalsByCampaignId(GdStrategyExtractorFacade gdStrategyExtractorFacade, Set<GdCampaign> campaigns, Map<Long, Set<Long>> meaningfulGoalsByCampaignId) {
        return StreamEx.of(campaigns)
                .mapToEntry(Function.identity(), GdCampaign::getStrategy)
                .mapValues(gdStrategyExtractorFacade::extractStrategyGoalId)
                .nonNullValues()
                .filterValues(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID::equals)
                .mapToValue((campaign, goalId) -> List.of(new GdiAggregatorGoal().withId(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)
                        .withSubGoalIds(List.copyOf(meaningfulGoalsByCampaignId.get(campaign.getId())))))
                .mapKeys(GdCampaign::getId)
                .toMap();
    }

    private static Map<Long, Set<Long>> extractMeaningfulGoalsByCampaignId(BatchLoaderEnvironment environment) {
        return EntryStream.of(environment.getKeyContexts())
                .mapKeys(campaign -> (GdCampaign) campaign)
                .mapKeys(GdCampaign::getId)
                .mapValues(context -> context != null ? (Set<Long>) context : Collections.<Long>emptySet())
                .toMap();
    }

    private static boolean isStrategyWithSupportLearningStatus(GdStrategyExtractorFacade gdStrategyExtractorFacade,
                                                               GdCampaignFlatStrategy gdCampaignFlatStrategy,
                                                               boolean isEnabledForCrrStrategy) {
        Long goalId = gdStrategyExtractorFacade.extractStrategyGoalId(gdCampaignFlatStrategy);
        GdStrategyType strategyType = gdCampaignFlatStrategy.getStrategyType();
        var isEnabledCrr = !strategyType.equals(GdStrategyType.CRR) || isEnabledForCrrStrategy;
        return goalId != null && STRATEGY_TYPES_WITH_LEARNING_STATUS.contains(strategyType) && isEnabledCrr;
    }

    private static boolean needToHideLearningStatus(GdCampaign campaign) {
        GdSelfStatusEnum status = Optional.ofNullable(campaign)
                .map(GdCampaign::getAggregatedStatusInfo)
                .map(GdCampaignAggregatedStatusInfo::getStatus)
                .orElse(GdSelfStatusEnum.DRAFT);
        return status == GdSelfStatusEnum.DRAFT || status == GdSelfStatusEnum.ON_MODERATION;
    }

    private static GdConversionStrategyLearningStatusData getStatusData(
            GdStrategyExtractorFacade gdStrategyExtractorFacade,
            GdiCampaignStats stats,
            GdCampaign campaign,
            GdiCampaignStats campaignLastDayStat,
            LocalDate now) {
        GdiGoalStats goalStats = stats.getGoalStats().stream().findAny().get();

        Long conversionCount = goalStats.getGoals();
        BigDecimal costPerAction = goalStats.getCostPerAction();
        BigDecimal conversionRate =
                goalStats.getConversionRate().divide(DECIMAL_MULT, 2, RoundingMode.HALF_UP);

        LocalDate restartDate = gdStrategyExtractorFacade.extractStrategyLastRestartTime(campaign.getFlatStrategy());
        LocalDate statisticStartDate = calculateStatisticStartDate(gdStrategyExtractorFacade, campaign, now);
        var strategyGoalId = gdStrategyExtractorFacade.extractStrategyGoalId(campaign.getStrategy());
        var learningData = new ConversionStrategyWithCampaignLearningData(
                campaign,
                restartDate,
                strategyGoalId,
                stats,
                campaignLastDayStat
        );

        var status = learningData.getStatus(now);

        return new GdConversionStrategyLearningStatusData()
                .withRestartDate(getRestartDate(learningData, now))
                .withStatisticCalculationStartTime(statisticStartDate)
                .withConversionCount(conversionCount)
                .withAverageConversion(costPerAction)
                .withConversionRate(conversionRate)
                .withCostRevenueRatio(learningData.goalsCrr())
                .withStatus(status);
    }

    static LocalDate calculateStatisticStartDate(GdStrategyExtractorFacade gdStrategyExtractorFacade,
                                                 GdCampaign campaign, LocalDate now) {
        LocalDate lastRestartDate =
                gdStrategyExtractorFacade.extractStrategyLastRestartTime(campaign.getFlatStrategy());

        LocalDate startDate = campaign.getStartDate();
        LocalDate maxDaysForCampaignLearningAgo = now.minusDays(MAX_DAYS_COUNT_FOR_CAMPAIGN_LEARNING);

        return max(lastRestartDate, startDate, maxDaysForCampaignLearningAgo);
    }

    @Nullable
    static LocalDate getRestartDate(ConversionStrategyLearningData data,
                                    LocalDate now) {
        return data.startDate().isAfter(now) ? null :
                max(data.restartOrStartDate(), CONVERSION_STRATEGY_LEARNING_START_DATE);
    }
}
