package ru.yandex.direct.grid.processing.service.campaign;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.CpmPriceCampaignService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.grid.model.campaign.GdPriceCampaign;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmPriceCampaignGeo;
import ru.yandex.direct.grid.processing.model.campaign.GdCpmPriceCampaignGeoNode;
import ru.yandex.direct.grid.processing.model.placement.GdRegionDesc;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageForClient;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignsHasDefaultAdGroupDataLoader;
import ru.yandex.direct.grid.processing.service.campaign.loader.CampaignsPricePackageDataLoader;
import ru.yandex.direct.grid.processing.util.GeoTreeUtils;

import static ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageValidator.REGION_TYPE_REGION;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CpmPriceCampaignInfoService {

    private final PricePackageService pricePackageService;
    private final CampaignsHasDefaultAdGroupDataLoader campaignsHasDefaultAdGroupDataLoader;
    private final CampaignsPricePackageDataLoader campaignsPricePackageDataLoader;
    private final RegionDescriptionLocalizer localizer;
    private final CpmPriceCampaignService cpmPriceCampaignService;
    private final ClientGeoService clientGeoService;

    @Autowired
    public CpmPriceCampaignInfoService(PricePackageService pricePackageService,
                                       CampaignsHasDefaultAdGroupDataLoader campaignsHasDefaultAdGroupDataLoader,
                                       CampaignsPricePackageDataLoader campaignsPricePackageDataLoader,
                                       RegionDescriptionLocalizer localizer,
                                       CpmPriceCampaignService cpmPriceCampaignService,
                                       ClientGeoService clientGeoService) {
        this.pricePackageService = pricePackageService;
        this.campaignsHasDefaultAdGroupDataLoader = campaignsHasDefaultAdGroupDataLoader;
        this.campaignsPricePackageDataLoader = campaignsPricePackageDataLoader;
        this.localizer = localizer;
        this.cpmPriceCampaignService = cpmPriceCampaignService;
        this.clientGeoService = clientGeoService;
    }

    GdCpmPriceCampaignGeo geoExpandedToGdCpmPriceCampaignGeo(GdPriceCampaign campaign) {
        List<Long> geoExpanded = campaign.getFlightTargetingsSnapshot().getGeoExpanded();
        var pricePackage = pricePackageService.getPricePackage(campaign.getPricePackageId());
        List<Long> geo = geoExpanded;
        if (isGeoTypeRegionOnly(pricePackage)) {//Только выбранный регион
            geo = pricePackage.getTargetingsCustom().getGeo();
            //если это РК с дефольной группой и она уже создана,
            // то используем дефолтную группу в качестве начала дерева
            if (pricePackage.needDefaultAdGroup()) {
                var defaultAdGroup = cpmPriceCampaignService
                        .getDefaultPriceSalesAdGroupByCampaignId(campaign.getId());
                if (defaultAdGroup != null) {
                    geo = clientGeoService.convertForWeb(defaultAdGroup.getGeo(), pricePackageService.getGeoTree());
                    geo.addAll(StreamEx.of(geo)
                            .filter(regionId -> regionId != null && regionId > 0)
                            .flatMap(regionId -> pricePackageService.getGeoTree().getAllChildren(regionId).stream())
                            .toList());
                }
            }
        }
        List<GdCpmPriceCampaignGeoNode> geoExpandedTrees = mapList(geo, this::buildRegionTree);
        if (isGeoTypeRegionOnly(pricePackage)) {
            geoExpandedTrees = filterOnlySelectedGeo(geoExpandedTrees, listToSet(geo));
        }
        sortGeoForFrontend(geoExpandedTrees);
        return new GdCpmPriceCampaignGeo()
                .withGeoExpandedTrees(geoExpandedTrees);
    }

    private List<GdCpmPriceCampaignGeoNode> filterOnlySelectedGeo(List<GdCpmPriceCampaignGeoNode> geoExpandedTrees,
                                                                  Set<Long> geo) {
        if (geoExpandedTrees == null) {
            return null;
        }
        // Для прайсового видео эта ручка должна строить дерево на основе ids из geo.
        // Дерево должно быть без дочерних элементов и состоять только из тех регионов, которые есть в geo
        List<GdCpmPriceCampaignGeoNode> tree = StreamEx.of(geoExpandedTrees)
                .filter(it -> geo.contains(it.getId()))//Оставляем из детей только нужные
                .map(it -> it.withInner(filterOnlySelectedGeo(it.getInner(), geo)))
                .toList();
        //дети отсутвующих нод могут перевесится на верхний уровень
        tree.addAll(StreamEx.of(geoExpandedTrees)
                .filter(it -> !geo.contains(it.getId()))
                .flatMap(it -> {
                    var children = filterOnlySelectedGeo(it.getInner(), geo);
                    return children == null ? Stream.empty() : children.stream();
                }).toList());

        Set<Long> geoInner = StreamEx.of(tree)
                .flatMap(it -> geoInner(it.getInner()).stream())
                .toSet();
        tree = StreamEx.of(tree)//Если эта нода есть у кого-то в дочерних, то на верхнем уровне дублировать не нужно
                .filter(it -> !geoInner.contains(it.getId()))
                .toList();
        return tree.isEmpty() ? null : tree;// фронт предпочитает null в листах, а не пустой массив
    }

    private Set<Long> geoInner(List<GdCpmPriceCampaignGeoNode> childs) {
        Set<Long> set = new HashSet<>();
        if (childs != null) {
            for (GdCpmPriceCampaignGeoNode inner :childs) {
                set.add(inner.getId());
                set.addAll(geoInner(inner.getInner()));
            }
        }
        return set;
    }

    private boolean isGeoTypeRegionOnly(PricePackage pricePackage) {
        Integer geoType = Optional.ofNullable(pricePackage)
                .map(PricePackage::getTargetingsCustom)
                .map(TargetingsCustom::getGeoType)
                .orElse(-1);
        return geoType.equals(REGION_TYPE_REGION);
    }

    private GdCpmPriceCampaignGeoNode buildRegionTree(Long parentRegionId) {
        return pricePackageService.getGeoTreeConverter().buildRegionTree(parentRegionId,
                id -> {
                    GdRegionDesc localized = localizer.localize(id, pricePackageService.getGeoTree());
                    return new GdCpmPriceCampaignGeoNode()
                            .withId(id)
                            .withName(localized.getName());
                },
                (geoNode, children) -> {
                    sortGeoForFrontend(children);
                    // фронт предпочитает null в листах, а не пустой массив
                    geoNode.setInner(children.size() == 0 ? null : children);
                });
    }

    private static void sortGeoForFrontend(List<GdCpmPriceCampaignGeoNode> geoExpandedTrees) {
        GeoTreeUtils.sortGeoForFrontend(geoExpandedTrees,
                GdCpmPriceCampaignGeoNode::getId,
                GdCpmPriceCampaignGeoNode::getName);
    }

    CompletableFuture<Boolean> getHasDefaultAdGroup(Long campaignId) {
        return campaignsHasDefaultAdGroupDataLoader.get().load(campaignId);
    }

    CompletableFuture<GdPricePackageForClient> getPricePackage(Long pricePackageId) {
        return campaignsPricePackageDataLoader.get().load(pricePackageId);
    }
}
