package ru.yandex.direct.grid.processing.service.campaign;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatus;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatusInfo;
import ru.yandex.direct.core.entity.campaign.service.TimeTargetStatusService;
import ru.yandex.direct.core.entity.campaign.service.WalletUtils;
import ru.yandex.direct.core.entity.timetarget.service.GeoTimezoneMappingService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.grid.model.campaign.GdCampaignPrimaryStatus;
import ru.yandex.direct.grid.model.campaign.GdCampaignPrimaryStatusDesc;
import ru.yandex.direct.grid.model.campaign.GdCampaignStatus;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignDelayedOperation;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStatusBsSynced;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;

import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignStatusModerate;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Сервис, предназначенный для расчета вычисляемых статусов кампаний
 */
@Service
@ParametersAreNonnullByDefault
public class GridCampaignAggregationFieldsService {
    /**
     * Settings.pm: сколько секунд после остановки кампании, с неё нельзя переносить средства
     */
    private static final int TRANSFER_DELAY_AFTER_STOP = 30 * 60;
    /**
     * Предел, при котором мы предлагаем пользователям оплатит кампанию
     */
    private static final BigDecimal NEED_PAYMENT_BORDER = BigDecimal.valueOf(0.1);

    private final TimeTargetStatusService timeTargetStatusService;
    private final GeoTimezoneMappingService geoTimezoneMappingService;

    @Autowired
    public GridCampaignAggregationFieldsService(
            TimeTargetStatusService timeTargetStatusService,
            GeoTimezoneMappingService geoTimezoneMappingService) {
        this.timeTargetStatusService = timeTargetStatusService;
        this.geoTimezoneMappingService = geoTimezoneMappingService;
    }

    /**
     * Рассчет остатка средств на кампании
     *
     * @param campaignSum      - сумма на кампании
     * @param campaignSumSpent - сумма расходов на кампании
     * @param wallet           - кошелек. Может быть не задан, если у клиента нет общего счета.
     */
    public static BigDecimal calcSumTotal(BigDecimal campaignSum, BigDecimal campaignSumSpent,
                                          @Nullable GdWallet wallet) {
        return WalletUtils.calcSumTotalIncludingOverdraft(campaignSum, campaignSumSpent, ifNotNull(wallet,
                        GdWallet::getSum),
                ifNotNull(wallet, GdWallet::getAutoOverdraftAddition));
    }

    /**
     * Получить информацию о текущем статусе кампании
     *
     * @param campaign       описание кампании
     * @param currentInstant момент времени, на который идет расчет
     */
    GdCampaignStatus extractStatus(GdiCampaign campaign, @Nullable GdWallet wallet,
                                   Instant currentInstant) {
        TimeTargetStatusInfo timeTargetStatus = timeTargetStatusService.getTimeTargetStatus(campaign.getTimeTarget(),
                geoTimezoneMappingService.getRegionIdByTimezoneId(campaign.getTimezoneId()),
                currentInstant);
        CampaignStatusModerate moderationStatus = calcCampaignEffectiveModerationStatus(campaign);
        BigDecimal sumTotal = calcSumTotal(campaign.getSum(), campaign.getSumSpent(), wallet);

        Pair<GdCampaignPrimaryStatus, GdCampaignPrimaryStatusDesc> primaryStatusPair = calcPrimaryStatus(
                campaign, wallet, sumTotal, timeTargetStatus, moderationStatus, currentInstant);
        boolean isDraft = isCampaignDraft(campaign, moderationStatus);

        return new GdCampaignStatus()
                .withCampaignId(campaign.getId())
                .withArchived(campaign.getArchived())
                .withDraft(isDraft)
                .withOver(isCampaignOver(primaryStatusPair.getRight()))
                .withWaitingForArchiving(isCampaignWaitingForArchiving(campaign))
                .withWaitingForUnArchiving(isCampaignWaitingForUnarchiving(campaign))
                .withReadOnly(!campaign.getActions().getCanEdit() || campaign.getArchived())
                .withNeedsNewPayment(isCampaignNeedsNewPayment(campaign, sumTotal)) //Todo(pashkus) DIRECT-82306
                .withMoneyBlocked(campaign.getMoneyBlocked())
                .withBudgetLimitationStopTime(getDayBudgetPauseTime(campaign, primaryStatusPair))
                .withModerationStatus(toGdCampaignStatusModerate(moderationStatus))
                .withWaitingForPayment(isCampaignWaitingForPayment(campaign, moderationStatus))
                .withTimeTargetStatus(timeTargetStatus)
                .withPrimaryStatus(primaryStatusPair.getLeft())
                .withActivating(isCampaignActivating(campaign, sumTotal, currentInstant))
                .withAllowDomainMonitoring(isAllowDomainMonitoringForCampaign(campaign))
                .withPrimaryStatusDesc(primaryStatusPair.getRight());
    }

    //Следим за приостановкой от БК, если выполнены следующие условия: кампания подходящего типа,
    //слежение на кампании включено клиентом, кампания не заархивирована и кампания не удалена
    static boolean isAllowDomainMonitoringForCampaign(GdiCampaign campaign) {
        return CampaignTypeKinds.ALLOW_DOMAIN_MONITORING.contains(campaign.getType())
                && campaign.getHasSiteMonitoring()
                && !campaign.getArchived()
                && !campaign.getEmpty();
    }

    private LocalDateTime getDayBudgetPauseTime(GdiCampaign campaign,
                                                Pair<GdCampaignPrimaryStatus, GdCampaignPrimaryStatusDesc> primaryStatusPair) {

        return (primaryStatusPair.getLeft() == GdCampaignPrimaryStatus.TEMPORARILY_PAUSED &&
                primaryStatusPair.getRight() == GdCampaignPrimaryStatusDesc.IS_PAUSED_BY_DAY_BUDGET) ?
                campaign.getDayBudgetStopTime() : null;
    }

    private boolean isCampaignOver(GdCampaignPrimaryStatusDesc statusDesc) {
        return statusDesc == GdCampaignPrimaryStatusDesc.IS_OVER;
    }

    private boolean isCampaignActivating(GdiCampaign campaign, BigDecimal sumTotal, Instant currentInstant) {//Todo
        // (pashkus) переписать активизацию
        // + учитывать кампании под кошельком.
        boolean statusActiveExpected = sumTotal.compareTo(BigDecimal.ZERO) > 0 && campaign.getShowing();
        return sumTotal.compareTo(Currencies.EPSILON) > 0 &&
                (campaign.getStatusBsSynced() != GdiCampaignStatusBsSynced.YES ||
                        campaign.getActive() != statusActiveExpected ||
                        transferAfterStopTimeDisallowed(campaign.getStopTime(), currentInstant)) &&
                campaign.getStatusPostModerate() == CampaignStatusPostmoderate.ACCEPTED &&
                // костыль для случая, когда все баннеры остановлены и из БК пришел statusActive = 'No', но
                // statusShow = 'Yes'
                !(!campaign.getHasActiveBanners() && campaign.getActive()
                        && campaign.getStatusBsSynced() == GdiCampaignStatusBsSynced.YES);
    }

    private boolean transferAfterStopTimeDisallowed(@Nullable LocalDateTime stopTime, Instant currentInstant) {
        if (stopTime == null) {
            return false;
        }
        long stopEpochSecond = stopTime.atZone(MSK).toEpochSecond();
        long currentEpochSecond = currentInstant.getEpochSecond();
        return (currentEpochSecond - stopEpochSecond) < TRANSFER_DELAY_AFTER_STOP;
    }

    private boolean isCampaignDraft(GdiCampaign campaign, CampaignStatusModerate effectiveModerationStatus) {
        return !campaign.getArchived() && effectiveModerationStatus == CampaignStatusModerate.NEW;
    }

    private boolean isCampaignWaitingForArchiving(GdiCampaign campaign) {
        return !campaign.getArchived() && campaign.getDelayedOperation() == GdiCampaignDelayedOperation.ARC;
    }

    private boolean isCampaignWaitingForUnarchiving(GdiCampaign campaign) {
        return campaign.getArchived() && campaign.getDelayedOperation() == GdiCampaignDelayedOperation.UNARC;
    }

    boolean isCampaignWaitingForPayment(GdiBaseCampaign campaign,
                                        CampaignStatusModerate effectiveModerationStatus) {
        return !campaign.getArchived() && campaign.getSumToPay().compareTo(Currencies.EPSILON) > 0 &&
                effectiveModerationStatus != CampaignStatusModerate.NO;
    }

    boolean isCampaignNeedsNewPayment(GdiBaseCampaign campaign, BigDecimal sumTotal) {
        return !isValidId(campaign.getWalletId()) &&
                !campaign.getArchived() &&
                BigDecimal.ZERO.compareTo(campaign.getSumLast()) < 0 &&
                BigDecimal.ZERO.compareTo(sumTotal) < 0 &&
                NEED_PAYMENT_BORDER.compareTo(sumTotal.divide(campaign.getSumLast(), 3, BigDecimal.ROUND_HALF_UP))
                        >= 0;
    }

    private CampaignStatusModerate calcCampaignEffectiveModerationStatus(GdiCampaign campaign) {
        if (campaign.getStatusModerate() == CampaignStatusModerate.YES &&
                campaign.getStatusPostModerate() == CampaignStatusPostmoderate.NO) {
            return CampaignStatusModerate.NO;
        }
        return campaign.getStatusModerate();
    }

    private Pair<GdCampaignPrimaryStatus, GdCampaignPrimaryStatusDesc> calcPrimaryStatus(GdiCampaign campaign,
                                                                                         @Nullable GdWallet wallet,
                                                                                         BigDecimal sumTotal,
                                                                                         @Nullable TimeTargetStatusInfo timeTargetStatus,
                                                                                         CampaignStatusModerate effectiveModerationStatus,
                                                                                         Instant currentInstant) {
        boolean campUnderWallet = (isValidId(campaign.getWalletId()) && wallet != null);

        GdCampaignPrimaryStatus primaryStatus;
        GdCampaignPrimaryStatusDesc primaryStatusDesc = null;

        LocalDate today = currentInstant.atZone(MSK).toLocalDate();
        if (campaign.getArchived()) {
            primaryStatus = GdCampaignPrimaryStatus.ARCHIVED;
            if (campaign.getCurrencyCode() == CurrencyCode.YND_FIXED && campaign.getCurrencyConverted()) {
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_NOT_RECOVERED;
            } else if (isCampaignWaitingForUnarchiving(campaign)) {
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_IN_PROGRESS;
            }
        } else if (effectiveModerationStatus == CampaignStatusModerate.NEW) {
            primaryStatus = GdCampaignPrimaryStatus.DRAFT;
        } else if (!campaign.getShowing()) {
            primaryStatus = GdCampaignPrimaryStatus.STOPPED;
            if (isCampaignWaitingForArchiving(campaign) ||
                    isCampaignActivating(campaign, sumTotal, currentInstant)) {
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_IN_PROGRESS;
            }
        } else if (campaign.getFinishDate() != null && campaign.getFinishDate().isBefore(today)) {
            primaryStatus = GdCampaignPrimaryStatus.STOPPED;
            primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_OVER;
        } else if (effectiveModerationStatus != CampaignStatusModerate.YES) {

            if (effectiveModerationStatus == CampaignStatusModerate.NO) {
                primaryStatus = GdCampaignPrimaryStatus.MODERATION_DENIED;
            } else {
                primaryStatus = GdCampaignPrimaryStatus.MODERATION;
            }

        } else { /*Кампания прошла модерацию*/
            /*Нет денег - кампания без кошелька*/
            if (!isValidId(campaign.getWalletId()) &&
                    //на кампании были деньги,но все потратилось
                    ((campaign.getSum().compareTo(Currencies.EPSILON) >= 0 &&
                            sumTotal.compareTo(Currencies.EPSILON) <= 0 && campaign.getOrderId() != 0)
                            //или на кампании вообще нет денег
                            || (campaign.getSum().compareTo(Currencies.EPSILON) <= 0
                            && campaign.getOrderId() == 0))) {

                primaryStatus =
                        GdCampaignPrimaryStatus.NO_MONEY;
                primaryStatusDesc = (campaign.getSumToPay().compareTo(Currencies.EPSILON) > 0)
                        ? GdCampaignPrimaryStatusDesc.WAIT_PAYMENT :
                        GdCampaignPrimaryStatusDesc.ADD_MONEY;

            } else if (isValidId(campaign.getWalletId()) &&
                    sumTotal.compareTo(Currencies.EPSILON) <= 0) {
                /*Нет денег - Кампания под кошельком*/
                primaryStatus = GdCampaignPrimaryStatus.NO_MONEY;
                primaryStatusDesc = (wallet.getStatus().getWaitingForPayment()) ?
                        GdCampaignPrimaryStatusDesc.WAIT_PAYMENT :
                        GdCampaignPrimaryStatusDesc.ADD_MONEY_TO_WALLET;
            } else if (!campaign.getHasBanners() || !campaign.getHasActiveBanners()) {
                //Кампания прошла модерацию, но в ней нет активных баннеров или нет баннеров вовсе (удалили)
                primaryStatus = GdCampaignPrimaryStatus.STOPPED;
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.NO_ACTIVE_BANNERS;

            } else if (campaign.getStartDate() != null && campaign.getStartDate().isAfter(today)) {
                //Кампания временно неактивна. Для возобновления не требуются действия пользователя.
                primaryStatus = GdCampaignPrimaryStatus.TEMPORARILY_PAUSED;
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_WAITING_START;

            } else if (campaign.getDayBudget() != null && campaign.getDayBudget().compareTo(BigDecimal.ZERO) > 0 &&
                    campaign.getDayBudgetStopTime() != null &&
                    campaign.getDayBudgetStopTime().toLocalDate().equals(today)) {
                /*Сработало ограниение дневного бюджета на кампании*/
                primaryStatus = GdCampaignPrimaryStatus.TEMPORARILY_PAUSED;
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_PAUSED_BY_DAY_BUDGET;
            } else if (campUnderWallet &&
                    wallet.getStatus().getBudgetLimitationStopTime() != null &&
                    wallet.getStatus().getBudgetLimitationStopTime().toLocalDate().isEqual(today)) {
                /*Сработало ограничение дневного бюджета на кошельке */
                primaryStatus = GdCampaignPrimaryStatus.TEMPORARILY_PAUSED;
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_PAUSED_BY_WALLET_DAY_BUDGET;
            } else if (timeTargetStatus != null && timeTargetStatus.getStatus()
                    != TimeTargetStatus.ACTIVE /*Кампания остановлена временным тарегетингом*/) {
                primaryStatus = GdCampaignPrimaryStatus.TEMPORARILY_PAUSED;
                primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_PAUSED_BY_TIMETARGETING;
            } else {
                primaryStatus = GdCampaignPrimaryStatus.ACTIVE;
                if (isCampaignActivating(campaign, sumTotal, currentInstant)) {
                    primaryStatusDesc = GdCampaignPrimaryStatusDesc.IS_IN_PROGRESS;
                }
            }
        }
        return Pair.of(primaryStatus, primaryStatusDesc);
    }
}
