package ru.yandex.direct.grid.processing.service.campaign.converter;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.grid.model.campaign.GdAutobudgetProblem;
import ru.yandex.direct.grid.model.campaign.GdAutobudgetProblemRecommendation;
import ru.yandex.direct.grid.model.campaign.GdAutobudgetProblemType;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;

@ParametersAreNonnullByDefault
public class AutobudgetProblemConverter {

    private static final Map<AutobudgetAggregatedHourlyProblem, GdAutobudgetProblemType> TO_GD_AUTOBUDGET_PROBLEM_TYPE
            = ImmutableMap.<AutobudgetAggregatedHourlyProblem, GdAutobudgetProblemType>builder()
            .put(AutobudgetAggregatedHourlyProblem.WALLET_DAY_BUDGET_REACHED,
                    GdAutobudgetProblemType.WALLET_DAY_BUDGET_REACHED)
            .put(AutobudgetAggregatedHourlyProblem.ENGINE_MIN_COST_LIMITED,
                    GdAutobudgetProblemType.ENGINE_MIN_COST_LIMITED)
            .put(AutobudgetAggregatedHourlyProblem.UPPER_POSITIONS_REACHED,
                    GdAutobudgetProblemType.UPPER_POSITIONS_REACHED)
            .put(AutobudgetAggregatedHourlyProblem.MAX_BID_REACHED,
                    GdAutobudgetProblemType.MAX_BID_REACHED)
            .build();

    private AutobudgetProblemConverter() {
    }

    @Nullable
    public static GdAutobudgetProblem toGdAutobudgetProblem(
            GdiCampaign campaign,
            @Nullable AutobudgetAggregatedHourlyProblem problem) {
        if (problem == null || problem == AutobudgetAggregatedHourlyProblem.NO_PROBLEM) {
            return null;
        }

        return new GdAutobudgetProblem()
                .withProblem(toGdAutobudgetProblemType(problem))
                .withRecommendations(getAutobudgetRecommendations(campaign, problem));
    }

    private static GdAutobudgetProblemType toGdAutobudgetProblemType(AutobudgetAggregatedHourlyProblem problem) {
        return TO_GD_AUTOBUDGET_PROBLEM_TYPE.get(problem);
    }

    private static List<GdAutobudgetProblemRecommendation> getAutobudgetRecommendations(
            GdiCampaign campaign, AutobudgetAggregatedHourlyProblem problem) {
        List<GdAutobudgetProblemRecommendation> recommendations = new ArrayList<>();

        if (problem != AutobudgetAggregatedHourlyProblem.UPPER_POSITIONS_REACHED) {
            recommendations.add(GdAutobudgetProblemRecommendation.INCREASE_MAX_BID);
        }

        if (campaign.getType() == CampaignType.DYNAMIC) {
            recommendations.add(GdAutobudgetProblemRecommendation.ADD_DYNAMIC_AD_TARGETS);
        } else if (campaign.getType() == CampaignType.PERFORMANCE) {
            recommendations.add(GdAutobudgetProblemRecommendation.ADD_PERFORMANCE_FILTERS);
        } else {
            if (campaign.getPlatform() == CampaignsPlatform.SEARCH) {
                recommendations.add(GdAutobudgetProblemRecommendation.ENABLE_CONTEXT);
            } else if (campaign.getContextLimit() != 100) {
                recommendations.add(GdAutobudgetProblemRecommendation.DISABLE_CONTEXT_LIMIT);
            } else if (!campaign.getHasBroadMatch()) {
                recommendations.add(GdAutobudgetProblemRecommendation.ENABLE_BROAD_MATCH);
            } else if (campaign.getBroadMatchLimit() > 0) {
                recommendations.add(GdAutobudgetProblemRecommendation.DISABLE_BROAD_MATCH_LIMIT);
            }

            recommendations.add(GdAutobudgetProblemRecommendation.ADD_KEYWORDS);
        }

        recommendations.add(GdAutobudgetProblemRecommendation.INCREASE_EFFICIENCY);

        return recommendations;
    }
}
