package ru.yandex.direct.grid.processing.service.campaign.converter;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurer;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurerParamsIas;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurerSystem;
import ru.yandex.direct.core.entity.campaign.model.CampaignWarnPlaceInterval;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.PlacementType;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ContactPhone;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.grid.model.GdContactPhone;
import ru.yandex.direct.grid.model.GdTimeInterval;
import ru.yandex.direct.grid.model.campaign.GdCampaignAgencyInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignManagerInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignMeasurer;
import ru.yandex.direct.grid.model.campaign.GdCampaignMeasurerSystem;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlacementType;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform;
import ru.yandex.direct.grid.model.campaign.GdCpmYndxFrontpageCampaignShowType;
import ru.yandex.direct.grid.model.campaign.GdDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaignDeviceTypeTargeting;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaignNetworkTargeting;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignCheckPositionInterval;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignSmsEvent;
import ru.yandex.direct.grid.processing.model.campaign.GdPayForConversionInfo;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCampaignVcard;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdBroadMatchRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignBiddingStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignBrandSafetyRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyData;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdMeaningfulGoalRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdShowsFrequencyLimitRequest;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddAddress;
import ru.yandex.direct.grid.processing.service.client.converter.WorkTimeConverter;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.crypta.utils.CryptaSegmentBrandSafetyUtils.makeBrandSafetyCategories;
import static ru.yandex.direct.currency.Money.MONEY_CENT_SCALE;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_AVG_CPV;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_MAX_IMPRESSIONS;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_MAX_REACH;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD;
import static ru.yandex.direct.grid.processing.service.client.converter.VcardDataConverter.toInstantMessenger;
import static ru.yandex.direct.grid.processing.service.client.converter.VcardDataConverter.toPhone;
import static ru.yandex.direct.grid.processing.service.client.converter.VcardDataConverter.toPointOnMap;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@ParametersAreNonnullByDefault
public class CommonCampaignConverter {
    public static final Set AUTOBUDGET_STRATEGIES = Set.of(
            GdCampaignStrategyName.AUTOBUDGET,
            GdCampaignStrategyName.AUTOBUDGET_MEDIA,
            GdCampaignStrategyName.AUTOBUDGET_WEEK_BUNDLE,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CLICK,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CPA,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER,
            GdCampaignStrategyName.AUTOBUDGET_AVG_CPI,
            GdCampaignStrategyName.AUTOBUDGET_ROI,
            GdCampaignStrategyName.AUTOBUDGET_CRR,
            AUTOBUDGET_AVG_CPV,
            AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
            AUTOBUDGET_MAX_REACH,
            AUTOBUDGET_MAX_IMPRESSIONS,
            AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD
    );

    private CommonCampaignConverter() {
    }

    public static EnumSet<SmsFlag> toSmsFlags(Set<GdCampaignSmsEvent> enableEvents) {
        Set<SmsFlag> smsFlags = mapSet(enableEvents, e -> SmsFlag.valueOf(e.getTypedValue()));
        return smsFlags.isEmpty() ? EnumSet.noneOf(SmsFlag.class) : EnumSet.copyOf(smsFlags);
    }

    public static TimeInterval toTimeInterval(GdTimeInterval smsTime) {
        return new TimeInterval()
                .withStartHour(smsTime.getStartTime().getHour())
                .withStartMinute(smsTime.getStartTime().getMinute())
                .withEndHour(smsTime.getEndTime().getHour())
                .withEndMinute(smsTime.getEndTime().getMinute());
    }

    @Nullable
    public static CampaignWarnPlaceInterval toCampaignWarnPlaceInterval(
            @Nullable GdCampaignCheckPositionInterval gdCampaignCheckPositionInterval) {
        return ifNotNull(gdCampaignCheckPositionInterval,
                gdEnum -> CampaignWarnPlaceInterval.valueOf(gdEnum.getTypedValue()));
    }

    @Nullable
    public static List<MeaningfulGoal> toMeaningfulGoals(@Nullable List<GdMeaningfulGoalRequest> goals) {
        return mapList(goals, goal -> new MeaningfulGoal()
                .withGoalId(goal.getGoalId())
                .withConversionValue(setScaleForCore(goal.getConversionValue()))
                .withIsMetrikaSourceOfValue(goal.getIsMetrikaSourceOfValue()));
    }

    @Nullable
    public static BroadMatch toBroadMatch(@Nullable GdBroadMatchRequest broadMatchRequest) {
        if (broadMatchRequest == null) {
            return null;
        }
        return new BroadMatch()
                .withBroadMatchFlag(broadMatchRequest.getBroadMatchFlag())
                .withBroadMatchLimit(broadMatchRequest.getBroadMatchLimit())
                .withBroadMatchGoalId(broadMatchRequest.getBroadMatchGoalId());
    }

    @Nullable
    public static List<CampaignMeasurer> toMeasurers(@Nullable List<GdCampaignMeasurer> gd) {
        if (gd == null) {
            return null;
        }
        return StreamEx.of(gd)
                .map(it -> new CampaignMeasurer()
                        .withParams(toCoreParams(it))
                        .withMeasurerSystem(toCoreMeasurerSystem(it.getMeasurerSystem()))
                )
                .toList();
    }

    private static String toCoreParams(GdCampaignMeasurer gd) {
        switch (gd.getMeasurerSystem()) {
            case MOAT:
            case DV:
                return gd.getParams();
            case IAS:
                var params = CampaignMeasurerParamsIas.from(gd.getParams());
                return toJson(params);
            default:
                return null;
        }
    }

    private static CampaignMeasurerSystem toCoreMeasurerSystem(GdCampaignMeasurerSystem gd) {
        switch (gd) {
            case MOAT:
                return CampaignMeasurerSystem.MOAT;
            case DV:
                return CampaignMeasurerSystem.DV;
            case IAS:
                return CampaignMeasurerSystem.IAS;
            default:
                return null;
        }
    }

    @Nullable
    public static List<Long> toBrandSafetyCategories(@Nullable List<Long> brandSafetyCategories,
                                                     @Nullable GdCampaignBrandSafetyRequest brandSafetyRequest) {
        if (brandSafetyRequest == null) {
            return brandSafetyCategories;
        }

        return makeBrandSafetyCategories(brandSafetyRequest.getIsEnabled(),
                brandSafetyRequest.getAdditionalCategories());
    }

    public static DayBudgetShowMode toCampaignDayBudgetShowMode(@Nullable GdDayBudgetShowMode dayBudgetShowMode) {
        if (dayBudgetShowMode == null) {
            return null;
        } else if (dayBudgetShowMode == GdDayBudgetShowMode.STRETCHED) {
            return DayBudgetShowMode.STRETCHED;
        }
        return DayBudgetShowMode.DEFAULT_;
    }

    @Nullable
    public static Vcard toCoreVcard(@Nullable GdAddCampaignVcard vcard) {
        if (vcard == null) {
            return null;
        }
        GdAddAddress gdAddress = vcard.getAddress();

        return new Vcard()
                .withCompanyName(vcard.getCompanyName())
                .withContactPerson(vcard.getContactPerson())
                .withEmail(vcard.getEmail())
                .withExtraMessage(vcard.getExtraMessage())
                .withOgrn(vcard.getOgrn())
                .withWorkTime(WorkTimeConverter.toWorkTime(vcard.getWorkTimes()))
                .withInstantMessenger(toInstantMessenger(vcard.getInstantMessenger()))
                .withPhone(toPhone(vcard.getPhone()))
                .withCountry(gdAddress.getCountry())
                .withCity(gdAddress.getCity())
                .withStreet(gdAddress.getStreet())
                .withHouse(gdAddress.getHouseWithBuilding())
                .withApart(gdAddress.getApartment())
                .withMetroId(ifNotNull(gdAddress.getMetroStationId(), Integer::longValue))
                .withManualPoint(toPointOnMap(gdAddress.getPointOnMap()));
    }

    //TODO: Убрать nullable после поддержки фронтом
    @Nullable
    public static DbStrategy toCampaignStrategy(GdCampaignBiddingStrategy gdCampaignBiddingStrategy) {
        if (null == gdCampaignBiddingStrategy.getStrategyName()) {
            return null;
        }
        var dbStrategy = new DbStrategy();
        dbStrategy.withAutobudget(toCampaignsAutobudget(gdCampaignBiddingStrategy.getStrategyName()));
        dbStrategy.withPlatform(toCampaignPlatform(gdCampaignBiddingStrategy.getPlatform()));
        dbStrategy.withStrategy(toCampOptionsStrategy(gdCampaignBiddingStrategy.getStrategy()));
        dbStrategy.withStrategyName(toCampaignStrategyName(gdCampaignBiddingStrategy.getStrategyName()));
        dbStrategy.withStrategyData(toCampaignStrategyData(gdCampaignBiddingStrategy.getStrategyData(),
                gdCampaignBiddingStrategy.getStrategyName()));
        return dbStrategy;
    }

    private static CampaignsAutobudget toCampaignsAutobudget(GdCampaignStrategyName gdStrategyName) {
        return isAutobubgetStrategy(gdStrategyName) ? CampaignsAutobudget.YES : CampaignsAutobudget.NO;
    }

    private static Boolean isAutobubgetStrategy(GdCampaignStrategyName gdStrategyName) {
        return AUTOBUDGET_STRATEGIES.contains(gdStrategyName);
    }

    private static CampaignsPlatform toCampaignPlatform(GdCampaignPlatform gdCampaignPlatform) {
        return CampaignsPlatform.valueOf(gdCampaignPlatform.name());
    }

    @Nullable
    private static CampOptionsStrategy toCampOptionsStrategy(GdCampaignStrategy gdStrategy) {
        return ifNotNull(gdStrategy, strategy -> CampOptionsStrategy.valueOf(strategy.name()));
    }

    private static StrategyName toCampaignStrategyName(GdCampaignStrategyName gdStrategyName) {
        return StrategyName.valueOf(gdStrategyName.name());
    }

    private static StrategyData toCampaignStrategyData(GdCampaignStrategyData gdStrategyData,
                                                       GdCampaignStrategyName gdStrategyName) {
        StrategyData strategyData = new StrategyData().withVersion(1L)

                .withAvgCpm(setScaleForCore(gdStrategyData.getAvgCpm()))
                .withStart(gdStrategyData.getStartDate())
                .withFinish(gdStrategyData.getFinishDate())
                .withName(gdStrategyName == GdCampaignStrategyName.DEFAULT_
                        ? "default"
                        : gdStrategyName.name().toLowerCase())
                .withAvgBid(setScaleForCore(gdStrategyData.getAvgBid()))
                .withFilterAvgBid(setScaleForCore(gdStrategyData.getFilterAvgBid()))
                .withGoalId(gdStrategyData.getGoalId())
                .withBid(setScaleForCore(gdStrategyData.getBid()))
                .withAvgCpv(setScaleForCore(gdStrategyData.getAvgCpv()))
                .withRfDecay(gdStrategyData.getRfDecay())
                .withRfMinCpm(gdStrategyData.getRfMinCpm())
                .withAvgCpa(setScaleForCore(gdStrategyData.getAvgCpa()))
                .withAvgCpi(setScaleForCore(gdStrategyData.getAvgCpi()))
                .withFilterAvgCpa(setScaleForCore(gdStrategyData.getFilterAvgCpa()))
                .withRoiCoef(setScaleForCore(gdStrategyData.getRoiCoef()))
                .withProfitability(setScaleForCore(gdStrategyData.getProfitability()))
                .withReserveReturn(gdStrategyData.getReserveReturn())
                .withLimitClicks(gdStrategyData.getLimitClicks())
                .withPayForConversion(gdStrategyData.getPayForConversion())
                .withCrr(gdStrategyData.getCrr());

        if (gdStrategyName == AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD ||
                gdStrategyName == AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD ||
                        gdStrategyName == AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD) {
            strategyData.withBudget(setScaleForCore(gdStrategyData.getSum()));
            strategyData.withAutoProlongation(
                    gdStrategyData.getAutoProlongation() != null && gdStrategyData.getAutoProlongation() ? 1L : 0L);
        } else {
            strategyData.withSum(setScaleForCore(gdStrategyData.getSum()));
        }
        return strategyData;
    }

    public static List<Long> toMetrikaCounters(List<Integer> gdMetrikaCounters) {
        return mapList(gdMetrikaCounters, Integer::longValue);
    }

    @Nullable
    public static List<Long> toNullableMetrikaCounters(List<Integer> gdMetrikaCounters) {
        return isEmpty(gdMetrikaCounters) ? null : toMetrikaCounters(gdMetrikaCounters);
    }

    //По идее это уже должно делаться на фронте, но на случай прямых вызовов и всяких странных ситуаций при конвертации
    //в ядровую модель оставляем только 2 десятичных знака
    @Nullable
    public static BigDecimal setScaleForCore(@Nullable BigDecimal value) {
        return ifNotNull(value, v -> v.setScale(MONEY_CENT_SCALE, RoundingMode.DOWN));
    }

    public static List<String> getDisabledSsp(
            HostingsHandler hostingsHandler,
            List<String> knownSsp,
            List<String> disabledPlaces) {
        if (!isEmpty(disabledPlaces)) {
            return StreamEx.of(disabledPlaces)
                    .map(hostingsHandler::stripWww)
                    .map(place -> filterList(knownSsp, place::equalsIgnoreCase))
                    .toFlatList(Function.identity());
        }
        return emptyList();
    }

    public static List<String> filterSsp(
            List<String> knownSsp,
            List<String> domain) {
        if (!isEmpty(domain)) {
            return StreamEx.of(domain)
                    .map(place -> filterList(knownSsp, place::equalsIgnoreCase))
                    .toFlatList(Function.identity());
        }
        return emptyList();
    }

    public static Set<PlacementType> toPlacementTypes(@Nullable Set<GdCampaignPlacementType> placementTypes) {
        return nvl(mapSet(placementTypes, CommonCampaignConverter::toPlacementType), Set.of());
    }

    public static PlacementType toPlacementType(GdCampaignPlacementType placementType) {
        switch (placementType) {
            case SEARCH_PAGE:
                return PlacementType.SEARCH_PAGE;
            case ADV_GALLERY:
                return PlacementType.ADV_GALLERY;
            default:
                throw new IllegalArgumentException("Unknown GdCampaignPlacementType: " + placementType);
        }
    }

    public static FrontpageCampaignShowType toFrontpageCampaignShowType(GdCpmYndxFrontpageCampaignShowType type) {
        if (type == GdCpmYndxFrontpageCampaignShowType.FRONTPAGE) {
            return FrontpageCampaignShowType.FRONTPAGE;
        } else if (type == GdCpmYndxFrontpageCampaignShowType.FRONTPAGE_MOBILE) {
            return FrontpageCampaignShowType.FRONTPAGE_MOBILE;
        } else if (type == GdCpmYndxFrontpageCampaignShowType.BROWSER_NEW_TAB) {
            return FrontpageCampaignShowType.BROWSER_NEW_TAB;
        } else {
            throw new IllegalArgumentException("Unknown GdCpmYndxFrontpageCampaignShowType: " + type);
        }
    }

    public static EnumSet<MobileAppDeviceTypeTargeting> toMobileAppDeviceTypeTargeting(
            Set<GdMobileContentCampaignDeviceTypeTargeting> deviceTypeTargeting) {
        Set<MobileAppDeviceTypeTargeting> deviceTypes = mapSet(deviceTypeTargeting,
                deviceType -> MobileAppDeviceTypeTargeting.fromTypedValue(deviceType.getTypedValue().toLowerCase()));
        return deviceTypes == null || deviceTypes.isEmpty()
                ? EnumSet.noneOf(MobileAppDeviceTypeTargeting.class)
                : EnumSet.copyOf(deviceTypes);
    }

    public static EnumSet<MobileAppNetworkTargeting> toMobileAppNetworkTargeting(
            Set<GdMobileContentCampaignNetworkTargeting> networkTargeting) {
        Set<MobileAppNetworkTargeting> networks = mapSet(networkTargeting,
                network -> MobileAppNetworkTargeting.fromTypedValue(network.getTypedValue().toLowerCase()));
        return networks == null || networks.isEmpty()
                ? EnumSet.noneOf(MobileAppNetworkTargeting.class)
                : EnumSet.copyOf(networks);
    }

    public static GdPayForConversionInfo toGdPayForConversionInfo(@Nullable Boolean lackOfConversions) {
        return new GdPayForConversionInfo()
                .withLackOfConversions(lackOfConversions);
    }

    public static GdPayForConversionInfo toGdPayForConversionInfo(@Nullable Boolean lackOfConversions, @Nullable Boolean lackOfFunds) {
        return new GdPayForConversionInfo()
                .withLackOfConversions(lackOfConversions)
                .withLackOfFunds(lackOfFunds);
    }

    @Nullable
    public static Integer getImpressionRateIntervalDays(@Nullable GdShowsFrequencyLimitRequest showsFrequencyLimit) {
        if (showsFrequencyLimit != null) {
            if (showsFrequencyLimit.getIsForCampaignTime()) {
                return null;
            }
            return showsFrequencyLimit.getDays();
        }
        return null;
    }

    public static GdCampaignManagerInfo toGdManagerInfo(User userInfo, List<ContactPhone> contactPhones) {
        return new GdCampaignManagerInfo()
                .withName(userInfo.getFio())
                .withEmail(userInfo.getEmail())
                .withContactInfo(mapList(contactPhones, CommonCampaignConverter::toGdPhoneContact));
    }

    public static GdContactPhone toGdPhoneContact(ContactPhone contactPhone) {
        return new GdContactPhone()
                .withPhone(contactPhone.getPhone())
                .withExtension(contactPhone.getExtension());
    }

    /**
     * @param gdAgencyUserInfo Информация о главном представителе агентства
     * @param gdRepresentativeUserInfo Информация о представителе, обслуживающем кампанию
     */
    @Nullable
    public static GdCampaignAgencyInfo toCampaignGdAgencyInfo(@Nullable Client agencyClient, GdUserInfo gdAgencyUserInfo,
                                                              GdUserInfo gdRepresentativeUserInfo, boolean showContacts) {
        if (agencyClient == null) {
            return null;
        }

        GdCampaignAgencyInfo gdAgencyInfo = new GdCampaignAgencyInfo()
                .withName(nvl(agencyClient.getName(), ""))
                .withShowAgencyContacts(showContacts);

        if (showContacts) {
            boolean isChief = Objects.equals(gdAgencyUserInfo.getUserId(), gdRepresentativeUserInfo.getUserId());
            gdAgencyInfo
                    .withRepresentativeName(isChief ? null : gdRepresentativeUserInfo.getName())
                    .withEmail(gdRepresentativeUserInfo.getEmail())
                    .withPhone(gdRepresentativeUserInfo.getPhone());
        } else {
            gdAgencyInfo
                    .withRepresentativeName(gdAgencyUserInfo.getName());
        }

        return gdAgencyInfo;
    }

}
