package ru.yandex.direct.grid.processing.service.campaign.converter;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.keyword.model.AutoBudgetPriority;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.touchsocdem.service.converter.GridTouchSocdemConverter;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTextCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignBiddingStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyData;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddTouchAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateTouchAdGroup;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateCpmRetargetingConditionItem;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdiTouchSocdem;
import ru.yandex.direct.grid.processing.service.campaign.CampaignConverterContext;
import ru.yandex.direct.grid.processing.service.campaign.converter.type.CampaignConverterFacade;
import ru.yandex.direct.grid.processing.service.touchsocdem.service.converter.TouchSocdemConverter;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.MappingPathNodeConverter;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdDayBudgetShowMode;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdBroadMatch;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdCampaignNotification;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdTimeTarget;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupsMutationDataConverter.toRetargetingCondition;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.path;

public class TouchCampaignConverter {
    private static final PathNodeConverter SKIP_CONVERTER = SkipByDefaultMappingPathNodeConverter.emptyConverter();

    private static final PathNodeConverter CAMPAIGN_CONVERTER = MappingPathNodeConverter
            .builder("TouchCampaignConverter")
            .skip(DbStrategyBase.STRATEGY.name())
            .build();

    private static final PathNodeConverter BUDGET_CONVERTER = SkipByDefaultMappingPathNodeConverter.builder()
            .replace(StrategyData.SUM.name(),
                    List.of(GdAddTouchCampaignInput.BUDGET.name(), GdCampaignBudget.SUM.name()))
            .build();

    private static final PathNodeConverter AD_GROUP_CONVERTER = new SpecialTouchAdGroupPathNodeConverter()
            .replace(AdGroup.GEO.name(), GdAddTouchAdGroup.REGION_IDS.name())
            .replace(ComplexTextAdGroup.COMPLEX_BANNERS.name(), GdAddTouchAdGroup.AD.name())
            .skipIndex(ComplexTextAdGroup.COMPLEX_BANNERS.name())
            .build();

    public static final PathNodeConverterProvider PATH_NODE_CONVERTER_PROVIDER = DefaultPathNodeConverterProvider
            .builder()
            .register(TextCampaign.class, CAMPAIGN_CONVERTER)
            .register(DbStrategy.class, SKIP_CONVERTER)
            .register(StrategyData.class, BUDGET_CONVERTER)
            .register(TextAdGroup.class, AD_GROUP_CONVERTER)
            .build();

    static class SpecialTouchAdGroupPathNodeConverter implements PathNodeConverter {
        private List<PathNode.Field> listItemsToSkipIndex = new ArrayList<>();
        private MappingPathNodeConverter.Builder innerConverterBuilder =
                MappingPathNodeConverter.builder("TouchGroupConverter");
        private MappingPathNodeConverter innerConverter;

        SpecialTouchAdGroupPathNodeConverter skipIndex(String fieldName) {
            listItemsToSkipIndex.add(new PathNode.Field(fieldName));
            return this;
        }

        SpecialTouchAdGroupPathNodeConverter replace(String sourceFieldName, String destFieldName) {
            innerConverterBuilder.replace(sourceFieldName, destFieldName);
            return this;
        }

        SpecialTouchAdGroupPathNodeConverter build() {
            innerConverter = innerConverterBuilder.build();
            return this;
        }

        @Override
        public Path convert(PathNode.Field field) {
            return innerConverter.convert(field);
        }

        @Override
        public Path convert(PathNode.Field field, PathNode.Index index) {
            if (listItemsToSkipIndex.contains(field)) {
                return path();
            } else {
                return innerConverter.convert(field, index);
            }
        }
    }

    public static TextCampaign toCoreCampaign(String campaignName, GdCampaignBudget budget, String userEmail,
                                              CampaignConverterContext converterContext) {
        // Для всего, что не приходит с фронта, используем значения отсюда:
        // https://wiki.yandex-team.ru/direct/TechnicalDesign/touch-interface/#m-podrobnosti
        var gdCampaign = new GdAddTextCampaign()
                .withName(campaignName)
                .withBiddingStategy(new GdCampaignBiddingStrategy()
                        .withStrategyName(GdCampaignStrategyName.AUTOBUDGET)
                        .withPlatform(GdCampaignPlatform.CONTEXT)
                        .withStrategyData(new GdCampaignStrategyData().withSum(budget.getSum()))
                )
                .withNotification(defaultGdCampaignNotification(userEmail))
                .withBroadMatch(defaultGdBroadMatch())
                .withDayBudget(CampaignConstants.DEFAULT_DAY_BUDGET)
                .withDayBudgetShowMode(toGdDayBudgetShowMode(CampaignConstants.DEFAULT_DAY_BUDGET_SHOW_MODE))
                .withEnableCompanyInfo(CampaignConstants.DEFAULT_ENABLE_COMPANY_INFO)
                .withEnableCpcHold(CampaignConstants.DEFAULT_ENABLE_CPC_HOLD)
                .withExcludePausedCompetingAds(CampaignConstants.DEFAULT_EXCLUDE_PAUSED_COMPETING_ADS)
                .withHasAddMetrikaTagToUrl(CampaignConstants.DEFAULT_ADD_METRIKA_TAG_TO_URL)
                .withHasAddOpenstatTagToUrl(CampaignConstants.DEFAULT_ADD_OPENSTAT_TAG_TO_URL)
                .withHasExtendedGeoTargeting(true)
                .withHasSiteMonitoring(false)
                .withHasTitleSubstitute(true)
                .withMetrikaCounters(emptyList())
                .withStartDate(LocalDate.now())
                .withTimeTarget(defaultGdTimeTarget());

        var support = CampaignConverterFacade.getSupportByAddType(GdAddTextCampaign.class, converterContext);

        return (TextCampaign)support.toCoreCampaign(gdCampaign);
    }

    public static ModelChanges<TextCampaign> toCoreCampaignModelChanges(
            Long cid, String campaignName, GdCampaignBudget budget, CampaignConverterContext converterContext
    ) {
        TextCampaign coreCampaign = toCoreCampaign(campaignName, budget, null, converterContext);
        return new ModelChanges<>(cid, TextCampaign.class)
                .process(campaignName, TextCampaign.NAME)
                .process(coreCampaign.getStrategy(), TextCampaign.STRATEGY);
    }

    public static ComplexTextAdGroup toComplexTextAdGroup(
            ClientId clientId, Long campaignId,
            GdAddTouchAdGroup group, String groupName,
            Boolean isAutotargetingEnabled, TranslationService translationService
    ) {
        List<TargetInterest> targetInterestWithInterestsCondition = null;
        RetargetingCondition retargetingCondition = null;
        if (hasGoals(group.getRetargetingCondition())) {
            targetInterestWithInterestsCondition = List.of(new TargetInterest().withAdGroupId(0L));
            retargetingCondition = toRetargetingCondition(
                    clientId, targetInterestWithInterestsCondition.get(0), group.getRetargetingCondition(),
                    true, translationService);
        }

        var ad = group.getAd();
        GdiTouchSocdem innerTouchSocdem = TouchSocdemConverter.toInnerTouchSocdem(group.getSocdem());
        ComplexBidModifier complexBidModifier = new ComplexBidModifier()
                .withDemographyModifier(GridTouchSocdemConverter.toSocdemBidModifier(innerTouchSocdem));

        var result = new ComplexTextAdGroup()
                .withAdGroup(new TextAdGroup()
                        .withType(AdGroupType.BASE)
                        .withCampaignId(campaignId)
                        .withName(groupName)
                        .withGeo(mapList(group.getRegionIds(), Integer::longValue))
                        .withHyperGeoId(group.getHyperGeoRetargetingId())
                )
                .withRetargetingCondition(retargetingCondition)
                .withTargetInterests(targetInterestWithInterestsCondition)
                .withComplexBanners(singletonList(new ComplexBanner().withBanner(
                        new ImageBanner()
                                .withIsMobileImage(false)
                                .withCreativeId(ad.getCreativeId())
                                .withHref(ad.getHref())
                                .withTurboLandingId(ad.getTurbolandingId())
                )))
                .withComplexBidModifier(complexBidModifier);
        if (isAutotargetingEnabled) {
            addDefaultTouchRelevanceMatch(result);
        }
        return result;
    }

    public static void addDefaultTouchRelevanceMatch(ComplexTextAdGroup complexTextAdGroup) {
        complexTextAdGroup.setRelevanceMatches(singletonList(
                new RelevanceMatch().withAutobudgetPriority(AutoBudgetPriority.MEDIUM.getTypedValue())
        ));
    }

    private static boolean hasGoals(@Nullable GdUpdateCpmRetargetingConditionItem retargetingCondition) {
        return retargetingCondition != null
                && !isEmpty(retargetingCondition.getConditionRules())
                // тачёвый фронт присылает интересы в виде retargetingConditions с одним правилом
                // retargetingCondition: {
                //      name: "interests", conditionRules: [{type: "OR", interestType: "short_term", goals: []}]
                // }
                && !isEmpty(retargetingCondition.getConditionRules().get(0).getGoals());
    }

    public static ComplexTextAdGroup toComplexTextAdGroup(
            ClientId clientId, Long campaignId, GdUpdateTouchAdGroup group, String groupName,
            List<TargetInterest> interests, @Nullable RetargetingCondition retCond,
            @Nullable ComplexBidModifier complexAdGroupBidModifier, TranslationService translationService
    ) {
        List<TargetInterest> targetInterests = null;
        RetargetingCondition retargetingCondition = null;
        if (hasGoals(group.getRetargetingCondition())) {
            targetInterests = List.of(new TargetInterest().withAdGroupId(0L));
            if (retCond != null) {
                List<TargetInterest> targetInterestsForRetCond = ifNotNull(interests,
                        ti -> filterList(ti, ret -> ret.getRetargetingConditionId().equals(retCond.getId())));
                if (!isEmpty(targetInterestsForRetCond)) {
                    checkState(targetInterestsForRetCond.size() == 1,
                            "There should be exactly one target for group %s, found %s",
                            group.getId(), targetInterestsForRetCond.size());
                    targetInterests = targetInterestsForRetCond;
                }
            }
            retargetingCondition = toRetargetingCondition(
                    clientId, targetInterests.get(0), group.getRetargetingCondition(), true,
                    translationService);
        }

        if (complexAdGroupBidModifier == null) {
            complexAdGroupBidModifier = new ComplexBidModifier();
        }
        if (group.getSocdem() != null) {
            complexAdGroupBidModifier.setDemographyModifier(GridTouchSocdemConverter.toSocdemBidModifier(
                    TouchSocdemConverter.toInnerTouchSocdem(group.getSocdem())
            ));
        }

        var ad = group.getAd();
        return new ComplexTextAdGroup()
                .withAdGroup(new TextAdGroup()
                        .withId(group.getId())
                        .withType(AdGroupType.BASE)
                        .withCampaignId(campaignId)
                        .withName(groupName)
                        .withGeo(mapList(group.getRegionIds(), Integer::longValue))
                        .withHyperGeoId(group.getHyperGeoRetargetingId())
                )
                .withRetargetingCondition(retargetingCondition)
                .withTargetInterests(targetInterests)
                .withComplexBidModifier(complexAdGroupBidModifier)
                .withComplexBanners(singletonList(new ComplexBanner().withBanner(
                        new ImageBanner()
                                .withId(ad.getId())
                                .withIsMobileImage(false)
                                .withCreativeId(ad.getCreativeId())
                                .withHref(ad.getHref())
                                .withTurboLandingId(ad.getTurbolandingId())
                )));
    }
}
