package ru.yandex.direct.grid.processing.service.campaign.converter;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.adgroup.container.ComplexDynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnlyAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAdditionalDataHref;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMinusKeywords;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrganization;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.feature.service.FeatureHelper;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.touchsocdem.service.converter.GridTouchSocdemConverter;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform;
import ru.yandex.direct.grid.model.campaign.facelift.GdBudgetDisplayFormat;
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelinkSet;
import ru.yandex.direct.grid.processing.model.campaign.facelift.GdAddUpdateCampaignAdditionalData;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddDynamicCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddSmartCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTextCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignBiddingStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategy;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyData;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateUcCampaignInput;
import ru.yandex.direct.grid.processing.model.forecast.GdDeviceType;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupKeywordItem;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemInput;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdiTouchSocdem;
import ru.yandex.direct.grid.processing.service.banner.converter.SitelinkConverter;
import ru.yandex.direct.grid.processing.service.campaign.CampaignConverterContext;
import ru.yandex.direct.grid.processing.service.touchsocdem.service.converter.TouchSocdemConverter;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.MappingPathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.DYNAMIC;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.PERFORMANCE;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.TEXT;
import static ru.yandex.direct.core.entity.uac.UacCommonUtils.getHrefWithTrackingParams;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdAttributionModel;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdDayBudgetShowMode;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_AVG_CPA;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER;
import static ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignStrategyName.AUTOBUDGET_CRR;
import static ru.yandex.direct.grid.processing.service.campaign.converter.AddCampaignMutationConverter.campaignToCoreCampaign;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdBroadMatch;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdCampaignNotification;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdTimeTarget;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupsMutationDataConverter.toCoreKeywords;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class UcCampaignConverter {
    private static final PathNodeConverter SKIP_CONVERTER = SkipByDefaultMappingPathNodeConverter.emptyConverter();
    private static final PathNodeConverter CAMPAIGN_CONVERTER = MappingPathNodeConverter
            .builder("CampaignConverter")
            .skip(DbStrategyBase.STRATEGY.name())
            .build();
    private static final PathNodeConverter AD_GROUP_CONVERTER = SkipByDefaultMappingPathNodeConverter.builder()
            .keep(ComplexTextAdGroup.KEYWORDS.name())
            .keep(TextAdGroup.GEO.name())
            .keep(TextAdGroup.HYPER_GEO_ID.name())
            .build();
    private static final PathNodeConverter KEYWORD_CONVERTER = MappingPathNodeConverter
            .builder("KeywordConverter")
            .skip(Keyword.PHRASE.name())
            .build();
    private static final PathNodeConverter BANNER_CONVERTER = MappingPathNodeConverter
            .builder("BannerConverter")
            .skip(ComplexBanner.SITELINK_SET.name())
            .replace(TextBanner.TITLE.name(), GdUcCampaignMutationInput.AD_TITLE.name())
            .replace(TextBanner.BODY.name(), GdUcCampaignMutationInput.AD_TEXT.name())
            .build();

    public static final PathNodeConverterProvider PATH_NODE_CONVERTER_PROVIDER = DefaultPathNodeConverterProvider
            .builder()
            .register(TextCampaign.class, CAMPAIGN_CONVERTER)
            .register(DbStrategy.class, SKIP_CONVERTER)
            .register(TextAdGroup.class, AD_GROUP_CONVERTER)
            .register(ComplexTextAdGroup.class, AD_GROUP_CONVERTER)
            .register(Keyword.class, KEYWORD_CONVERTER)
            .register(TextBanner.class, BANNER_CONVERTER)
            .build();

    /**
     * Создание модели универсальной кампании. Важно, что кампания создается в режиме новичка:
     * <li>флаг, что кампания поддерживает простую стратегию</li>:
     * {@link GdAddTextCampaign#getIsSimplifiedStrategyViewEnabled()}
     * <li>простая стратегия</li>:
     * {@link ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils#isStrategyAvailableForSimpleView}
     * <li>простая модель атрибуции</li>:
     * {@link ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils#isAttributionModelAvailableForSimpleView}
     */
    public static TextCampaign toCoreTextCampaign(GdUcCampaignMutationInput input, String userEmail,
                                                  List<String> minusKeywords,
                                                  CampaignConverterContext converterContext,
                                                  CampaignAttributionModel defaultAttributionModel,
                                                  boolean advancedGeoTargeting) {
        final var source = input.getSource() != null ? input.getSource() : CampaignSource.UAC;
        final var metatype = nvl(input.getIsEcom(), false) ? CampaignMetatype.ECOM : CampaignMetatype.DEFAULT_;
        var gdCampaign = new GdAddTextCampaign()
                .withName(input.getName())
                .withBiddingStategy(getCampaignStrategy(input, TEXT))
                .withMeaningfulGoals(input.getMeaningfulGoals())
                .withNotification(defaultGdCampaignNotification(userEmail))
                .withBroadMatch(defaultGdBroadMatch())
                .withDayBudget(CampaignConstants.DEFAULT_DAY_BUDGET)
                .withDayBudgetShowMode(toGdDayBudgetShowMode(CampaignConstants.DEFAULT_DAY_BUDGET_SHOW_MODE))
                .withAttributionModel(toGdAttributionModel(defaultAttributionModel))
                .withEnableCompanyInfo(CampaignConstants.DEFAULT_ENABLE_COMPANY_INFO)
                .withEnableCpcHold(CampaignConstants.DEFAULT_ENABLE_CPC_HOLD)
                .withExcludePausedCompetingAds(CampaignConstants.DEFAULT_EXCLUDE_PAUSED_COMPETING_ADS)
                .withHasAddMetrikaTagToUrl(CampaignConstants.DEFAULT_ADD_METRIKA_TAG_TO_URL)
                .withHasAddOpenstatTagToUrl(CampaignConstants.DEFAULT_ADD_OPENSTAT_TAG_TO_URL)
                .withHasExtendedGeoTargeting(true)
                .withHasSiteMonitoring(true)
                .withHasTitleSubstitute(true)
                .withMetrikaCounters(input.getMetrikaCounters())
                .withStartDate(LocalDate.now())
                .withCalltrackingSettingsId(input.getCalltrackingSettingsId())
                .withTimeTarget(nvl(input.getTimeTarget(), defaultGdTimeTarget()))
                .withDefaultPermalinkId(input.getPermalinkId())
                .withDefaultTrackingPhoneId(input.getPhoneId())
                .withIsUniversalCamp(true)
                .withIsSimplifiedStrategyViewEnabled(true)
                .withAdditionalData(new GdAddUpdateCampaignAdditionalData()
                        .withHref(input.getHref())
                        .withBusinessCategory(input.getBusinessCategory())
                )
                .withIsS2sTrackingEnabled(false)
                .withIsRecommendationsManagementEnabled(input.getIsRecommendationsManagementEnabled())
                .withIsPriceRecommendationsManagementEnabled(input.getIsPriceRecommendationsManagementEnabled())
                .withMinusKeywords(minusKeywords);
        if (advancedGeoTargeting) {
            gdCampaign
                    .withUseCurrentRegion(true)
                    .withUseRegularRegion(false);
        }
        TextCampaign campaign = (TextCampaign) campaignToCoreCampaign(gdCampaign, converterContext);
        return campaign
                .withSource(source)
                .withMetatype(metatype)
                .withBudgetDisplayFormat(GdBudgetDisplayFormat.toSource(input.getBudgetDisplayFormat()))
                .withWidgetPartnerId(input.getWidgetPartnerId());
    }

    public static DynamicCampaign toDynamicCampaign(Long masterCid,
                                                    GdUcCampaignMutationInput input,
                                                    String userEmail,
                                                    List<String> minusKeywords,
                                                    CampaignConverterContext converterContext,
                                                    CampaignAttributionModel defaultAttributionModel) {
        final var source = input.getSource() != null ? input.getSource() : CampaignSource.UAC;
        final var metatype = nvl(input.getIsEcom(), false) ? CampaignMetatype.ECOM : CampaignMetatype.DEFAULT_;
        var gdCampaign = new GdAddDynamicCampaign()
                .withName(input.getName())
                .withBiddingStategy(getCampaignStrategy(input, DYNAMIC))
                .withMeaningfulGoals(input.getMeaningfulGoals())
                .withNotification(defaultGdCampaignNotification(userEmail))
                .withDayBudget(CampaignConstants.DEFAULT_DAY_BUDGET)
                .withDayBudgetShowMode(toGdDayBudgetShowMode(CampaignConstants.DEFAULT_DAY_BUDGET_SHOW_MODE))
                .withAttributionModel(toGdAttributionModel(defaultAttributionModel))
                .withEnableCompanyInfo(CampaignConstants.DEFAULT_ENABLE_COMPANY_INFO)
                .withEnableCpcHold(CampaignConstants.DEFAULT_ENABLE_CPC_HOLD)
                .withHasAddMetrikaTagToUrl(CampaignConstants.DEFAULT_ADD_METRIKA_TAG_TO_URL)
                .withHasAddOpenstatTagToUrl(CampaignConstants.DEFAULT_ADD_OPENSTAT_TAG_TO_URL)
                .withHasExtendedGeoTargeting(true)
                .withHasSiteMonitoring(true)
                .withHasTitleSubstitute(true)
                .withMetrikaCounters(input.getMetrikaCounters())
                .withStartDate(LocalDate.now())
                .withTimeTarget(nvl(input.getTimeTarget(), defaultGdTimeTarget()))
                .withDefaultPermalinkId(input.getPermalinkId())
                .withIsSimplifiedStrategyViewEnabled(true)
                .withAdditionalData(new GdAddUpdateCampaignAdditionalData()
                        .withHref(input.getHref())
                        .withBusinessCategory(input.getBusinessCategory())
                )
                .withIsS2sTrackingEnabled(false)
                .withIsRecommendationsManagementEnabled(input.getIsRecommendationsManagementEnabled())
                .withIsPriceRecommendationsManagementEnabled(input.getIsPriceRecommendationsManagementEnabled())
                .withMinusKeywords(minusKeywords);

        var campaign = (DynamicCampaign) campaignToCoreCampaign(gdCampaign, converterContext);
        return campaign
                .withMasterCid(masterCid)
                .withSource(source)
                .withMetatype(metatype)
                .withCalltrackingSettingsId(input.getCalltrackingSettingsId())
                .withWidgetPartnerId(input.getWidgetPartnerId());
    }

    public static SmartCampaign toSmartCampaign(Long masterCid,
                                                GdUcCampaignMutationInput input,
                                                String userEmail,
                                                List<String> minusKeywords,
                                                Integer counterId,
                                                CampaignConverterContext converterContext,
                                                CampaignAttributionModel defaultAttributionModel) {
        final var source = input.getSource() != null ? input.getSource() : CampaignSource.UAC;
        final var metatype = nvl(input.getIsEcom(), false) ? CampaignMetatype.ECOM : CampaignMetatype.DEFAULT_;
        var notificationSettings = defaultGdCampaignNotification(userEmail);
        notificationSettings.getEmailSettings()
                .withStopByReachDailyBudget(false)
                .withXlsReady(false);
        var gdCampaign = new GdAddSmartCampaign()
                .withName(input.getName())
                .withBiddingStrategy(getCampaignStrategy(input, PERFORMANCE))
                .withMeaningfulGoals(input.getMeaningfulGoals())
                .withNotification(notificationSettings)
                .withAttributionModel(toGdAttributionModel(defaultAttributionModel))
                .withHasAddMetrikaTagToUrl(CampaignConstants.DEFAULT_ADD_METRIKA_TAG_TO_URL)
                .withContextLimit(CampaignConstants.MIN_CONTEXT_LIMIT)
                .withMetrikaCounters(List.of(counterId))
                .withStartDate(LocalDate.now())
                .withTimeTarget(nvl(input.getTimeTarget(), defaultGdTimeTarget()))
                .withIsSimplifiedStrategyViewEnabled(true)
                .withIsS2sTrackingEnabled(false)
                .withIsRecommendationsManagementEnabled(input.getIsRecommendationsManagementEnabled())
                .withIsPriceRecommendationsManagementEnabled(input.getIsPriceRecommendationsManagementEnabled())
                .withAdditionalData(new GdAddUpdateCampaignAdditionalData()
                        .withHref(input.getHref())
                        .withBusinessCategory(input.getBusinessCategory())
                )
                .withMinusKeywords(minusKeywords);

        var campaign = (SmartCampaign) campaignToCoreCampaign(gdCampaign, converterContext);
        return campaign
                .withIsServiceRequested(null)
                .withMasterCid(masterCid)
                .withSource(source)
                .withMetatype(metatype)
                .withCalltrackingSettingsId(input.getCalltrackingSettingsId())
                .withWidgetPartnerId(input.getWidgetPartnerId());
    }

    private static boolean isCpaStrategy(GdUcCampaignMutationInput input) {
        return input.getAvgCpa() != null;
    }

    private static boolean isCrrStrategy(GdUcCampaignMutationInput input) {
        return input.getCrr() != null && input.getCrr() > 0;
    }

    private static GdCampaignBiddingStrategy getCampaignStrategy(GdUcCampaignMutationInput input,
                                                                 CampaignType campaignType) {
        if (isCpaStrategy(input)) { // максимум конверсий (одна цель), оплата за конверсии
            GdCampaignPlatform campaignPlatform;
            switch (campaignType) {
                case DYNAMIC:
                    campaignPlatform = GdCampaignPlatform.SEARCH;
                    break;
                case PERFORMANCE:
                    campaignPlatform = GdCampaignPlatform.CONTEXT;
                    break;
                default:
                    campaignPlatform = GdCampaignPlatform.BOTH;
            }
            return new GdCampaignBiddingStrategy()
                    .withStrategyName(campaignType == PERFORMANCE ? AUTOBUDGET_AVG_CPA_PER_FILTER : AUTOBUDGET_AVG_CPA)
                    .withStrategy(campaignType == PERFORMANCE ? GdCampaignStrategy.AUTOBUDGET_AVG_CPA_PER_FILTER : null)
                    .withPlatform(campaignPlatform)
                    .withStrategyData(new GdCampaignStrategyData()
                            .withGoalId(input.getGoalId())
                            .withAvgCpa(campaignType != PERFORMANCE ? input.getAvgCpa() : null)
                            .withFilterAvgCpa(campaignType == PERFORMANCE ? input.getAvgCpa() : null)
                            .withSum(getBudget(input, campaignType))
                            .withPayForConversion(true)
                    );
        } else if (isCrrStrategy(input)) { // максимум конверсий (несколько целей), оплата за конверсии
            return new GdCampaignBiddingStrategy()
                    .withStrategyName(AUTOBUDGET_CRR)
                    .withPlatform(GdCampaignPlatform.BOTH)
                    .withStrategyData(new GdCampaignStrategyData()
                            .withGoalId(input.getGoalId())
                            .withCrr(input.getCrr())
                            .withSum(getBudget(input, campaignType))
                            .withPayForConversion(true)
                    );
        } else { // максимум конверсий (несколько целей) или переходов в зависимости от наличия цели, оплата за переходы
            return new GdCampaignBiddingStrategy()
                    .withStrategyName(GdCampaignStrategyName.AUTOBUDGET)
                    .withPlatform(GdCampaignPlatform.BOTH)
                    .withStrategyData(new GdCampaignStrategyData()
                            .withGoalId(input.getGoalId())
                            .withSum(getBudget(input, campaignType))
                    );
        }
    }

    private static BigDecimal getBudget(GdUcCampaignMutationInput input,
                                        CampaignType campaignType) {
        if (!nvl(input.getIsEcom(), false) || FeatureHelper.feature(FeatureName.ECOM_UC_NEW_BACKEND_ENABLED).enabled()) {
            return input.getBudget();
        }
        var budget = input.getBudget();
        var budgetForText = budget.divide(new BigDecimal(3), budget.scale(), RoundingMode.UP);
        if (campaignType == TEXT) {
            return budgetForText;
        } else {
            return budget
                    .subtract(budgetForText)
                    .divide(new BigDecimal(2), budget.scale() + 1, RoundingMode.DOWN);
        }
    }

    @SuppressWarnings("java:S3252")
    public static ModelChanges<TextCampaign> toCoreCampaignModelChanges(GdUpdateUcCampaignInput input,
                                                                        String userEmail,
                                                                        List<String> minusKeywords,
                                                                        CampaignConverterContext converterContext,
                                                                        CampaignAttributionModel defaultAttributionModel,
                                                                        boolean advancedGeoTargeting) {
        TextCampaign campaign = toCoreTextCampaign(input, userEmail, minusKeywords, converterContext,
                defaultAttributionModel,
                advancedGeoTargeting
        );
        return new ModelChanges<>(input.getId(), TextCampaign.class)
                .process(campaign.getName(), TextCampaign.NAME)
                .process(campaign.getStrategy(), TextCampaign.STRATEGY)
                .process(campaign.getMetrikaCounters(), TextCampaign.METRIKA_COUNTERS)
                .process(campaign.getDefaultPermalinkId(), TextCampaign.DEFAULT_PERMALINK_ID)
                .process(campaign.getDefaultTrackingPhoneId(), TextCampaign.DEFAULT_TRACKING_PHONE_ID)
                .process(campaign.getHref(), TextCampaign.HREF)
                .process(campaign.getBusinessCategory(), TextCampaign.BUSINESS_CATEGORY)
                .process(campaign.getBudgetDisplayFormat(), TextCampaign.BUDGET_DISPLAY_FORMAT)
                .process(campaign.getMinusKeywords(), TextCampaign.MINUS_KEYWORDS)
                .process(campaign.getMeaningfulGoals(), TextCampaign.MEANINGFUL_GOALS)
                .process(campaign.getCalltrackingSettingsId(), TextCampaign.CALLTRACKING_SETTINGS_ID)
                .process(campaign.getTimeTarget(), TextCampaign.TIME_TARGET)
                .processNotNull(campaign.getIsRecommendationsManagementEnabled(),
                        CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .processNotNull(campaign.getIsPriceRecommendationsManagementEnabled(),
                        CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .process(campaign.getTimeZoneId(), TextCampaign.TIME_ZONE_ID);
    }

    @SuppressWarnings("java:S3252")
    public static ModelChanges<DynamicCampaign> toCoreDynamicCampaignModelChanges(
            Long cid, Long masterCid, GdUpdateUcCampaignInput input, String userEmail, List<String> minusKeywords,
            CampaignConverterContext converterContext, CampaignAttributionModel defaultAttributionModel) {
        var campaign = toDynamicCampaign(masterCid, input, userEmail, minusKeywords, converterContext,
                defaultAttributionModel);
        return new ModelChanges<>(cid, DynamicCampaign.class)
                .process(campaign.getName(), CommonCampaign.NAME)
                .process(campaign.getStrategy(), CampaignWithCustomStrategy.STRATEGY)
                .process(campaign.getMetrikaCounters(), CampaignWithMetrikaCounters.METRIKA_COUNTERS)
                .process(campaign.getDefaultPermalinkId(), CampaignWithOrganization.DEFAULT_PERMALINK_ID)
                .process(campaign.getHref(), CampaignWithAdditionalDataHref.HREF)
                .process(campaign.getMinusKeywords(), CampaignWithMinusKeywords.MINUS_KEYWORDS)
                .process(campaign.getMeaningfulGoals(), CampaignWithMeaningfulGoals.MEANINGFUL_GOALS)
                .process(campaign.getTimeTarget(), CommonCampaign.TIME_TARGET)
                .processNotNull(campaign.getIsRecommendationsManagementEnabled(),
                        CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .processNotNull(campaign.getIsPriceRecommendationsManagementEnabled(),
                        CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .process(campaign.getTimeZoneId(), CommonCampaign.TIME_ZONE_ID);
    }

    @SuppressWarnings({"java:S3252", "parameternumber"})
    public static ModelChanges<SmartCampaign> toCoreSmartCampaignModelChanges(
            Long cid, Long masterCid, GdUpdateUcCampaignInput input,
            String userEmail, List<String> minusKeywords, Integer counterId,
            CampaignConverterContext converterContext,
            CampaignAttributionModel defaultAttributionModel) {
        var campaign = toSmartCampaign(masterCid, input, userEmail, minusKeywords, counterId, converterContext,
                defaultAttributionModel);
        return new ModelChanges<>(cid, SmartCampaign.class)
                .process(campaign.getName(), CommonCampaign.NAME)
                .process(campaign.getStrategy(), CampaignWithCustomStrategy.STRATEGY)
                .process(campaign.getMetrikaCounters(), CampaignWithMetrikaCounters.METRIKA_COUNTERS)
                .process(campaign.getHref(), CampaignWithAdditionalDataHref.HREF)
                .process(campaign.getMinusKeywords(), CampaignWithMinusKeywords.MINUS_KEYWORDS)
                .process(campaign.getMeaningfulGoals(), CampaignWithMeaningfulGoals.MEANINGFUL_GOALS)
                .process(campaign.getTimeTarget(), CommonCampaign.TIME_TARGET)
                .processNotNull(campaign.getIsRecommendationsManagementEnabled(),
                        CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .processNotNull(campaign.getIsPriceRecommendationsManagementEnabled(),
                        CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .process(campaign.getTimeZoneId(), CommonCampaign.TIME_ZONE_ID);
    }

    public static ComplexTextAdGroup toComplexTextAdGroup(
            GdUcCampaignMutationInput input,
            Long campaignId,
            RetargetingCondition retargetingCondition,
            boolean tabletBidModifierEnabled
    ) {
        return toComplexTextAdGroupInner(input, campaignId,
                null, toTextBanner(input, null), null, retargetingCondition, tabletBidModifierEnabled, false);
    }

    public static ComplexTextAdGroup toComplexTextAdGroupWithDynamicBanners(
            GdUcCampaignMutationInput input,
            Long campaignId,
            RetargetingCondition retargetingCondition,
            boolean tabletBidModifierEnabled
    ) {
        return toComplexTextAdGroupInner(input, campaignId,
                null, toDynamicBanner(input, null), null, retargetingCondition, tabletBidModifierEnabled, true);
    }

    public static ComplexTextAdGroup toComplexTextAdGroupWithSmartBanners(
            GdUcCampaignMutationInput input,
            Long campaignId,
            RetargetingCondition retargetingCondition,
            boolean tabletBidModifierEnabled
    ) {
        return toComplexTextAdGroupInner(input, campaignId,
                null, toSmartBanner(null), null, retargetingCondition, tabletBidModifierEnabled, true);
    }

    public static ComplexTextAdGroup toComplexTextAdGroup(
            GdUpdateUcCampaignInput input, Long adGroupId, Long bannerId,
            List<Keyword> existingKeywords, RetargetingCondition retargetingCondition,
            boolean tabletBidmodifierEnabled
    ) {
        return toComplexTextAdGroupInner(input, input.getId(), adGroupId, toTextBanner(input, bannerId),
                existingKeywords, retargetingCondition, tabletBidmodifierEnabled, false);
    }

    public static ComplexTextAdGroup toComplexTextAdGroupWithDynamicBanner(
            GdUpdateUcCampaignInput input, Long adGroupId, Long bannerId,
            List<Keyword> existingKeywords, RetargetingCondition retargetingCondition,
            boolean tabletBidmodifierEnabled
    ) {
        return toComplexTextAdGroupInner(input, input.getId(), adGroupId, toDynamicBanner(input, bannerId),
                existingKeywords, retargetingCondition, tabletBidmodifierEnabled, true);
    }

    public static ComplexTextAdGroup toComplexTextAdGroupWithSmartBanner(
            GdUpdateUcCampaignInput input, Long adGroupId, Long bannerId,
            List<Keyword> existingKeywords, RetargetingCondition retargetingCondition,
            boolean tabletBidmodifierEnabled
    ) {
        return toComplexTextAdGroupInner(input, input.getId(), adGroupId, toSmartBanner(bannerId),
                existingKeywords, retargetingCondition, tabletBidmodifierEnabled, true);
    }

    public static ComplexDynamicAdGroup toComplexDynamicAdGroup(GdUcCampaignMutationInput input,
                                                                Long campaignId,
                                                                boolean tabletBidmodifierEnabled) {
        return toComplexDynamicAdGroupInner(input, campaignId, null, null, tabletBidmodifierEnabled);
    }

    public static ComplexDynamicAdGroup toComplexDynamicAdGroup(
            GdUcCampaignMutationInput input, Long campaignId, Long adGroupId, Long bannerId,
            boolean tabletBidmodifierEnabled
    ) {
        return toComplexDynamicAdGroupInner(input, campaignId, adGroupId, bannerId, tabletBidmodifierEnabled);
    }

    private static ComplexDynamicAdGroup toComplexDynamicAdGroupInner(GdUcCampaignMutationInput input,
                                                                      Long campaignId, Long adGroupId, Long bannerId,
                                                                      boolean tabletBidModifierEnabled) {
        var complexBidModifier = toComplexBidModifier(
                input.getSocdem(),
                input.getDeviceTypes(),
                campaignId,
                tabletBidModifierEnabled);

        var banner = new DynamicBanner()
                .withId(bannerId)
                .withHref(input.getHref())
                .withBody(input.getAdText())
                .withImageHash(input.getImageHash());
        return new ComplexDynamicAdGroup()
                .withAdGroup(new DynamicFeedAdGroup()
                        .withId(adGroupId)
                        .withCampaignId(campaignId)
                        .withName(input.getName())
                        .withType(AdGroupType.DYNAMIC)
                        .withGeo(input.getGeo())
                        .withHyperGeoId(input.getHyperGeoId())
                        .withTrackingParams(input.getTrackingParams())
                        .withFeedId(input.getFeedId())
                        .withBanners(List.of(banner)))
                .withComplexBidModifier(complexBidModifier);
    }

    public static ComplexPerformanceAdGroup toComplexPerformanceAdGroup(GdUcCampaignMutationInput input,
                                                                        Long campaignId,
                                                                        boolean tabletBidModifierEnabled) {
        return toComplexPerformanceAdGroupInner(input, campaignId, null, tabletBidModifierEnabled);
    }

    public static ComplexPerformanceAdGroup toComplexPerformanceAdGroup(
            GdUcCampaignMutationInput input, Long campaignId, Long adGroupId,
            boolean tabletBidModifierEnabled) {
        return toComplexPerformanceAdGroupInner(input, campaignId, adGroupId, tabletBidModifierEnabled);
    }

    private static ComplexPerformanceAdGroup toComplexPerformanceAdGroupInner(
            GdUcCampaignMutationInput input, Long campaignId, Long adGroupId, boolean tabletBidModifierEnabled) {
        var complexBidModifier = toComplexBidModifier(
                input.getSocdem(),
                input.getDeviceTypes(),
                campaignId,
                tabletBidModifierEnabled);

        // Здесь мы не создаём фиктивные смарт баннеры, потому что в еком сценарии они создаются по дефолтным креативам
        // и пользовательского ввода там вообще нет, соответственно, валидировать нечего
        var group = new PerformanceAdGroup()
                .withId(adGroupId)
                .withCampaignId(campaignId)
                .withName(input.getName())
                .withType(AdGroupType.PERFORMANCE)
                .withGeo(input.getGeo())
                .withHyperGeoId(input.getHyperGeoId())
                .withTrackingParams(input.getTrackingParams())
                .withFeedId(input.getFeedId());
        return new ComplexPerformanceAdGroup()
                .withAdGroup(group)
                .withComplexBidModifier(complexBidModifier);
    }

    private static ComplexTextAdGroup toComplexTextAdGroupInner(
            GdUcCampaignMutationInput input,
            Long campaignId, Long adGroupId, BannerWithSystemFields banner,
            List<Keyword> existingKeywords,
            RetargetingCondition retargetingCondition,
            boolean tabletBidModifierEnabled,
            boolean ecomUcNewBackendEnabled
    ) {
        List<GdUpdateAdGroupKeywordItem> keywordItems = mapList(input.getKeywords(),
                keyword -> new GdUpdateAdGroupKeywordItem().withPhrase(keyword));

        ComplexBidModifier complexBidModifier = toComplexBidModifier(input.getSocdem(), input.getDeviceTypes(),
                campaignId, tabletBidModifierEnabled);

        var adGroup = new TextAdGroup()
                .withId(adGroupId)
                .withType(AdGroupType.BASE)
                .withCampaignId(campaignId)
                .withName("New group") // по задумке имя группы нигде не светится
                .withGeo(input.getGeo())
                .withHyperGeoId(input.getHyperGeoId());

        if (ecomUcNewBackendEnabled && Boolean.TRUE.equals(input.getIsEcom())) {
            adGroup.withFeedId(input.getFeedId());
        }

        return new ComplexTextAdGroup()
                .withAdGroup(adGroup)
                .withComplexBanners(singletonList(new ComplexBanner()
                        .withBanner(banner)
                        .withSitelinkSet(CollectionUtils.isEmpty(input.getSitelinks()) ? null :
                                // Добавляем к сайтлинкам трекинговые параметры
                                SitelinkConverter.toCoreSet(new GdAddSitelinkSet().withSitelinks(input.getSitelinks()),
                                        input.getTrackingParams())
                        )
                ))
                .withRelevanceMatches(singletonList(new RelevanceMatch()))
                .withKeywords(toCoreKeywords(keywordItems, existingKeywords))
                .withComplexBidModifier(complexBidModifier)
                .withRetargetingCondition(retargetingCondition)
                .withTargetInterests(retargetingCondition != null ?
                        List.of(retargetingCondition.getTargetInterest()) : null);
    }

    private static BannerWithSystemFields toTextBanner(GdUcCampaignMutationInput input, @Nullable Long bannerId) {
        return new TextBanner()
                .withId(bannerId)
                .withTitle(input.getAdTitle())
                .withBody(input.getAdText())
                .withHref(getHrefWithTrackingParams(input.getHref(), input.getTrackingParams()))
                .withImageHash(input.getImageHash())
                .withCreativeId(input.getCreativeId())
                .withPermalinkId(input.getPermalinkId())
                .withPhoneId(input.getPhoneId());
    }

    private static BannerWithSystemFields toDynamicBanner(GdUcCampaignMutationInput input, @Nullable Long bannerId) {
        return new DynamicBanner()
                .withId(bannerId)
                .withBody(input.getAdText())
                .withHref(getHrefWithTrackingParams(input.getHref(), input.getTrackingParams()));
    }

    private static BannerWithSystemFields toSmartBanner(@Nullable Long bannerId) {
        return new PerformanceBanner()
                .withId(bannerId);
    }

    // https://a.yandex-team.ru/arc_vcs/direct/core/src/main/java/ru/yandex/direct/core/entity/uac/UacCommonUtils.kt?rev=r9172965#L59
    private static ComplexBidModifier toComplexBidModifier(GdTouchSocdemInput touchSocdemInput,
                                                           Set<GdDeviceType> deviceTypes,
                                                           Long campaignId,
                                                           boolean tabletBidModifierEnabled) {

        GdiTouchSocdem innerTouchSocdem = TouchSocdemConverter.toInnerTouchSocdem(touchSocdemInput);
        ComplexBidModifier complexBidModifier = new ComplexBidModifier()
                .withDemographyModifier(GridTouchSocdemConverter.toSocdemBidModifier(innerTouchSocdem));

        if (CollectionUtils.isEmpty(deviceTypes) || deviceTypes.contains(GdDeviceType.ALL)) {
            return complexBidModifier;
        }

        if (tabletBidModifierEnabled) {
            if (!deviceTypes.contains(GdDeviceType.DESKTOP)) {
                var bidModifierDesktopOnly = new BidModifierDesktopOnly()
                        .withCampaignId(campaignId)
                        .withEnabled(true)
                        .withDesktopOnlyAdjustment(new BidModifierDesktopOnlyAdjustment().withPercent(0))
                        .withType(BidModifierType.DESKTOP_ONLY_MULTIPLIER);
                complexBidModifier.withDesktopOnlyModifier(bidModifierDesktopOnly);
            }
            if (!deviceTypes.contains(GdDeviceType.TABLET)) {
                var bidModifierTablet = new BidModifierTablet()
                        .withCampaignId(campaignId)
                        .withEnabled(true)
                        .withTabletAdjustment(new BidModifierTabletAdjustment().withPercent(0))
                        .withType(BidModifierType.TABLET_MULTIPLIER);
                complexBidModifier.withTabletModifier(bidModifierTablet);
            }
        } else {
            if (!deviceTypes.contains(GdDeviceType.DESKTOP) && !deviceTypes.contains(GdDeviceType.TABLET)) {
                var bidModifierDesktop = new BidModifierDesktop()
                        .withCampaignId(campaignId)
                        .withEnabled(true)
                        .withDesktopAdjustment(new BidModifierDesktopAdjustment().withPercent(0))
                        .withType(BidModifierType.DESKTOP_MULTIPLIER);
                complexBidModifier.withDesktopModifier(bidModifierDesktop);
            }
        }

        if (!deviceTypes.contains(GdDeviceType.PHONE)) {
            var bidModifierMobile = new BidModifierMobile()
                    .withCampaignId(campaignId)
                    .withEnabled(true)
                    .withMobileAdjustment(new BidModifierMobileAdjustment().withPercent(0))
                    .withType(BidModifierType.MOBILE_MULTIPLIER);
            complexBidModifier.withMobileModifier(bidModifierMobile);
        }
        return complexBidModifier;
    }
}
