package ru.yandex.direct.grid.processing.service.campaign.converter.type;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.processing.model.campaign.facelift.GdAddUpdateCampaignAdditionalData;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddCpmPriceCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignEmailSettingsRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignSmsSettingsRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdShowsFrequencyLimitRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateCpmPriceCampaign;
import ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataConverter;
import ru.yandex.direct.grid.processing.service.campaign.CampaignConverterContext;
import ru.yandex.direct.model.ModelChanges;

import static ru.yandex.direct.core.entity.campaign.CampaignUtils.getDisabledDomains;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignAttributionModel;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toContentLanguage;
import static ru.yandex.direct.grid.model.entity.campaign.converter.PriceFlightDataConverter.toPriceFlightTargetingsSnapshot;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.getDisabledSsp;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.getImpressionRateIntervalDays;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toBrandSafetyCategories;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toCampaignWarnPlaceInterval;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toMeasurers;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toMetrikaCounters;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toSmsFlags;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toTimeInterval;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
public class CpmPriceCampaignConverterSupport implements CampaignConverterSupport<CpmPriceCampaign,
        GdUpdateCpmPriceCampaign,
        GdAddCpmPriceCampaign> {

    private final CampaignConverterContext converterContext;

    CpmPriceCampaignConverterSupport(CampaignConverterContext converterContext) {
        this.converterContext = converterContext;
    }

    @Override
    public CpmPriceCampaign toCoreCampaign(GdAddCpmPriceCampaign campaign) {
        GdCampaignSmsSettingsRequest smsSettings = campaign.getNotification().getSmsSettings();
        GdCampaignEmailSettingsRequest emailSettings = campaign.getNotification().getEmailSettings();

        List<String> knownSsp = converterContext.getSspPlatforms();
        List<String> disabledSsp = getDisabledSsp(converterContext.getHostingsHandler(), knownSsp,
                campaign.getDisabledPlaces());
        List<String> disabledDomains = getDisabledDomains(knownSsp, campaign.getDisabledPlaces());

        return new CpmPriceCampaign()
                .withType(CampaignType.CPM_PRICE)
                .withName(campaign.getName())
                .withStartDate(campaign.getStartDate())
                .withEndDate(campaign.getEndDate())
                .withMetrikaCounters(toMetrikaCounters(campaign.getMetrikaCounters()))
                .withDisabledIps(campaign.getDisabledIps())
                .withHasAddMetrikaTagToUrl(campaign.getHasAddMetrikaTagToUrl())
                .withHasAddOpenstatTagToUrl(campaign.getHasAddOpenstatTagToUrl())
                .withSmsTime(toTimeInterval(smsSettings.getSmsTime()))
                .withSmsFlags(toSmsFlags(smsSettings.getEnableEvents()))
                .withEmail(emailSettings.getEmail())
                .withWarningBalance(emailSettings.getWarningBalance())
                .withEnableSendAccountNews(emailSettings.getSendAccountNews())
                .withEnablePausedByDayBudgetEvent(emailSettings.getStopByReachDailyBudget())
                .withEnableOfflineStatNotice(emailSettings.getXlsReady())
                .withEnableCheckPositionEvent(emailSettings.getCheckPositionInterval() != null)
                .withContentLanguage(toContentLanguage(campaign.getContentLanguage()))
                .withAttributionModel(toCampaignAttributionModel(campaign.getAttributionModel()))
                .withAutoProlongation(campaign.getAutoProlongation())
                .withCheckPositionIntervalEvent(toCampaignWarnPlaceInterval(emailSettings.getCheckPositionInterval()))
                .withTimeZoneId(ifNotNull(campaign.getTimeTarget(), GdTimeTarget::getIdTimeZone))
                .withPricePackageId(campaign.getPricePackageId())
                .withFlightTargetingsSnapshot(toPriceFlightTargetingsSnapshot(campaign.getFlightTargetingsSnapshot()))
                .withFlightOrderVolume(campaign.getFlightOrderVolume())
                .withImpressionRateCount(ifNotNull(campaign.getShowsFrequencyLimit(),
                        GdShowsFrequencyLimitRequest::getLimit))
                .withImpressionRateIntervalDays(getImpressionRateIntervalDays(campaign.getShowsFrequencyLimit()))
                .withRequireFiltrationByDontShowDomains(campaign.getRequireFiltrationByDontShowDomains())
                .withDisabledVideoPlacements(campaign.getDisabledVideoAdsPlaces())
                .withDisabledDomains(disabledDomains)
                .withDisabledSsp(disabledSsp)
                .withHref(ifNotNull(campaign.getAdditionalData(), GdAddUpdateCampaignAdditionalData::getHref))
                .withIsRecommendationsManagementEnabled(campaign.getIsRecommendationsManagementEnabled())
                .withIsPriceRecommendationsManagementEnabled(campaign.getIsPriceRecommendationsManagementEnabled())
                .withBrandSurveyId(campaign.getBrandSurveyId())
                .withBrandSurveyName(campaign.getBrandSurveyName())
                .withBrandSafetyCategories(toBrandSafetyCategories(
                        campaign.getBrandSafetyCategories(), campaign.getBrandSafety()))
                .withMeasurers(toMeasurers(campaign.getMeasurers()))
                .withBidModifiers(BidModifierDataConverter.toBidModifiers(campaign.getBidModifiers()))
                .withDisallowedPageIds(campaign.getDisallowedPageIds())
                .withIsAllowedOnAdultContent(campaign.getIsAllowedOnAdultContent())
                .withStrategyId(campaign.getStrategyId());
    }

    @Override
    public CpmPriceCampaign toCoreCampaign(GdUpdateCpmPriceCampaign campaign) {
        // minusKeywords есть в GdUpdateCpmPriceCampaign, пришли из базового GdUpdateAbstractCampaign
        // но в прайсовых нам minusKeywords не нужны, игнорируем их здесь
        GdCampaignSmsSettingsRequest smsSettings = campaign.getNotification().getSmsSettings();
        GdCampaignEmailSettingsRequest emailSettings = campaign.getNotification().getEmailSettings();

        List<String> knownSsp = converterContext.getSspPlatforms();
        List<String> disabledSsp = getDisabledSsp(converterContext.getHostingsHandler(), knownSsp,
                campaign.getDisabledPlaces());
        List<String> disabledDomains = getDisabledDomains(knownSsp, campaign.getDisabledPlaces());

        return new CpmPriceCampaign()
                .withId(campaign.getId())
                .withType(CampaignType.CPM_PRICE)
                .withName(campaign.getName())
                .withStartDate(campaign.getStartDate())
                .withEndDate(campaign.getEndDate())
                .withMetrikaCounters(toMetrikaCounters(campaign.getMetrikaCounters()))
                .withDisabledIps(campaign.getDisabledIps())
                .withHasAddMetrikaTagToUrl(campaign.getHasAddMetrikaTagToUrl())
                .withHasAddOpenstatTagToUrl(campaign.getHasAddOpenstatTagToUrl())
                .withSmsTime(toTimeInterval(smsSettings.getSmsTime()))
                .withSmsFlags(toSmsFlags(smsSettings.getEnableEvents()))
                .withEmail(emailSettings.getEmail())
                .withWarningBalance(emailSettings.getWarningBalance())
                .withEnableSendAccountNews(emailSettings.getSendAccountNews())
                .withEnablePausedByDayBudgetEvent(emailSettings.getStopByReachDailyBudget())
                .withEnableOfflineStatNotice(emailSettings.getXlsReady())
                .withEnableCheckPositionEvent(emailSettings.getCheckPositionInterval() != null)
                .withContentLanguage(toContentLanguage(campaign.getContentLanguage()))
                .withAttributionModel(toCampaignAttributionModel(campaign.getAttributionModel()))
                .withAutoProlongation(campaign.getAutoProlongation())
                .withCheckPositionIntervalEvent(toCampaignWarnPlaceInterval(emailSettings.getCheckPositionInterval()))
                .withTimeZoneId(ifNotNull(campaign.getTimeTarget(), GdTimeTarget::getIdTimeZone))
                // timeTarget не выставляем, т.к. мы не принимаем его от пользователя
                .withFlightOrderVolume(campaign.getFlightOrderVolume())
                .withImpressionRateCount(campaign.getShowsFrequencyLimit() == null ?
                        null : campaign.getShowsFrequencyLimit().getLimit())
                .withImpressionRateIntervalDays(getImpressionRateIntervalDays(campaign.getShowsFrequencyLimit()))
                .withRequireFiltrationByDontShowDomains(campaign.getRequireFiltrationByDontShowDomains())
                .withDisabledVideoPlacements(campaign.getDisabledVideoAdsPlaces())
                .withDisabledDomains(disabledDomains)
                .withDisabledSsp(disabledSsp)
                .withHref(ifNotNull(campaign.getAdditionalData(), GdAddUpdateCampaignAdditionalData::getHref))
                .withBrandSurveyId(campaign.getBrandSurveyId())
                .withBrandSurveyName(campaign.getBrandSurveyName())
                .withBrandSafetyCategories(toBrandSafetyCategories(
                        campaign.getBrandSafetyCategories(), campaign.getBrandSafety()))
                .withMeasurers(toMeasurers(campaign.getMeasurers()))
                .withBidModifiers(BidModifierDataConverter.toBidModifiers(campaign.getBidModifiers()))
                .withDisallowedPageIds(campaign.getDisallowedPageIds())
                .withIsRecommendationsManagementEnabled(campaign.getIsRecommendationsManagementEnabled())
                .withIsPriceRecommendationsManagementEnabled(campaign.getIsPriceRecommendationsManagementEnabled())
                .withIsAllowedOnAdultContent(campaign.getIsAllowedOnAdultContent())
                .withStrategyId(campaign.getStrategyId());
    }

    @Override
    @SuppressWarnings("java:S3252")
    public ModelChanges<CpmPriceCampaign> toModelChanges(CpmPriceCampaign campaign) {
        ModelChanges<CpmPriceCampaign> changes = new ModelChanges<>(campaign.getId(), CpmPriceCampaign.class)
                .process(campaign.getType(), CpmPriceCampaign.TYPE)
                .process(campaign.getName(), CpmPriceCampaign.NAME)
                .process(campaign.getStartDate(), CpmPriceCampaign.START_DATE)
                .process(campaign.getEndDate(), CpmPriceCampaign.END_DATE)
                .process(campaign.getMetrikaCounters(), CpmPriceCampaign.METRIKA_COUNTERS)
                .process(campaign.getDisabledIps(), CpmPriceCampaign.DISABLED_IPS)
                .process(campaign.getHasAddMetrikaTagToUrl(), CpmPriceCampaign.HAS_ADD_METRIKA_TAG_TO_URL)
                .process(campaign.getHasAddOpenstatTagToUrl(), CpmPriceCampaign.HAS_ADD_OPENSTAT_TAG_TO_URL)
                .process(campaign.getSmsTime(), CpmPriceCampaign.SMS_TIME)
                .process(campaign.getSmsFlags(), CpmPriceCampaign.SMS_FLAGS)
                .process(campaign.getEmail(), CpmPriceCampaign.EMAIL)
                .process(campaign.getWarningBalance(), CpmPriceCampaign.WARNING_BALANCE)
                .process(campaign.getEnableSendAccountNews(), CpmPriceCampaign.ENABLE_SEND_ACCOUNT_NEWS)
                .process(campaign.getEnablePausedByDayBudgetEvent(), CpmPriceCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT)
                .process(campaign.getEnableOfflineStatNotice(), CpmPriceCampaign.ENABLE_OFFLINE_STAT_NOTICE)
                .process(campaign.getEnableCheckPositionEvent(), CpmPriceCampaign.ENABLE_CHECK_POSITION_EVENT)
                .process(campaign.getAttributionModel(), CpmPriceCampaign.ATTRIBUTION_MODEL)
                .process(campaign.getAutoProlongation(), CpmPriceCampaign.AUTO_PROLONGATION)
                .process(campaign.getCheckPositionIntervalEvent(), CpmPriceCampaign.CHECK_POSITION_INTERVAL_EVENT)
                .process(campaign.getTimeZoneId(), CpmPriceCampaign.TIME_ZONE_ID)
                // timeTarget не процессим, т.к. мы его не выставили
                .process(campaign.getFlightOrderVolume(), CpmPriceCampaign.FLIGHT_ORDER_VOLUME)
                .process(campaign.getImpressionRateCount(), CpmPriceCampaign.IMPRESSION_RATE_COUNT)
                .process(campaign.getImpressionRateIntervalDays(), CpmPriceCampaign.IMPRESSION_RATE_INTERVAL_DAYS)
                .process(campaign.getDisabledVideoPlacements(), CpmPriceCampaign.DISABLED_VIDEO_PLACEMENTS)
                .process(campaign.getDisabledDomains(), CpmPriceCampaign.DISABLED_DOMAINS)
                .process(campaign.getDisabledSsp(), CpmPriceCampaign.DISABLED_SSP)
                .process(campaign.getHref(), CpmPriceCampaign.HREF)
                .process(campaign.getBidModifiers(), CpmPriceCampaign.BID_MODIFIERS)
                .processNotNull(campaign.getBrandSafetyCategories(), CpmPriceCampaign.BRAND_SAFETY_CATEGORIES)
                .processNotNull(campaign.getMeasurers(), CpmBannerCampaign.MEASURERS)
                .processNotNull(campaign.getIsRecommendationsManagementEnabled(),
                        CpmBannerCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .processNotNull(campaign.getIsPriceRecommendationsManagementEnabled(),
                        CpmBannerCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .processNotNull(campaign.getIsAllowedOnAdultContent(), CpmPriceCampaign.IS_ALLOWED_ON_ADULT_CONTENT)
                .processNotNull(campaign.getStrategyId(), CpmPriceCampaign.STRATEGY_ID);

        if (converterContext.getShouldProcessProperties().contains(CpmPriceCampaign.CONTENT_LANGUAGE)) {
            changes.process(campaign.getContentLanguage(), CpmPriceCampaign.CONTENT_LANGUAGE);
        }
        if (converterContext.getShouldProcessProperties().contains(CpmPriceCampaign.BRAND_SURVEY_ID)) {
            changes.process(campaign.getBrandSurveyId(), CpmPriceCampaign.BRAND_SURVEY_ID);
        }
        if (converterContext.getShouldProcessProperties().contains(CpmPriceCampaign.BRAND_SURVEY_NAME)) {
            changes.process(campaign.getBrandSurveyName(), CpmPriceCampaign.BRAND_SURVEY_NAME);
        }
        if (campaign.getRequireFiltrationByDontShowDomains() != null) {
            changes.process(campaign.getRequireFiltrationByDontShowDomains(),
                    CpmPriceCampaign.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS);
        }
        if (campaign.getDisallowedPageIds() != null) {
            changes.process(campaign.getDisallowedPageIds(), CpmPriceCampaign.DISALLOWED_PAGE_IDS);
        }

        return changes;
    }
}
