package ru.yandex.direct.grid.processing.service.campaign.converter.type;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.InternalDistribCampaign;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddInternalDistribCampaign;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateInternalDistribCampaign;
import ru.yandex.direct.model.ModelChanges;

import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignAttributionModel;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toRfCloseByClickType;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toCampaignWarnPlaceInterval;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toMeaningfulGoals;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toMetrikaCounters;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toSmsFlags;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toTimeInterval;
import static ru.yandex.direct.grid.processing.util.RfConverter.toCoreRfPeriod;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
public class InternalDistribCampaignConverterSupport implements CampaignConverterSupport<
        InternalDistribCampaign, GdUpdateInternalDistribCampaign, GdAddInternalDistribCampaign> {

    private static final CampaignType CAMPAIGN_TYPE = CampaignType.INTERNAL_DISTRIB;

    public InternalDistribCampaignConverterSupport() {
    }

    @Override
    public InternalDistribCampaign toCoreCampaign(GdUpdateInternalDistribCampaign campaign) {
        var smsSettings = campaign.getNotification().getSmsSettings();
        var emailSettings = campaign.getNotification().getEmailSettings();

        var resultingCampaign = new InternalDistribCampaign()
                .withId(campaign.getId())
                .withType(CAMPAIGN_TYPE)
                .withName(campaign.getName())
                .withStartDate(campaign.getStartDate())
                .withEndDate(campaign.getEndDate())
                .withDisabledIps(campaign.getDisabledIps())
                .withSmsTime(toTimeInterval(smsSettings.getSmsTime()))
                .withSmsFlags(toSmsFlags(smsSettings.getEnableEvents()))
                .withEmail(emailSettings.getEmail())
                .withWarningBalance(emailSettings.getWarningBalance())
                .withEnableSendAccountNews(emailSettings.getSendAccountNews())
                .withEnablePausedByDayBudgetEvent(emailSettings.getStopByReachDailyBudget())
                .withEnableOfflineStatNotice(emailSettings.getXlsReady())
                .withEnableCheckPositionEvent(emailSettings.getCheckPositionInterval() != null)
                .withCheckPositionIntervalEvent(toCampaignWarnPlaceInterval(emailSettings.getCheckPositionInterval()))
                .withTimeTarget(ifNotNull(campaign.getTimeTarget(), CampaignDataConverter::toTimeTarget))
                .withTimeZoneId(ifNotNull(campaign.getTimeTarget(), GdTimeTarget::getIdTimeZone))

                .withMetrikaCounters(toMetrikaCounters(campaign.getMetrikaCounters()))
                .withMeaningfulGoals(toMeaningfulGoals(campaign.getMeaningfulGoals()))
                .withAttributionModel(toCampaignAttributionModel(campaign.getAttributionModel()))
                .withRotationGoalId(campaign.getRotationGoalId())
                .withPageId(campaign.getPageId())
                .withRfCloseByClick(toRfCloseByClickType(campaign.getRfCloseByClick()))

                .withAbSegmentRetargetingConditionId(campaign.getAbSegmentRetargetingConditionId())
                .withAbSegmentStatisticRetargetingConditionId(campaign.getAbSegmentStatisticRetargetingConditionId())
                .withSectionIds(campaign.getSectionIds())
                .withAbSegmentGoalIds(campaign.getAbSegmentGoalIds());

        var impressionFrequencyControl = campaign.getImpressionFrequencyControl();
        if (impressionFrequencyControl != null) {
            resultingCampaign
                    .withImpressionRateIntervalDays(impressionFrequencyControl.getDays())
                    .withImpressionRateCount(impressionFrequencyControl.getImpressions())
                    .withMaxClicksCount(impressionFrequencyControl.getMaxClicksCount())
                    .withMaxClicksPeriod(toCoreRfPeriod(impressionFrequencyControl.getMaxClicksPeriod()))
                    .withMaxStopsCount(impressionFrequencyControl.getMaxStopsCount())
                    .withMaxStopsPeriod(toCoreRfPeriod(impressionFrequencyControl.getMaxStopsPeriod()));
        }

        return resultingCampaign;
    }

    @Override
    public InternalDistribCampaign toCoreCampaign(GdAddInternalDistribCampaign campaign) {
        var smsSettings = campaign.getNotification().getSmsSettings();
        var emailSettings = campaign.getNotification().getEmailSettings();

        var resultingCampaign = new InternalDistribCampaign()
                .withType(CAMPAIGN_TYPE)
                .withName(campaign.getName())
                .withStartDate(campaign.getStartDate())
                .withEndDate(campaign.getEndDate())
                .withDisabledIps(campaign.getDisabledIps())
                .withSmsTime(toTimeInterval(smsSettings.getSmsTime()))
                .withSmsFlags(toSmsFlags(smsSettings.getEnableEvents()))
                .withEmail(emailSettings.getEmail())
                .withWarningBalance(emailSettings.getWarningBalance())
                .withEnableSendAccountNews(emailSettings.getSendAccountNews())
                .withEnablePausedByDayBudgetEvent(emailSettings.getStopByReachDailyBudget())
                .withEnableOfflineStatNotice(emailSettings.getXlsReady())
                .withEnableCheckPositionEvent(emailSettings.getCheckPositionInterval() != null)
                .withCheckPositionIntervalEvent(toCampaignWarnPlaceInterval(emailSettings.getCheckPositionInterval()))
                .withTimeTarget(ifNotNull(campaign.getTimeTarget(), CampaignDataConverter::toTimeTarget))
                .withTimeZoneId(ifNotNull(campaign.getTimeTarget(), GdTimeTarget::getIdTimeZone))

                .withMetrikaCounters(toMetrikaCounters(campaign.getMetrikaCounters()))
                .withMeaningfulGoals(toMeaningfulGoals(campaign.getMeaningfulGoals()))
                .withAttributionModel(toCampaignAttributionModel(campaign.getAttributionModel()))
                .withRotationGoalId(campaign.getRotationGoalId())
                .withIsMobile(campaign.getIsMobile())
                .withPlaceId(campaign.getPlaceId())
                .withPageId(campaign.getPageId())
                .withRfCloseByClick(toRfCloseByClickType(campaign.getRfCloseByClick()))

                .withAbSegmentRetargetingConditionId(campaign.getAbSegmentRetargetingConditionId())
                .withAbSegmentStatisticRetargetingConditionId(campaign.getAbSegmentStatisticRetargetingConditionId())
                .withSectionIds(campaign.getSectionIds())
                .withAbSegmentGoalIds(campaign.getAbSegmentGoalIds());

        var impressionFrequencyControl = campaign.getImpressionFrequencyControl();
        if (impressionFrequencyControl != null) {
            resultingCampaign
                    .withImpressionRateIntervalDays(impressionFrequencyControl.getDays())
                    .withImpressionRateCount(impressionFrequencyControl.getImpressions())
                    .withMaxClicksCount(impressionFrequencyControl.getMaxClicksCount())
                    .withMaxClicksPeriod(toCoreRfPeriod(impressionFrequencyControl.getMaxClicksPeriod()))
                    .withMaxStopsCount(impressionFrequencyControl.getMaxStopsCount())
                    .withMaxStopsPeriod(toCoreRfPeriod(impressionFrequencyControl.getMaxStopsPeriod()));
        }

        return resultingCampaign;
    }

    @Override
    public ModelChanges<InternalDistribCampaign> toModelChanges(InternalDistribCampaign campaign) {
        return new ModelChanges<>(campaign.getId(), InternalDistribCampaign.class)
                .process(campaign.getType(), InternalDistribCampaign.TYPE)
                .process(campaign.getName(), InternalDistribCampaign.NAME)
                .process(campaign.getStartDate(), InternalDistribCampaign.START_DATE)
                .process(campaign.getEndDate(), InternalDistribCampaign.END_DATE)
                .process(campaign.getDisabledIps(), InternalDistribCampaign.DISABLED_IPS)
                .process(campaign.getSmsTime(), InternalDistribCampaign.SMS_TIME)
                .process(campaign.getSmsFlags(), InternalDistribCampaign.SMS_FLAGS)
                .process(campaign.getEmail(), InternalDistribCampaign.EMAIL)
                .process(campaign.getWarningBalance(), InternalDistribCampaign.WARNING_BALANCE)
                .process(campaign.getEnableSendAccountNews(), InternalDistribCampaign.ENABLE_SEND_ACCOUNT_NEWS)
                .process(campaign.getEnablePausedByDayBudgetEvent(),
                        InternalDistribCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT)
                .process(campaign.getEnableOfflineStatNotice(), InternalDistribCampaign.ENABLE_OFFLINE_STAT_NOTICE)
                .process(campaign.getEnableCheckPositionEvent(), InternalDistribCampaign.ENABLE_CHECK_POSITION_EVENT)
                .process(campaign.getCheckPositionIntervalEvent(),
                        InternalDistribCampaign.CHECK_POSITION_INTERVAL_EVENT)
                .process(campaign.getTimeTarget(), InternalDistribCampaign.TIME_TARGET)
                .process(campaign.getTimeZoneId(), InternalDistribCampaign.TIME_ZONE_ID)

                .process(campaign.getMetrikaCounters(), InternalDistribCampaign.METRIKA_COUNTERS)
                .process(campaign.getMeaningfulGoals(), InternalDistribCampaign.MEANINGFUL_GOALS)
                .process(campaign.getAttributionModel(), InternalDistribCampaign.ATTRIBUTION_MODEL)
                .process(campaign.getRotationGoalId(), InternalDistribCampaign.ROTATION_GOAL_ID)
                .process(campaign.getPageId(), InternalDistribCampaign.PAGE_ID)
                .process(campaign.getImpressionRateIntervalDays(),
                        InternalDistribCampaign.IMPRESSION_RATE_INTERVAL_DAYS)
                .process(campaign.getImpressionRateCount(), InternalDistribCampaign.IMPRESSION_RATE_COUNT)
                .process(campaign.getRfCloseByClick(), InternalDistribCampaign.RF_CLOSE_BY_CLICK)
                .process(campaign.getMaxClicksCount(), InternalDistribCampaign.MAX_CLICKS_COUNT)
                .process(campaign.getMaxClicksPeriod(), InternalDistribCampaign.MAX_CLICKS_PERIOD)
                .process(campaign.getMaxStopsCount(), InternalDistribCampaign.MAX_STOPS_COUNT)
                .process(campaign.getMaxStopsPeriod(), InternalDistribCampaign.MAX_STOPS_PERIOD)

                .process(campaign.getAbSegmentRetargetingConditionId(),
                        InternalDistribCampaign.AB_SEGMENT_RETARGETING_CONDITION_ID)
                .process(campaign.getAbSegmentStatisticRetargetingConditionId(),
                        InternalDistribCampaign.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID)
                .process(campaign.getSectionIds(), InternalDistribCampaign.SECTION_IDS)
                .process(campaign.getAbSegmentGoalIds(), InternalDistribCampaign.AB_SEGMENT_GOAL_IDS);
    }

}
