package ru.yandex.direct.grid.processing.service.campaign.copy

import org.assertj.core.util.VisibleForTesting
import org.springframework.stereotype.Service
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags
import ru.yandex.direct.core.entity.campaign.container.CampaignsCopySelectionCriteria
import ru.yandex.direct.core.entity.campaign.container.CampaignsCopySelectionCriteria.SelectionType
import ru.yandex.direct.core.entity.campaign.repository.CampaignCopyRepository
import ru.yandex.direct.core.entity.campaign.service.CopyCampaignService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.api.GdValidationResult
import ru.yandex.direct.grid.processing.model.campaign.GdCheckCopyCampaigns
import ru.yandex.direct.grid.processing.model.campaign.GdCheckCopyCampaignsPayload
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaigns
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsFilter
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsInterclient
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsInterclientPayload
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsPayload
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsSelectorType
import ru.yandex.direct.grid.processing.service.campaign.CampaignValidationService
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.grid.processing.util.ResponseConverter.getSuccessfullyResults
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.result.PathHelper.field
import ru.yandex.direct.validation.result.PathHelper.path
import ru.yandex.direct.validation.util.validateObject

@Service
class CampaignCopyMutationService(
    private val campaignCopyValidatorProvider: CampaignCopyValidatorProvider,
    private val campaignValidationService: CampaignValidationService,
    private val copyCampaignService: CopyCampaignService,
    private val gridValidationService: GridValidationService,
    private val userService: UserService,
    private val clientService: ClientService,
    private val shardHelper: ShardHelper,
    private val campaignCopyRepository: CampaignCopyRepository,
) {
    fun copyCampaigns(context: GridGraphQLContext, input: GdCopyCampaigns): GdCopyCampaignsPayload {
        val validator = campaignCopyValidatorProvider.copyCampaignsValidator()
        gridValidationService.applyValidator(validator, input, false)

        val operatorUid = context.operator.uid
        val clientId = context.subjectUser!!.clientId

        val copyResult: MassResult<Long> = copyCampaignService
            .copyCampaignsScheduled(operatorUid, clientId, input.campaignIds)

        val validationResult: GdValidationResult? = campaignValidationService
            .getValidationResult(copyResult, path(field(GdCopyCampaigns::campaignIds)))

        return GdCopyCampaignsPayload(
            campaignIdsToCopy = getSuccessfullyResults(copyResult) { it },
        ).apply {
            withValidationResult(validationResult)
        }
    }

    fun checkCopyCampaigns(
        context: GridGraphQLContext,
        input: GdCheckCopyCampaigns,
    ): GdCheckCopyCampaignsPayload {
        val userFrom = userService.getUserByLogin(input.loginFrom)
        val userTo = userService.getUserByLogin(input.loginTo)

        val validator = campaignCopyValidatorProvider
            .copyCampaignsInterclientBaseValidator<GdCheckCopyCampaigns>(context.operator, userFrom, userTo)
        gridValidationService.applyValidator(validator, input, false)

        val clientFrom = clientService.getClient(userFrom!!.clientId)!!
        val clientTo = clientService.getClient(userTo!!.clientId)!!

        return GdCheckCopyCampaignsPayload(
            currencyChangeTo = if (clientFrom.workCurrency != clientTo.workCurrency) {
                clientTo.workCurrency
            } else {
                null
            },
        )
    }

    fun copyCampaignsInterclient(
        context: GridGraphQLContext,
        input: GdCopyCampaignsInterclient,
    ): GdCopyCampaignsInterclientPayload {
        val userFrom = userService.getUserByLogin(input.loginFrom)
        val userTo = userService.getUserByLogin(input.loginTo)

        val vr = validateObject(input) {
            checkBy(campaignCopyValidatorProvider.copyCampaignsInterclientBaseValidator(context.operator, userFrom, userTo))
            checkBy(campaignCopyValidatorProvider.copyCampaignsInterclientValidator(context.operator))
        }
        gridValidationService.throwGridValidationExceptionIfHasErrors(vr)

        checkNotNull(userFrom)
        checkNotNull(userTo)

        val campaignIds: List<Long> = selectCampaignsForInterclientCopy(userFrom.clientId, input.filter)
        val flags: CopyCampaignFlags = toCoreFlags(input)

        validateObject<GdCopyCampaignsInterclient>(vr) {
            checkBy(campaignCopyValidatorProvider.campaignIdsValidator(campaignIds))
        }
        gridValidationService.throwGridValidationExceptionIfHasErrors(vr)

        val copyResult = copyCampaignService
            .copyCampaignsInterclientScheduled(context.operator.uid, userFrom, userTo, campaignIds, flags)

        val validationResult: GdValidationResult? = campaignValidationService.getValidationResult(
            copyResult,
            path(field(GdCopyCampaignsInterclient::filter), field(GdCopyCampaignsFilter::campaignIds))
        )

        val campaignIdsToCopy: List<Long> = getSuccessfullyResults(copyResult) { it }

        return GdCopyCampaignsInterclientPayload(
            campaignIdsToCopy = campaignIdsToCopy,
            skippedCampaignIds = campaignIds - campaignIdsToCopy.toSet(),
            // TODO синхронное копирование единичных кампаний
            copiedCampaignId = null,
        ).apply {
            withValidationResult(validationResult)
        }
    }

    /**
     * Выбирает id кампаний для копирования. Если фильтр задает список [GdCopyCampaignsFilter.campaignIds],
     * то выбранные id будут возвращены в том же порядке
     */
    @VisibleForTesting
    fun selectCampaignsForInterclientCopy(clientId: ClientId, filter: GdCopyCampaignsFilter): List<Long> {
        val shard = shardHelper.getShardByClientId(clientId)
        val selectionCriteria = CampaignsCopySelectionCriteria(
            clientId = clientId,
            campaignIds = filter.campaignIds,
            selectionType = filter.campaignsType?.let(this::toCoreFilterType),
        )

        val campaignIds: Set<Long> = campaignCopyRepository.getCampaignsForCopy(shard, selectionCriteria)

        return if (filter.campaignIds != null) {
            filter.campaignIds
                .filter { it in campaignIds }
        } else {
            campaignIds.toList()
        }
    }

    private fun toCoreFilterType(filterType: GdCopyCampaignsSelectorType): SelectionType {
        return when (filterType) {
            GdCopyCampaignsSelectorType.ACTIVE -> SelectionType.ACTIVE
            GdCopyCampaignsSelectorType.STOPPED -> SelectionType.STOPPED
            GdCopyCampaignsSelectorType.DRAFT -> SelectionType.DRAFT
            GdCopyCampaignsSelectorType.ARCHIVED -> SelectionType.ARCHIVED
            GdCopyCampaignsSelectorType.ALL -> SelectionType.ALL
        }
    }

    private fun toCoreFlags(input: GdCopyCampaignsInterclient) = CopyCampaignFlags(
        isCopyStopped = input.flags.copyStoppedAds,
        isCopyArchived = input.flags.copyArchivedAds,
        isCopyArchivedCampaigns = input.flags.copyStatuses
            || input.filter.campaignsType == GdCopyCampaignsSelectorType.ALL
            || input.filter.campaignsType == GdCopyCampaignsSelectorType.ARCHIVED,
        isCopyCampaignStatuses = input.flags.copyStatuses,
        isCopyBannerStatuses = input.flags.copyStatuses,
        isCopyKeywordStatuses = input.flags.copyShowConditionStatuses,
        isCopyConversionStrategies = input.flags.copyConversionStrategy,
        // TODO ... = flags.dontCopyLibraryMinusKeywords,
        isStopCopiedCampaigns = input.flags.stopCopiedCampaigns,
        isGenerateReport = input.flags.generateReport,
    )
}
