package ru.yandex.direct.grid.processing.service.campaign.copy

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.flagIsNotAllowed
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.mustBeClient
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.mustBeSameSubClient
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.selectorNotAllowed
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.entity.user.service.validation.UserDefects.userNotFound
import ru.yandex.direct.core.validation.defects.RightsDefects.noRights
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaigns
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsFilter
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsFlags
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsInterclient
import ru.yandex.direct.grid.processing.model.campaign.GdCopyCampaignsInterclientBase
import ru.yandex.direct.rbac.PpcRbac
import ru.yandex.direct.rbac.RbacRole
import ru.yandex.direct.rbac.model.RbacAccessType
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateModel
import ru.yandex.direct.validation.util.validateObject

@Component
class CampaignCopyValidatorProvider(
    private val ppcRbac: PpcRbac,
) {
    companion object {
        val ROLES_ALLOWED_TO_COPY = setOf(
            RbacRole.SUPER, RbacRole.MANAGER, RbacRole.SUPPORT,
            RbacRole.PLACER, RbacRole.AGENCY, RbacRole.LIMITED_SUPPORT,
        )

        @JvmField
        val ROLES_ALLOWED_TO_COPY_BY_SELECTOR =
            ROLES_ALLOWED_TO_COPY - RbacRole.PLACER

        @JvmField
        val ROLES_ALLOWED_TO_COPY_WITH_REPORT =
            setOf(RbacRole.SUPER, RbacRole.SUPPORT, RbacRole.MANAGER)

        @JvmField
        val ROLES_ALLOWED_TO_STOP_COPIED_CAMPAIGNS =
            setOf(RbacRole.SUPER, RbacRole.SUPPORT, RbacRole.MANAGER, RbacRole.LIMITED_SUPPORT)

        @JvmField
        val ROLES_ALLOWED_TO_COPY_CONVERSION_STRATEGY =
            setOf(RbacRole.SUPER, RbacRole.SUPPORT, RbacRole.MANAGER, RbacRole.LIMITED_SUPPORT)
    }

    fun copyCampaignsValidator(): Validator<GdCopyCampaigns, Defect<*>> {
        return Validator { input ->
            validateObject(input) {
                listProperty(GdCopyCampaigns::campaignIds) {
                    check(notEmptyCollection())
                    checkEach(unique())
                    checkEach(validId())
                }
            }
        }
    }

    private fun <T> hasRightsToCopy(operator: User): Constraint<T, Defect<*>> =
        fromPredicate({ operator.role in ROLES_ALLOWED_TO_COPY }, noRights())

    private fun userExists(user: User?): Constraint<String, Defect<*>> =
        fromPredicate({ user != null }, userNotFound())

    private fun userIsClient(user: User?): Constraint<String, Defect<*>> =
        fromPredicate({ user?.role == RbacRole.CLIENT }, mustBeClient())

    private fun sameSubClient(userFrom: User?, userTo: User?): Constraint<String, Defect<*>> =
        fromPredicate({ userFrom?.clientId == userTo?.clientId }, mustBeSameSubClient())

    private fun copyIsAllowed(accessType: RbacAccessType?): Constraint<String, Defect<*>> =
        fromPredicate({ accessType?.isOwner ?: false }, noRights())

    private fun loginValidator(operator: User, user: User?, otherUser: User?, accessType: RbacAccessType?) =
        Validator<String, Defect<*>> {
            validateObject(it) {
                check(userExists(user))
                check(userIsClient(user), When.isValid())

                check(copyIsAllowed(accessType), When.isValid())

                check(
                    sameSubClient(user, otherUser),
                    When.isValidAnd(When.isTrue(operator.role == RbacRole.AGENCY))
                )
            }
        }

    fun <T : GdCopyCampaignsInterclientBase> copyCampaignsInterclientBaseValidator(
        operator: User,
        userFrom: User?,
        userTo: User?,
    ): Validator<T, Defect<*>> {
        val userIds = listOf(userFrom, userTo).mapNotNull { it?.uid }
        val accessTypes = ppcRbac.getAccessTypes(operator.uid, userIds)

        return Validator { input ->
            validateObject(input) {
                check(hasRightsToCopy(operator))

                property(GdCopyCampaignsInterclientBase::loginFrom) {
                    checkBy(loginValidator(operator, userFrom, userTo, accessTypes[userFrom?.uid]))
                }

                property(GdCopyCampaignsInterclientBase::loginTo) {
                    checkBy(loginValidator(operator, userTo, userFrom, accessTypes[userTo?.uid]))
                }
            }
        }
    }

    private fun eitherCampaignIdsOrCampaignsTypeShouldBeSet(): Constraint<GdCopyCampaignsFilter, Defect<*>> =
        fromPredicate({ filter -> (filter.campaignIds == null) != (filter.campaignsType == null) }, invalidValue())

    private fun canUseSelector(operatorRole: RbacRole): Constraint<GdCopyCampaignsFilter, Defect<*>> =
        fromPredicate({ operatorRole in ROLES_ALLOWED_TO_COPY_BY_SELECTOR }, selectorNotAllowed())

    private fun flagIsAllowed(operatorRole: RbacRole, allowedRoles: Set<RbacRole>): Constraint<Boolean, Defect<*>> =
        fromPredicate({ value -> value == false || operatorRole in allowedRoles }, flagIsNotAllowed())

    fun copyCampaignsInterclientValidator(operator: User): Validator<GdCopyCampaignsInterclient, Defect<*>> {
        return Validator { input ->
            validateObject(input) {
                property(GdCopyCampaignsInterclient::filter) {
                    check(eitherCampaignIdsOrCampaignsTypeShouldBeSet())

                    property(GdCopyCampaignsFilter::campaignsType) {
                        check(canUseSelector(operator.role))
                    }

                    listProperty(GdCopyCampaignsFilter::campaignIds) {
                        checkEach(CommonConstraints.notNull())
                    }
                }

                property(GdCopyCampaignsInterclient::flags) {
                    property(GdCopyCampaignsFlags::copyConversionStrategy) {
                        check(flagIsAllowed(operator.role, ROLES_ALLOWED_TO_COPY_CONVERSION_STRATEGY))
                    }
                    property(GdCopyCampaignsFlags::stopCopiedCampaigns) {
                        check(flagIsAllowed(operator.role, ROLES_ALLOWED_TO_STOP_COPIED_CAMPAIGNS))
                    }
                    property(GdCopyCampaignsFlags::generateReport) {
                        check(flagIsAllowed(operator.role, ROLES_ALLOWED_TO_COPY_WITH_REPORT))
                    }
                }
            }
        }
    }

    fun campaignIdsValidator(selectedCampaignIds: Collection<Long>): Validator<GdCopyCampaignsInterclient, Defect<*>> {
        return Validator { input ->
            validateObject(input) {
                property(GdCopyCampaignsInterclient::filter) {
                    listProperty(GdCopyCampaignsFilter::campaignIds) {
                        checkEach(inSet(selectedCampaignIds.toSet()), objectNotFound())
                    }
                }
            }
        }
    }
}
