package ru.yandex.direct.grid.processing.service.campaign.loader;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class BannerGeoLegalFlagsInAggrStatusDataLoader extends GridBatchingDataLoader<Long, Set<String>> {
    private static final Logger logger = LoggerFactory.getLogger(BannerGeoLegalFlagsInAggrStatusDataLoader.class);

    public BannerGeoLegalFlagsInAggrStatusDataLoader(GridContextProvider gridContextProvider,
                                                     ShardHelper shardHelper,
                                                     BannerModerationRepository bannerModerationRepository) {
        this.dataLoader = mappedDataLoader(gridContextProvider,
                getBatchLoadFunction(bannerModerationRepository, shardHelper,
                        () -> retrieveClientId(gridContextProvider)));
    }

    private MappedBatchLoaderWithContext<Long, Set<String>> getBatchLoadFunction(
            BannerModerationRepository bannerModerationRepository,
            ShardHelper shardHelper,
            Supplier<ClientId> clientIdSupplier) {
        return (cids, environment) -> {
            if (CollectionUtils.isEmpty(cids)) {
                return CompletableFuture.completedFuture(new HashMap<>());
            }
            ClientId clientId = clientIdSupplier.get();
            if (clientId == null) {
                logger.error("clientId cannot be null");
                return CompletableFuture.completedFuture(new HashMap<>());
            }
            int shard = shardHelper.getShardByClientId(clientId);
            Map<Long, Set<String>> result = bannerModerationRepository.getBannerGeoLegalFlagsByCids(shard, cids);

            return CompletableFuture.completedFuture(result);
        };
    }

    //Дальше идёт костыль, так как clientId  получают разными способами.
    // Других пока нет, так что текущий костыль должен быть исчерпывающим
    private static ClientId retrieveClientId(GridContextProvider gridContextProvider) {
        ClientId clientId = ClientId.fromNullableLong(
                ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                        GridGraphQLContext::getQueriedClient),
                        GdClientInfo::getId));
        if (clientId == null) {
            clientId = ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                    GridGraphQLContext::getSubjectUser),
                    User::getClientId);
        }
        return clientId;
    }
}
