package ru.yandex.direct.grid.processing.service.campaign.loader;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.banner.container.AdsCountCriteria;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CampaignAdsCountDataLoader extends GridBatchingDataLoader<Long, Integer> {

    public CampaignAdsCountDataLoader(GridContextProvider gridContextProvider, BannerService bannerService) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(bannerService));
    }

    private MappedBatchLoaderWithContext<Long, Integer> getBatchLoadFunction(BannerService bannerService) {
        return (campaignsIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");

            Map<Long, Integer> bannerQuantitiesByAdGroupIds =
                    bannerService.getBannersCounterByCampaigns(ClientId.fromLong(queriedClient.getId()), campaignsIds,
                            new AdsCountCriteria().withExcludedTypes(BannersBannerType.performance_main));

            Map<Long, Integer> result = listToMap(campaignsIds, Function.identity(),
                    campaignId -> bannerQuantitiesByAdGroupIds.getOrDefault(campaignId, 0));
            return CompletableFuture.completedFuture(result);
        };
    }
}
