package ru.yandex.direct.grid.processing.service.campaign.loader;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.campaign.service.CampaignBudgetReachService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.campaign.repository.GridCampaignRepository;
import ru.yandex.direct.grid.model.campaign.GdBrandSurveyStatus;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CampaignBrandSurveyStatusDataLoader extends GridBatchingDataLoader<GetBrandSurveyStatusKey, GdBrandSurveyStatus> {
    private static final Logger logger = LoggerFactory.getLogger(CampaignBrandSurveyStatusDataLoader.class);

    public CampaignBrandSurveyStatusDataLoader(
            GridContextProvider gridContextProvider,
            ShardHelper shardHelper,
            CampaignBudgetReachService campaignBudgetReachService,
            CampaignInfoService campaignInfoService,
            FeatureService featureService
    ) {
        var batchLoadFunction = getBatchLoadFunction(
                campaignBudgetReachService, shardHelper, campaignInfoService, featureService,
                () -> retrieveClientId(gridContextProvider));
        this.dataLoader = mappedDataLoader(gridContextProvider, batchLoadFunction);
    }

    private MappedBatchLoaderWithContext<GetBrandSurveyStatusKey, GdBrandSurveyStatus> getBatchLoadFunction(
            CampaignBudgetReachService campaignBudgetReachService,
            ShardHelper shardHelper,
            CampaignInfoService campaignInfoService,
            FeatureService featureService,
            Supplier<ClientId> clientIdSupplier) {
        return (brandSurveyStatusKeys, environment) -> {
            if (CollectionUtils.isEmpty(brandSurveyStatusKeys)) {
                return CompletableFuture.completedFuture(Map.of());
            }
            GridGraphQLContext context = environment.getContext();
            ClientId clientId = clientIdSupplier.get();
            if (clientId == null) {
                logger.error("clientId cannot be null");
                return CompletableFuture.completedFuture(new HashMap<>());
            }
            int shard = shardHelper.getShardByClientId(clientId);

            boolean isBrandLiftHiddenEnabled = featureService.isEnabled(context.getOperator().getUid(),
                    FeatureName.BRAND_LIFT_HIDDEN);

            Map<Long, GdiCampaign> allCampaigns = campaignInfoService.getAllCampaignsMap(clientId);

            var campaignIdToBrandSurveyId = StreamEx.of(brandSurveyStatusKeys)
                    .mapToEntry(GetBrandSurveyStatusKey::getCid, GetBrandSurveyStatusKey::getBrandSurveyId)
                    .filterKeys(cid -> isBrandLiftHiddenEnabled ||
                            allCampaigns.containsKey(cid) && !allCampaigns.get(cid).getIsBrandLiftHidden())
                    .toMap();

            var campaignIdToBrandSurveyStatus = campaignBudgetReachService.getBrandStatusForCampaigns(
                    shard, clientId, campaignIdToBrandSurveyId);
            Map<GetBrandSurveyStatusKey, GdBrandSurveyStatus> result = StreamEx.of(brandSurveyStatusKeys)
                    .mapToEntry(t -> campaignIdToBrandSurveyStatus.get(t.getCid()))
                    .nonNullValues()
                    .mapValues(GridCampaignRepository::convertBrandSurveyStatus)
                    .toMap();
            return CompletableFuture.completedFuture(result);
        };
    }

    //Дальше идёт костыль, так как clientId  получают разными способами.
    // Других пока нет, так что текущий костыль должен быть исчерпывающим
    private static ClientId retrieveClientId(GridContextProvider gridContextProvider) {
        ClientId clientId = ClientId.fromNullableLong(
                ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                        GridGraphQLContext::getQueriedClient),
                        GdClientInfo::getId));
        if (clientId == null) {
            clientId = ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                    GridGraphQLContext::getSubjectUser),
                    User::getClientId);
        }
        return clientId;
    }
}
