package ru.yandex.direct.grid.processing.service.campaign.loader;

import java.net.IDN;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.dataloader.BatchLoaderEnvironment;
import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CampaignDomainsDataLoader extends GridBatchingDataLoader<Long, List<String>> {

    private final BannerCommonRepository bannerCommonRepository;
    private final BannersUrlHelper bannersUrlHelper;
    private final BannerService bannerService;
    private final TurboLandingRepository turboLandingRepository;

    public CampaignDomainsDataLoader(
            GridContextProvider gridContextProvider,
            BannerCommonRepository bannerCommonRepository,
            BannersUrlHelper bannersUrlHelper,
            BannerService bannerService,
            TurboLandingRepository turboLandingRepository
    ) {
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannersUrlHelper = bannersUrlHelper;
        this.bannerService = bannerService;
        this.turboLandingRepository = turboLandingRepository;

        var batchLoadFunction = getBatchLoadFunction();
        this.dataLoader = mappedDataLoader(gridContextProvider, batchLoadFunction);
    }

    private MappedBatchLoaderWithContext<Long, List<String>> getBatchLoadFunction() {
        return (campaignIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            Long operatorUid = context.getOperator().getUid();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");
            ClientId clientId = ClientId.fromLong(queriedClient.getId());
            int shard = queriedClient.getShard();

            var campaignHrefByCampaignId = extractCampaignHref(environment);
            var bannerHrefsByCampaignId = bannerCommonRepository.getHrefsByCampaignId(shard, campaignIds);
            var turboHrefByCampaignId = getTurboHrefByCampaignId(shard, clientId, operatorUid, campaignIds);

            Map<Long, List<String>> domainsByCampaignId =
                    StreamEx.of(campaignIds)
                            .distinct()
                            .toMap(
                                    Function.identity(),
                                    c -> StreamEx.of(bannerHrefsByCampaignId.getOrDefault(c, emptySet()))
                                            .append(Optional.ofNullable(campaignHrefByCampaignId.get(c)).stream())
                                            .append(turboHrefByCampaignId.getOrDefault(c, emptyList()))
                                            .map(h -> extractDomainFromHref(bannersUrlHelper, h))
                                            .remove(Objects::isNull)
                                            .map(String::toLowerCase)
                                            .distinct()
                                            .sorted()
                                            .toList()
                            );

            return CompletableFuture.completedFuture(domainsByCampaignId);
        };
    }

    /**
     * Для переданных кампаний возвращает список URL'ов турболендингов, которые привязаны к баннерам в этих кампаниях
     */
    private Map<Long, List<String>> getTurboHrefByCampaignId(
            int shard,
            ClientId clientId,
            Long operatorUid,
            Set<Long> campaignIds
    ) {
        var banners = bannerService.getBannersByCampaignIds(shard, operatorUid, clientId, campaignIds, null);
        Map<Long, List<Long>> campaignIdByTurbolandingId = StreamEx.of(banners)
                .select(BannerWithTurboLanding.class)
                .mapToEntry(BannerWithTurboLanding::getTurboLandingId, BannerWithTurboLanding::getCampaignId)
                .filterKeys(Objects::nonNull)
                .grouping();

        var turbolandingIds = campaignIdByTurbolandingId.keySet();
        List<TurboLanding> turbolandings =
                turboLandingRepository.getClientTurboLandingsbyId(shard, clientId, turbolandingIds);
        //noinspection UnnecessaryLocalVariable
        var turboHrefByCampaignId = StreamEx.of(turbolandings)
                .mapToEntry(t -> campaignIdByTurbolandingId.get(t.getId()), TurboLanding::getTurboSiteHref)
                .flatMapKeys(Collection::stream)
                .nonNullValues()
                .grouping();
        return turboHrefByCampaignId;
    }

    @Nullable
    private String extractDomainFromHref(BannersUrlHelper bannersUrlHelper, @Nullable String href) {
        if (href == null) {
            return null;
        }
        String domain = bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(href);
        if (domain == null) {
            return null;
        }
        return IDN.toUnicode(domain);
    }

    private Map<Long, String> extractCampaignHref(BatchLoaderEnvironment environment) {
        return EntryStream.of(environment.getKeyContexts())
                .mapKeys(campaignId -> (Long) campaignId)
                .mapValues(campaignHref -> (String) campaignHref)
                .toMap();
    }

}
