package ru.yandex.direct.grid.processing.service.campaign.loader;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.Period;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.dataloader.BatchLoaderEnvironment;
import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.autobudget.service.CpaAutobudgetPessimizedUsersService;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdPayForConversionInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.strategyDataFromDb;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.BOUNDARY_NUMBER_OF_CONVERSIONS_PER_PERIOD;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.NUM_OF_DAYS_FOR_CONVERSION_LOOK_UP;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.hasLackOfFundsOnCampaignWithPayForConversion;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.PAY_FOR_CONVERSION_AVG_CPA_WARNING_RATIO_DEFAULT_VALUE;
import static ru.yandex.direct.grid.processing.service.autooverdraft.converter.AutoOverdraftDataConverter.toClientAutoOverdraftInfo;
import static ru.yandex.direct.grid.processing.service.campaign.GridCampaignAggregationFieldsService.calcSumTotal;
import static ru.yandex.direct.grid.processing.service.campaign.converter.CommonCampaignConverter.toGdPayForConversionInfo;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CampaignPayForConversionInfoDataLoader extends GridBatchingDataLoader<Long, GdPayForConversionInfo> {

    private final GridCampaignService gridCampaignService;
    private final CampaignInfoService campaignInfoService;
    private final GdStrategyExtractorFacade strategyExtractorFacade;
    private final CpaAutobudgetPessimizedUsersService pessimizedLoginsService;
    private final ClientService clientService;

    /**
     * Вычисление дополнительной информации для стратегии с Оплатой за конверсии.
     * Вызывать для кампаний с параметром стратегии pay_for_conversion = 1
     */
    public CampaignPayForConversionInfoDataLoader(GridContextProvider gridContextProvider,
                                                  GridCampaignService gridCampaignService,
                                                  CampaignInfoService campaignInfoService,
                                                  GdStrategyExtractorFacade strategyExtractorFacade,
                                                  CpaAutobudgetPessimizedUsersService pessimizedLoginsService,
                                                  ClientService clientService) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction());
        this.gridCampaignService = gridCampaignService;
        this.campaignInfoService = campaignInfoService;
        this.strategyExtractorFacade = strategyExtractorFacade;
        this.pessimizedLoginsService = pessimizedLoginsService;
        this.clientService = clientService;
    }

    private static Map<Long, GdPayForConversionInfo> convertToGdPayForConversionInfo(Collection<Long> campaignIds,
                                                                                     Map<Long, Boolean> lackOfConversionsByCampaignId,
                                                                                     Map<Long, Boolean> lackOfFundsByCampaignId) {
        return listToMap(campaignIds, Function.identity(), cid -> toGdPayForConversionInfo(
                lackOfConversionsByCampaignId.get(cid),
                lackOfFundsByCampaignId.get(cid)));
    }

    @Nullable
    private static Boolean hasStrategyGoalLackOfConversions(@Nullable Long goalId, List<GdiGoalStats> goalStatsList) {
        if (goalId == null) {
            return null;
        }

        Optional<GdiGoalStats> goalStats = StreamEx.of(goalStatsList)
                .findAny(goal -> goal.getGoalId().equals(goalId));

        return goalStats
                .map(gdiGoalStats -> gdiGoalStats.getGoals() < BOUNDARY_NUMBER_OF_CONVERSIONS_PER_PERIOD)
                .orElse(true);
    }

    private Map<Long, Long> extractFromEnvironmentGoalsByCampaignId(BatchLoaderEnvironment environment) {
        return EntryStream.of(environment.getKeyContexts())
                .mapKeys(cid -> (Long) cid)
                .mapValues(context -> (GdCampaignFlatStrategy) context)
                .mapValues(strategyExtractorFacade::extractStrategyGoalIdAndVerifyNonNull)
                .toMap();
    }

    private MappedBatchLoaderWithContext<Long, GdPayForConversionInfo> getBatchLoadFunction() {
        return (campaignsIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");
            Currency currency = clientService.getWorkCurrency(ClientId.fromLong(queriedClient.getId()));

            Map<Long, Long> strategyGoalIdByCampaignId = extractFromEnvironmentGoalsByCampaignId(environment);

            //проверяем необходимость показывать предупреждение о резерве средств для оплаты будущих конверсий
            Map<Long, Boolean> lackOfFundsByCampaignId = getLackOfFundsByCampaignId(context, campaignsIds, currency);

            //проверяем необходимость показывать предупреждение о недостатке конверсий
            Map<Long, Boolean> lackOfConversionsByCampaignId = getLackOfConversionsByCampaignId(queriedClient,
                    campaignsIds, strategyGoalIdByCampaignId);

            return CompletableFuture.completedFuture(convertToGdPayForConversionInfo(campaignsIds,
                    lackOfConversionsByCampaignId, lackOfFundsByCampaignId));
        };
    }

    /**
     * Рассчет необходимости отобржаения предупреждения о недостатке средств для оплаты будущих конверсий
     */
    private Map<Long, Boolean> getLackOfFundsByCampaignId(GridGraphQLContext context, Collection<Long> campaignIds,
                                                          Currency currency) {

        GdClientInfo clientInfo = context.getQueriedClient();
        List<GdiBaseCampaign> allCampaigns = campaignInfoService
                .getAllBaseCampaigns(ClientId.fromLong(clientInfo.getId()));

        Map<Long, GdiBaseCampaign> campaignsById = StreamEx.of(allCampaigns)
                .filter(campaign -> campaignIds.contains(campaign.getId()))
                .toMap(GdiBaseCampaign::getId, Function.identity());

        //достать текущий остаток на общем счете
        var walletsMap = campaignInfoService.extractWalletsMap(context.getOperator(), clientInfo,
                toClientAutoOverdraftInfo(clientInfo), allCampaigns, context.getInstant());

        //вычисляем остаток на кампаниях
        var campaignsSumTotalByCampaignId = EntryStream.of(campaignsById)
                .mapValues(campaign -> calcSumTotal(campaign.getSum(), campaign.getSumSpent(),
                        ifNotNull(campaign.getWalletId(), walletsMap::get)))
                .toMap();

        return EntryStream.of(campaignsById)
                .mapValues(campaign -> strategyDataFromDb(campaign.getStrategyData()))
                .filterValues(CampaignStrategyUtils::isPayForConversionStrategyData)
                .mapToValue((cid, strategyData) -> hasLackOfFundsOnCampaignWithPayForConversion(
                        campaignsSumTotalByCampaignId.get(cid), strategyData,
                        BigDecimal.valueOf(PAY_FOR_CONVERSION_AVG_CPA_WARNING_RATIO_DEFAULT_VALUE),
                        currency.getPayForConversionMinReservedSumDefaultValue()))
                .nonNullValues()
                .toMap();
    }

    /**
     * Рассчет признака показа предупреждения об отсутствии конверсий по цели указанной в стратегии
     */
    private Map<Long, Boolean> getLackOfConversionsByCampaignId(GdClientInfo queriedClient,
                                                                Collection<Long> campaignsIds,
                                                                Map<Long, Long> strategyGoalIdByCampaignId) {

        if (pessimizedLoginsService.isLoginPessimized(ClientId.fromLong(queriedClient.getId()))) {
            //если пользователь пессимизирован в ОКР,
            // то проверяем количество конверсий по цели указанной в стратегии
            LocalDate now = LocalDate.now();
            LocalDate dateFrom = now.minus(Period.ofDays(NUM_OF_DAYS_FOR_CONVERSION_LOOK_UP));
            Map<Long, GdiCampaignStats> stats = gridCampaignService.getCampaignStats(campaignsIds, dateFrom, now,
                    listToSet(strategyGoalIdByCampaignId.values()));

            return EntryStream.of(stats)
                    .mapToValue((cid, campaignStats) -> hasStrategyGoalLackOfConversions(
                            strategyGoalIdByCampaignId.get(cid),
                            campaignStats.getGoalStats()))
                    .nonNullValues()
                    .toMap();
        }
        return listToMap(campaignsIds, Function.identity(), v -> false);
    }
}
