package ru.yandex.direct.grid.processing.service.campaign.loader;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultAdGroup;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CampaignsHasDefaultAdGroupDataLoader extends GridBatchingDataLoader<Long, Boolean> {

    public CampaignsHasDefaultAdGroupDataLoader(GridContextProvider gridContextProvider,
                                                AdGroupRepository adGroupRepository,
                                                CampaignTypedRepository campaignTypedRepository,
                                                PricePackageRepository pricePackageRepository) {
        this.dataLoader = mappedDataLoader(gridContextProvider,
                getBatchLoadFunction(adGroupRepository, campaignTypedRepository,pricePackageRepository));
    }

    private MappedBatchLoaderWithContext<Long, Boolean> getBatchLoadFunction(
            AdGroupRepository adGroupRepository,
            CampaignTypedRepository campaignTypedRepository,
            PricePackageRepository pricePackageRepository) {
        return (campaignIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            Integer shard = context.getQueriedClient().getShard();

            Map<Long, CpmPriceCampaign> cpmPriceCampaigns = listToMap(mapList(
                    campaignTypedRepository.getTypedCampaigns(shard, campaignIds),
                    CpmPriceCampaign.class::cast),
                    CpmPriceCampaign::getId,
                    Function.identity());
            Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(
                    mapList(cpmPriceCampaigns.values(),
                            CpmPriceCampaign::getPricePackageId));

            Map<Long, List<Long>> adGroupIdsByCampaignId = adGroupRepository
                    .getAdGroupIdsByCampaignIds(shard, campaignIds);
            List<Long> allAdGroupIds = flatToList(adGroupIdsByCampaignId.values());
            // идём в новой транзакции. что-то может поменятся, например группа может удалится, но это не критично
            // просто будет в таком случае FALSE, хотя недавно ещё должно было бы быть TRUE, наша цель показать
            // какое-то относительно актуальное состояние
            Map<Long, Long> adGroupsPriority = adGroupRepository.getAdGroupsPriority(shard, allAdGroupIds);
            Map<Long, Boolean> result = listToMap(campaignIds,
                    Function.identity(),
                    campaignId -> {
                            // Если на пакете нет флага needDefaultAdGroup(), то считаем, что дефолтная группа уже есть.
                            PricePackage pricePackage = pricePackages.get(cpmPriceCampaigns.get(campaignId).getPricePackageId());
                            if (!pricePackage.needDefaultAdGroup()) {
                                    return Boolean.TRUE;
                            }
                            // Проверка для пакета на прайсовую морду
                            return Optional.ofNullable(adGroupIdsByCampaignId.get(campaignId))
                                    .map(adGroupIds -> adGroupIds.stream().anyMatch(isDefaultAdGroup(adGroupsPriority)))
                                    // если у кампании нет ни одной группы - то и дефолтной группы нет
                                    .orElse(Boolean.FALSE);
                    });
            return CompletableFuture.completedFuture(result);
        };
    }

}
