package ru.yandex.direct.grid.processing.service.campaign.touch;

import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.touchsocdem.service.converter.GridTouchSocdemConverter;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdTouchCampaignMutationPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdTouchCampaignMutationResult;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdiTouchSocdem;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignConverterContext;
import ru.yandex.direct.grid.processing.service.campaign.converter.TouchCampaignConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.Result;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.bidmodifiers.container.ComplexBidModifierConverter.convertToComplexBidModifier;
import static ru.yandex.direct.grid.processing.service.campaign.converter.TouchCampaignConverter.toComplexTextAdGroup;
import static ru.yandex.direct.grid.processing.service.campaign.converter.TouchCampaignConverter.toCoreCampaign;
import static ru.yandex.direct.grid.processing.service.campaign.converter.TouchCampaignConverter.toCoreCampaignModelChanges;
import static ru.yandex.direct.grid.processing.util.GdValidationResultUtils.concatValidationResults;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@ParametersAreNonnullByDefault
@Service
public class TouchCampaignMutationService {
    private static final Logger logger = LoggerFactory.getLogger(TouchCampaignMutationService.class);

    private final AttributeResolverService attributeResolverService;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final HostingsHandler hostingsHandler;
    private final CampaignOperationService campaignOperationService;
    private final ClientGeoService clientGeoService;
    private final ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory;
    private final ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final ShardHelper shardHelper;
    private final RetargetingService retargetingService;
    private final RetargetingConditionService retargetingConditionService;
    private final FeatureService featureService;
    private final BidModifierService bidModifierService;
    private final TranslationService translationService;
    private final TouchCampaignValidationService touchCampaignValidationService;

    @Autowired
    public TouchCampaignMutationService(
            CampaignOperationService campaignOperationService,
            ClientGeoService clientGeoService,
            ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory,
            ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory,
            AttributeResolverService attributeResolverService,
            SspPlatformsRepository sspPlatformsRepository,
            HostingsHandler hostingsHandler,
            BannerRelationsRepository bannerRelationsRepository, ShardHelper shardHelper,
            RetargetingService retargetingService,
            RetargetingConditionService retargetingConditionService,
            BidModifierService bidModifierService,
            FeatureService featureService,
            TranslationService translationService,
            TouchCampaignValidationService touchCampaignValidationService) {
        this.campaignOperationService = campaignOperationService;
        this.clientGeoService = clientGeoService;
        this.attributeResolverService = attributeResolverService;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.hostingsHandler = hostingsHandler;
        this.complexAdGroupAddOperationFactory = complexAdGroupAddOperationFactory;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.shardHelper = shardHelper;
        this.complexAdGroupUpdateOperationFactory = complexAdGroupUpdateOperationFactory;
        this.retargetingService = retargetingService;
        this.retargetingConditionService = retargetingConditionService;
        this.bidModifierService = bidModifierService;
        this.featureService = featureService;
        this.translationService = translationService;
        this.touchCampaignValidationService = touchCampaignValidationService;
    }

    public GdTouchCampaignMutationPayload addTouchCampaign(GridGraphQLContext context, GdAddTouchCampaignInput input) {
        ClientId clientId = context.getSubjectUser().getClientId();

        touchCampaignValidationService.validateAddRequest(input, clientId);

        return doMutation(
                context,
                ctx -> createCampaign(input, ctx.getOperator(), ctx.getSubjectUser()),
                (ctx, campaignId) -> createGroup(input, ctx.getOperator(), ctx.getSubjectUser(), campaignId)
        );
    }

    public GdTouchCampaignMutationPayload updateTouchCampaign(
            GridGraphQLContext context, GdUpdateTouchCampaignInput input) {
        ClientId clientId = context.getSubjectUser().getClientId();

        touchCampaignValidationService.validateUpdateRequest(input, clientId);

        return doMutation(context,
                ctx -> updateCampaign(input, ctx.getOperator(), ctx.getSubjectUser()),
                (ctx, campaignId) -> updateGroup(input, ctx.getOperator(), ctx.getSubjectUser(), campaignId));
    }

    private GdTouchCampaignMutationPayload doMutation(
            GridGraphQLContext context,
            Function<GridGraphQLContext, Result<Long>> campaignMutator,
            BiFunction<GridGraphQLContext, Long, Result<Long>> groupMutator
    ) {
        Result<Long> campaignResult = campaignMutator.apply(context);

        var campaignValidationResult = GridValidationResultConversionService.buildGridValidationResultIfErrors(
                campaignResult.getValidationResult(), emptyPath(), TouchCampaignConverter.PATH_NODE_CONVERTER_PROVIDER);
        if (campaignValidationResult != null && !campaignValidationResult.getErrors().isEmpty()) {
            return new GdTouchCampaignMutationPayload().withValidationResult(campaignValidationResult);
        }

        Result<Long> groupResult = groupMutator.apply(context, campaignResult.getResult());

        var groupValidationResult = GridValidationResultConversionService.buildGridValidationResultIfErrors(
                groupResult.getValidationResult(), path(field(GdAddTouchCampaignInput.AD_GROUP)),
                TouchCampaignConverter.PATH_NODE_CONVERTER_PROVIDER);

        GdValidationResult validationResult = concatValidationResults(campaignValidationResult, groupValidationResult);
        if (validationResult != null && !validationResult.getErrors().isEmpty()) {
            return new GdTouchCampaignMutationPayload().withValidationResult(validationResult);
        }

        Long groupId = groupResult.getResult();

        int shard = shardHelper.getShardByClientId(context.getSubjectUser().getClientId());
        Collection<Long> bannerIds = bannerRelationsRepository.getAdGroupIdToBannerIds(shard, singletonList(groupId))
                .get(groupId);
        Long bannerId = Iterables.getFirst(bannerIds, null);

        var result = new GdTouchCampaignMutationResult()
                .withCampaignId(campaignResult.getResult())
                .withAdGroupId(groupId)
                .withAdId(bannerId);

        return new GdTouchCampaignMutationPayload().withResult(result).withValidationResult(validationResult);
    }

    private Result<Long> createCampaign(GdAddTouchCampaignInput input, User operator, User client) {
        var campaign = toCoreCampaign(input.getName(), input.getBudget(), client.getEmail(),
                CampaignConverterContext.create(attributeResolverService, sspPlatformsRepository, hostingsHandler));
        campaign.setIsTouch(true);

        CampaignOptions options = new CampaignOptions.Builder()
                .withReadyToModerate(true)
                .build();
        var addCampaignOperation = campaignOperationService.createRestrictedCampaignAddOperation(
                singletonList(campaign), operator.getUid(), UidAndClientId.of(client.getUid(), client.getClientId()),
                options);
        return addCampaignOperation.prepareAndApply().get(0);
    }

    private Result<Long> createGroup(GdAddTouchCampaignInput input, User operator, User client, Long campaignId) {
        ClientId clientId = client.getClientId();
        var autotargetingEnabled =
                featureService.isEnabledForClientId(clientId, FeatureName.AUTOTARGETING_FOR_TOUCH_CAMPAIGNS_ENABLED);
        var complexGroup = toComplexTextAdGroup(clientId, campaignId, input.getAdGroup(), input.getName(),
                autotargetingEnabled, translationService);
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        var addGroupOperation = complexAdGroupAddOperationFactory.createTextAdGroupAddOperation(
                false, singletonList(complexGroup), geoTree, false, null,
                operator.getUid(), clientId, client.getUid(), DatabaseMode.ONLY_MYSQL);
        return addGroupOperation.prepareAndApply().get(0);
    }

    private Result<Long> updateCampaign(GdUpdateTouchCampaignInput input, User operator, User client) {
        var changes = toCoreCampaignModelChanges(input.getId(), input.getName(), input.getBudget(),
                CampaignConverterContext.create(attributeResolverService, sspPlatformsRepository, hostingsHandler));

        var options = new CampaignOptions();
        var updateCampaignOperation = campaignOperationService.createRestrictedCampaignUpdateOperation(
                singletonList(changes), operator.getUid(), UidAndClientId.of(client.getUid(), client.getClientId()), options);
        return updateCampaignOperation.apply().get(0);
    }

    private Result<Long> updateGroup(GdUpdateTouchCampaignInput input, User operator, User client, Long campaignId) {
        ClientId clientId = client.getClientId();
        int shard = shardHelper.getShardByClientId(clientId);

        Long adGroupId = input.getAdGroup().getId();
        var adGroupIds = singletonList(adGroupId);
        List<TargetInterest> interests = retargetingService.getTargetInterestsWithInterestByAdGroupIds(
                adGroupIds, clientId, shard);
        List<RetargetingCondition> retConds = retargetingConditionService.getRetargetingConditions(
                clientId, null, adGroupIds, null,
                singleton(RetargetingConditionsRetargetingConditionsType.interests), LimitOffset.limited(1)
        );
        RetargetingCondition interestRetCond = !retConds.isEmpty() ? retConds.get(0) : null;
        List<BidModifier> adGroupBidModifiers = bidModifierService.getByAdGroupIds(
                shard, Set.of(adGroupId), EnumSet.allOf(BidModifierType.class), Set.of(BidModifierLevel.ADGROUP));
        ComplexBidModifier complexAdGroupBidModifier = convertToComplexBidModifier(adGroupBidModifiers);
        if (input.getAdGroup().getSocdem() != null && complexAdGroupBidModifier != null) {
            GdiTouchSocdem oldSocdem = GridTouchSocdemConverter.socdemBidModifierToTouchSocdem(
                    complexAdGroupBidModifier.getDemographyModifier()
            );
            if (oldSocdem.getIncompleteBidModifier()) {
                logger.warn("Changing incompatible touch socdem bid modifier: {}",
                        complexAdGroupBidModifier.getDemographyModifier());
            }
        }
        var complexGroup = toComplexTextAdGroup(clientId, campaignId, input.getAdGroup(), input.getName(),
                interests, interestRetCond, complexAdGroupBidModifier, translationService);
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        Set<ModelProperty> updatedProperties = new HashSet<>();
        updatedProperties.add(ComplexTextAdGroup.RETARGETING_CONDITION);
        updatedProperties.add(ComplexTextAdGroup.TARGET_INTERESTS);
        updatedProperties.add(ComplexTextAdGroup.COMPLEX_BID_MODIFIER);

        // если пользователь не указал никаких интересов, а автотаргетинг был по какой-то причине выключен,
        // включаем его, чтобы было хотя бы какое-то условие показа.
        // Делаем это только в безвыходной ситуации, т.к. в тачёвом интерфейсе нет галки для явного
        // включения/выключения автотаргетинга.
        if (complexGroup.getTargetInterests() == null) {
            TouchCampaignConverter.addDefaultTouchRelevanceMatch(complexGroup);
            updatedProperties.add(ComplexTextAdGroup.RELEVANCE_MATCHES);
        }

        var updateGroupOperation = complexAdGroupUpdateOperationFactory.createRestrictedTextAdGroupUpdateOperation(
                singletonList(complexGroup), geoTree, false, null,
                operator.getUid(), clientId, client.getUid(), false, updatedProperties);
        return updateGroupOperation.prepareAndApply().get(0);
    }
}
