package ru.yandex.direct.grid.processing.service.campaign.touch;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateTouchCampaignInput;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddTouchAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateTouchAdGroup;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.feature.FeatureName.HYPERLOCAL_GEO_FOR_TOUCH_CAMPAIGNS_ENABLED;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;

@Service
@ParametersAreNonnullByDefault
public class TouchCampaignValidationService {

    private final GridValidationService gridValidationService;
    private final FeatureService featureService;

    @Autowired
    public TouchCampaignValidationService(GridValidationService gridValidationService, FeatureService featureService) {
        this.gridValidationService = gridValidationService;
        this.featureService = featureService;
    }

    void validateAddRequest(GdAddTouchCampaignInput addTouchCampaignInput, ClientId clientId) {
        gridValidationService.applyValidator(
                addRequest -> validateAddTouchCampaign(addRequest, clientId), addTouchCampaignInput, false);
    }

    // Validate add touch adgroup

    private ValidationResult<GdAddTouchCampaignInput, Defect> validateAddTouchCampaign(GdAddTouchCampaignInput gdAddTouchCampaignInput,
                                                                                       ClientId clientId) {
        ModelItemValidationBuilder<GdAddTouchCampaignInput> vb = ModelItemValidationBuilder.of(gdAddTouchCampaignInput);

        vb.item(GdAddTouchCampaignInput.AD_GROUP)
                .checkBy(addTouchAdGroup -> validateTouchAdGroup(addTouchAdGroup, clientId));

        return vb.getResult();
    }

    private ValidationResult<GdAddTouchAdGroup, Defect> validateTouchAdGroup(GdAddTouchAdGroup gdAddTouchAdGroup,
                                                                             ClientId clientId) {
        ModelItemValidationBuilder<GdAddTouchAdGroup> vb = ModelItemValidationBuilder.of(gdAddTouchAdGroup);

        boolean isHyperlocalGeoEnabled = featureService.isEnabledForClientId(clientId, HYPERLOCAL_GEO_FOR_TOUCH_CAMPAIGNS_ENABLED);
        vb.checkBy(addTouchAdGroup -> validateTouchAdGroupGeo(addTouchAdGroup, isHyperlocalGeoEnabled));

        return vb.getResult();
    }

    private ValidationResult<GdAddTouchAdGroup, Defect> validateTouchAdGroupGeo(GdAddTouchAdGroup gdAddTouchAdGroup,
                                                                                boolean isHyperlocalGeoEnabled) {
        ModelItemValidationBuilder<GdAddTouchAdGroup> vb = ModelItemValidationBuilder.of(gdAddTouchAdGroup);

        vb.item(GdAddTouchAdGroup.HYPER_GEO_RETARGETING_ID)
                .check(isNull(), When.isFalse(isHyperlocalGeoEnabled));

        return vb.getResult();
    }

    // Validate update touch adgroup

    public void validateUpdateRequest(GdUpdateTouchCampaignInput updateTouchCampaignInput, ClientId clientId) {
        gridValidationService.applyValidator(
                updateRequest -> validateUpdateTouchCampaign(updateRequest, clientId), updateTouchCampaignInput, false);
    }

    private ValidationResult<GdUpdateTouchCampaignInput, Defect> validateUpdateTouchCampaign(GdUpdateTouchCampaignInput gdUpdateTouchCampaignInput,
                                                                                             ClientId clientId) {
        ModelItemValidationBuilder<GdUpdateTouchCampaignInput> vb =
                ModelItemValidationBuilder.of(gdUpdateTouchCampaignInput);

        vb.item(GdUpdateTouchCampaignInput.AD_GROUP)
                .checkBy(updateTouchAdGroup -> validateTouchAdGroup(updateTouchAdGroup, clientId));

        return vb.getResult();
    }

    private ValidationResult<GdUpdateTouchAdGroup, Defect> validateTouchAdGroup(GdUpdateTouchAdGroup gdUpdateTouchAdGroup,
                                                                                ClientId clientId) {
        ModelItemValidationBuilder<GdUpdateTouchAdGroup> vb = ModelItemValidationBuilder.of(gdUpdateTouchAdGroup);

        boolean isHyperlocalGeoEnabled = featureService.isEnabledForClientId(clientId, HYPERLOCAL_GEO_FOR_TOUCH_CAMPAIGNS_ENABLED);
        vb.checkBy(updateTouchAdGroup -> validateTouchAdGroupGeo(updateTouchAdGroup, isHyperlocalGeoEnabled));

        return vb.getResult();
    }

    private ValidationResult<GdUpdateTouchAdGroup, Defect> validateTouchAdGroupGeo(GdUpdateTouchAdGroup gdUpdateTouchAdGroup,
                                                                                   boolean isHyperlocalGeoEnabled) {
        ModelItemValidationBuilder<GdUpdateTouchAdGroup> vb = ModelItemValidationBuilder.of(gdUpdateTouchAdGroup);

        vb.item(GdUpdateTouchAdGroup.HYPER_GEO_RETARGETING_ID)
                .check(isNull(), When.isFalse(isHyperlocalGeoEnabled));

        return vb.getResult();
    }
}
