package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.net.MalformedURLException;
import java.net.URL;
import java.time.LocalDate;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupService.ComplexAdGroupFetchParams;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.touchsocdem.service.converter.GridTouchSocdemConverter;
import ru.yandex.direct.grid.model.campaign.GdMeaningfulGoal;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.facelift.GdBudgetDisplayFormat;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.processing.model.campaign.GdUcKeyword;
import ru.yandex.direct.grid.processing.model.campaign.GdUcRegion;
import ru.yandex.direct.grid.processing.model.campaign.GdUniversalCampaign;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.forecast.GdDeviceType;
import ru.yandex.direct.grid.processing.service.banner.BannerDataConverter;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.hypergeo.HyperGeoUtils;
import ru.yandex.direct.grid.processing.service.touchsocdem.service.converter.TouchSocdemConverter;
import ru.yandex.direct.grid.processing.util.FaviconUtils;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;

import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdMeaningFulGoals;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdTimeTarget;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.defaultGdTimeTarget;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class GdUcCampaignService {
    private static final Logger logger = LoggerFactory.getLogger(GdUcCampaignService.class);

    private final CampaignInfoService campaignInfoService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final ComplexAdGroupService complexAdGroupService;
    private final TurboLandingService turboLandingService;
    private final ClientGeoService clientGeoService;
    private final HyperGeoService hyperGeoService;
    private final KeywordRepository keywordRepository;

    @Autowired
    public GdUcCampaignService(CampaignInfoService campaignInfoService,
                               CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                               ShardHelper shardHelper,
                               AdGroupRepository adGroupRepository,
                               ComplexAdGroupService complexAdGroupService,
                               TurboLandingService turboLandingService,
                               ClientGeoService clientGeoService,
                               HyperGeoService hyperGeoService,
                               KeywordRepository keywordRepository) {
        this.campaignInfoService = campaignInfoService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.complexAdGroupService = complexAdGroupService;
        this.turboLandingService = turboLandingService;
        this.clientGeoService = clientGeoService;
        this.hyperGeoService = hyperGeoService;
        this.keywordRepository = keywordRepository;
    }

    public GdUniversalCampaign getByCampaignId(Long cid,
                                               Long operatorUid,
                                               UidAndClientId subjectClient,
                                               GdClientInfo gdClientInfo) {
        return toGdModel(getUniversalCampData(cid, operatorUid, subjectClient), gdClientInfo);
    }

    @Nullable
    private UcCampaignDataContainer getUniversalCampData(Long cid, Long operatorUid, UidAndClientId subjectClient) {
        if (!campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, subjectClient.getClientId(), List.of(cid))
                .objectInVisibleCampaign(cid)) {
            return null;
        }

        GdiCampaign gdiCampaign = campaignInfoService
                .getCampaignsAndWalletsAndSaveToContext(subjectClient.getClientId(), Set.of(cid))
                .get(cid);
        if (gdiCampaign == null
                || gdiCampaign.getType() != CampaignType.TEXT
                || !nvl(gdiCampaign.getIsUniversal(), false)) {
            return null;
        }
        UcCampaignDataContainer resultContainer = new UcCampaignDataContainer();
        resultContainer.setGdiCampaign(gdiCampaign);
        resultContainer.setClientId(subjectClient.getClientId());

        int shard = shardHelper.getShardByClientId(subjectClient.getClientId());
        Long adGroupId = adGroupRepository
                .getAdGroupIdsBySelectionCriteria(
                        shard,
                        new AdGroupsSelectionCriteria().withCampaignIds(cid).withAdGroupTypes(AdGroupType.BASE),
                        LimitOffset.maxLimited())
                .stream()
                .min(Comparator.naturalOrder())
                .orElse(null);
        if (adGroupId == null) {
            return resultContainer;
        }

        resultContainer.setHyperGeo(
                hyperGeoService.getHyperGeoByAdGroupId(subjectClient.getClientId(), List.of(adGroupId)).get(adGroupId)
        );

        var fetchParams = ComplexAdGroupFetchParams.fetchAll()
                .withFetchOfferRetargetings(false)
                .withFetchRelevanceMatches(false)
                .withFetchRetargetingConditions(false)
                .withFetchTargetInterests(false)
                .withFetchKeywords(false);
        ComplexTextAdGroup complexTextAdGroup = complexAdGroupService
                .getComplexAdGroups(operatorUid, subjectClient, List.of(adGroupId), fetchParams)
                .stream()
                .findFirst()
                .orElse(null);
        if (complexTextAdGroup == null || complexTextAdGroup.getAdGroup() == null
                || !(complexTextAdGroup.getAdGroup() instanceof TextAdGroup)) { //вряд ли произойдёт
            return resultContainer;
        }
        resultContainer.setTextAdGroup((TextAdGroup) complexTextAdGroup.getAdGroup());

        ComplexBanner textComplexBanner = StreamEx.of(nvl(complexTextAdGroup.getComplexBanners(), List.of()))
                .mapToEntry(ComplexBanner::getBanner)
                .selectValues(TextBanner.class)
                .minByLong(t -> t.getValue().getId())
                .map(Map.Entry::getKey)
                .orElse(null);
        TextBanner textBanner = ifNotNull(textComplexBanner, t -> (TextBanner) t.getBanner());
        resultContainer.setTextBanner(textBanner);
        resultContainer.setSitelinkSet(ifNotNull(textComplexBanner, ComplexBanner::getSitelinkSet));

        resultContainer.setTurbolandingsForSitelinks(findTurbolandingsForSitelinks(resultContainer.getSitelinkSet()));

        boolean archived = nvl(gdiCampaign.getArchived(), false);
        Long pid = complexTextAdGroup.getAdGroup().getId();
        List<Keyword> keywords = archived ?
                keywordRepository.getArchivedKeywordsByAdGroupIds(shard, subjectClient.getClientId(),
                        List.of(cid), List.of(pid)).get(pid) :
                keywordRepository.getKeywordsByAdGroupIds(shard, subjectClient.getClientId(),
                        List.of(complexTextAdGroup.getAdGroup().getId())).get(pid);
        resultContainer.setKeywords(keywords);

        if (complexTextAdGroup.getComplexBidModifier() == null) {
            return resultContainer;
        }
        resultContainer.setBidModifierMobile(complexTextAdGroup.getComplexBidModifier().getMobileModifier());
        resultContainer.setBidModifierDesktop(complexTextAdGroup.getComplexBidModifier().getDesktopModifier());
        resultContainer.setBidModifierTablet(complexTextAdGroup.getComplexBidModifier().getTabletModifier());
        resultContainer.setBidModifierDesktopOnly(complexTextAdGroup.getComplexBidModifier().getDesktopOnlyModifier());
        resultContainer.setBidModifierDemographics(complexTextAdGroup.getComplexBidModifier().getDemographyModifier());

        return resultContainer;
    }

    @Nullable
    private List<TurboLanding> findTurbolandingsForSitelinks(SitelinkSet sitelinkSet) {
        if (sitelinkSet == null || sitelinkSet.getClientId() == null) {
            return null;
        }
        List<Long> turbolandingIds = mapList(sitelinkSet.getSitelinks(), Sitelink::getId);
        ClientId sitelinkClientId = ClientId.fromLong(sitelinkSet.getClientId());
        return turboLandingService.getClientTurboLandingsById(
                sitelinkClientId, turbolandingIds, LimitOffset.maxLimited());
    }

    private GdUniversalCampaign toGdModel(@Nullable UcCampaignDataContainer dataContainer, GdClientInfo gdClientInfo) {
        if (dataContainer == null) {
            return null;
        }

        Map<Long, TurboLanding> turbolandingMap = listToMap(
                nvl(dataContainer.getTurbolandingsForSitelinks(), List.of()),
                TurboLanding::getId
        );

        List<Long> regionsIdsForWeb = ifNotNull(dataContainer.getTextAdGroup(), TextAdGroup::getGeo);
        if (regionsIdsForWeb != null && dataContainer.getClientId() != null) {
            GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(dataContainer.getClientId());
            regionsIdsForWeb = clientGeoService.convertForWeb(regionsIdsForWeb, geoTree);
        }

        StrategyData strategyData = ifNotNull(dataContainer.getGdiCampaign(),
                gdiCampaign -> ifNotNull(gdiCampaign.getStrategyData(), CampaignMappings::strategyDataFromDb));

        List<GdMeaningfulGoal> meaningfulGoals = toGdMeaningFulGoals(
                ifNotNull(dataContainer.getGdiCampaign(),
                        gdiCampaign -> ifNotNull(gdiCampaign, GdiCampaign::getMeaningfulGoals)));
        GdTimeTarget timeTarget = ifNotNullOrDefault(dataContainer.getGdiCampaign(),
                gdiCampaign -> toGdTimeTarget(gdiCampaign.getTimeTarget()).withIdTimeZone(gdiCampaign.getTimezoneId()),
                defaultGdTimeTarget());

        return new GdUniversalCampaign()
                .withGdClientInfo(gdClientInfo)
                .withManagerUserId(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getManagerUserId))
                .withAgencyUserId(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getAgencyUserId))

                .withId(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getId))
                .withAvgCpa(ifNotNull(strategyData, StrategyData::getAvgCpa))
                .withBudget(ifNotNull(strategyData, StrategyData::getSum))
                .withBudgetDisplayFormat(nvl(GdBudgetDisplayFormat.fromSource(
                        ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getBudgetDisplayFormat)),
                        GdBudgetDisplayFormat.WEEKLY))
                .withGoalId(ifNotNull(strategyData, StrategyData::getGoalId))
                .withMeaningfulGoals(meaningfulGoals)
                .withBusinessCategory(nvl(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getBusinessCategory),
                        ""))
                .withMetrikaCounters(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getMetrikaCounters))
                .withName(nvl(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getName), "Имя кампании"))
                .withStartDate(nvl(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getStartDate),
                        LocalDate.EPOCH))
                .withShows(nvl(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getShows), 0L))
                .withHref(nvl(ifNotNull(dataContainer.getGdiCampaign(), GdiCampaign::getHref), ""))
                .withTimeTarget(timeTarget)

                .withAdGroupId(ifNotNull(dataContainer.getTextAdGroup(), TextAdGroup::getId))
                .withGeo(mapList(regionsIdsForWeb, t -> new GdUcRegion().withRegionId(t)))
                .withHyperGeoId(ifNotNull(dataContainer.getTextAdGroup(), TextAdGroup::getHyperGeoId))
                .withHyperGeo(ifNotNull(dataContainer.getHyperGeo(), HyperGeoUtils::convert))

                .withAdId(ifNotNull(dataContainer.getTextBanner(), TextBanner::getId))
                .withAdText(nvl(ifNotNull(dataContainer.getTextBanner(), TextBanner::getBody), ""))
                .withAdTitle(nvl(ifNotNull(dataContainer.getTextBanner(), TextBanner::getTitle), ""))
                .withCreativeId(ifNotNull(dataContainer.getTextBanner(), TextBanner::getCreativeId))
                .withFaviconLink(makeFaviconLink(ifNotNull(dataContainer.getTextBanner(), TextBanner::getHref)))
                .withImageHash(ifNotNull(dataContainer.getTextBanner(), TextBanner::getImageHash))
                .withPermalinkId(ifNotNull(dataContainer.getTextBanner(), TextBanner::getPermalinkId))
                .withPhoneId(ifNotNull(dataContainer.getTextBanner(), TextBanner::getPhoneId))
                .withSitelinks(mapList(ifNotNull(dataContainer.getSitelinkSet(), SitelinkSet::getSitelinks),
                        t -> BannerDataConverter.toGdSitelink(t,
                                ifNotNull(t.getTurboLandingId(), turbolandingMap::get))))
                .withFlags(ifNotNull(
                        ifNotNull(dataContainer.getTextBanner(), TextBanner::getFlags),
                        BannerFlags::getFlags))

                .withKeywords(nvl(mapList(dataContainer.getKeywords(), Keyword::getPhrase), List.of()))
                .withKeywordsDetailed(nvl(mapList(dataContainer.getKeywords(),
                        t -> new GdUcKeyword()
                                .withId(t.getId())
                                .withPid(t.getAdGroupId())
                                .withPhrase(t.getPhrase())),
                        List.of()))

                .withDeviceTypes(extractDeviceTypesFromDeviceBidModifiers(dataContainer))
                .withSocdem(ifNotNull(
                        GridTouchSocdemConverter.socdemBidModifierToTouchSocdem(
                                dataContainer.getBidModifierDemographics()),
                        TouchSocdemConverter::toOuterTouchSocdem));
    }

    public static String makeFaviconLink(@Nullable String href) {
        String domain = null;
        if (href != null) {
            try {
                domain = new URL(href).getHost();
            } catch (MalformedURLException e) {
                logger.warn("invalid href " + href);
            }
        }
        String preparedDomain = StringUtils.isBlank(domain) ? "jdjdjdjdj.ru" : domain;
        return FaviconUtils.getFaviconLink(preparedDomain);
    }

    static Set<GdDeviceType> extractDeviceTypesFromDeviceBidModifiers(UcCampaignDataContainer dataContainer) {
        boolean targetDesktop = true;
        boolean targetMobile = true;
        boolean targetTablet = true;

        var bidModifierDesktop = dataContainer.getBidModifierDesktop();
        if (bidModifierDesktop != null
                && nvl(bidModifierDesktop.getEnabled(), true)
                && bidModifierDesktop.getDesktopAdjustment() != null
                && bidModifierDesktop.getDesktopAdjustment().getPercent() != null
                && bidModifierDesktop.getDesktopAdjustment().getPercent().equals(0)) {
            targetDesktop = false;
            targetTablet = false;
        }

        var bidModifierMobile = dataContainer.getBidModifierMobile();
        if (bidModifierMobile != null
                && nvl(bidModifierMobile.getEnabled(), true)
                && bidModifierMobile.getMobileAdjustment() != null
                && bidModifierMobile.getMobileAdjustment().getPercent() != null
                && bidModifierMobile.getMobileAdjustment().getPercent().equals(0)) {
            targetMobile = false;
        }

        var bidModifierDesktopOnly = dataContainer.getBidModifierDesktopOnly();
        if (bidModifierDesktopOnly != null
                && nvl(bidModifierDesktopOnly.getEnabled(), true)
                && bidModifierDesktopOnly.getDesktopOnlyAdjustment() != null
                && bidModifierDesktopOnly.getDesktopOnlyAdjustment().getPercent() != null
                && bidModifierDesktopOnly.getDesktopOnlyAdjustment().getPercent().equals(0)) {
            targetDesktop = false;
        }

        var bidModifierTablet = dataContainer.getBidModifierTablet();
        if (bidModifierTablet != null
                && nvl(bidModifierTablet.getEnabled(), true)
                && bidModifierTablet.getTabletAdjustment() != null
                && bidModifierTablet.getTabletAdjustment().getPercent() != null
                && bidModifierTablet.getTabletAdjustment().getPercent().equals(0)) {
            targetTablet = false;
        }

        if (targetDesktop && targetMobile && targetTablet) {
            return Set.of(GdDeviceType.ALL);
        }

        Set<GdDeviceType> result = new HashSet<>();

        if (targetDesktop) {
            result.add(GdDeviceType.DESKTOP);
        }
        if (targetMobile) {
            result.add(GdDeviceType.PHONE);
        }
        if (targetTablet) {
            result.add(GdDeviceType.TABLET);
        }

        if (!targetDesktop && !targetMobile && !targetTablet) {
            logger.error("upyachka v korrektirovkah: all devices switched off");
            return null;
        }

        return result;
    }
}
