package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.canvas.client.model.video.UcCreative;
import ru.yandex.direct.canvas.client.model.video.VideoThumbnail;
import ru.yandex.direct.core.entity.creative.container.CreativeFilterContainer;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.creative.GdUcSchemeCreativeData;
import ru.yandex.direct.grid.processing.model.creative.GdUcSchemeCreativeMeta;
import ru.yandex.direct.grid.processing.model.creative.GdUcSchemeCreativeMetaFormat;
import ru.yandex.direct.grid.processing.model.creative.GdUcSchemeCreativeThumb;
import ru.yandex.direct.grid.processing.model.creative.GdUcSchemeCreativeThumbPreview;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class GdUcCreativeService {
    private static final Logger logger = LoggerFactory.getLogger(GdUcCreativeService.class);

    private final CreativeRepository creativeRepository;
    private final CreativeService creativeService;

    @Autowired
    public GdUcCreativeService(CreativeRepository creativeRepository,
                               CreativeService creativeService) {
        this.creativeRepository = creativeRepository;
        this.creativeService = creativeService;
    }


    @Nullable
    public GdUcSchemeCreativeData getUcCreativeData(int shard, ClientId clientId, Long creativeId, Long adId) {
        Creative coreSchemeCreative = creativeRepository
                .getCreatives(shard, clientId,
                        new CreativeFilterContainer().withCreativeIds(Set.of(creativeId)))
                .stream()
                .filter(t -> creativeId.equals(t.getId()))
                .findFirst()
                .orElse(null);
        if (coreSchemeCreative == null) {
            logger.error("no core creative for valid creative id in uc campaign " + creativeId);
            return null;
        }
        UcCreative ucSchemeCreative = creativeService
                .getUcCreatives(clientId, List.of(creativeId))
                .stream()
                .filter(t -> creativeId.equals(t.getId()))
                .findFirst()
                .orElse(null);
        if (ucSchemeCreative == null) {
            logger.error("no ext creative for valid creative id in uc campaign " + creativeId);
            return null;
        }

        List<GdUcSchemeCreativeMetaFormat> metaFormats = mapList(
                ifNotNull(ucSchemeCreative.getMeta(), UcCreative.UcCreativeMeta::getFormats), t ->
                        new GdUcSchemeCreativeMetaFormat()
                                .withUrl(t.getUrl())
                                .withType(t.getType())
                                .withBitrate(t.getBitrate())
        );
        VideoThumbnail ucThumb = ifNotNull(ucSchemeCreative.getMeta(), UcCreative.UcCreativeMeta::getThumb);
        GdUcSchemeCreativeThumb thumb = ifNotNull(ucThumb, t ->
                new GdUcSchemeCreativeThumb()
                        .withHeight(t.getHeight())
                        .withWidth(t.getWidth())
                        .withUrl(t.getUrl())
                        .withPreview(ifNotNull(t.getPreview(), s ->
                                        new GdUcSchemeCreativeThumbPreview()
                                                .withHeight(s.getHeight())
                                                .withWidth(s.getWidth())
                                                .withUrl(s.getUrl())
                                )
                        )
        );
        GdUcSchemeCreativeMeta meta = new GdUcSchemeCreativeMeta()
                .withFormats(metaFormats)
                .withMimeType(ifNotNull(ucSchemeCreative.getMeta(), UcCreative.UcCreativeMeta::getMimeType))
                .withStatus(ifNotNull(ucSchemeCreative.getMeta(), UcCreative.UcCreativeMeta::getStatus))
                .withThumb(thumb);
        return new GdUcSchemeCreativeData()
                .withAdId(adId)
                .withCreativeId(creativeId)
                .withHeight(ucSchemeCreative.getHeight())
                .withWidth(ucSchemeCreative.getWidth())
                .withDuration(ucSchemeCreative.getDuration())
                .withLivePreviewUrl(coreSchemeCreative.getLivePreviewUrl())
                .withFilename("file_name")
                .withSourceUrl(ucSchemeCreative.getSourceUrl())
                .withThumbUrl(ucSchemeCreative.getThumbUrl())
                .withThumbId(extractThumbIdFromThumburl(
                        creativeId, ucSchemeCreative.getThumbUrl()))
                .withNamespace(extractNamespaceFromThumburl(
                        creativeId, ucSchemeCreative.getThumbUrl()))
                .withMeta(meta);
    }

    private static String extractThumbIdFromThumburl(Long creativeId, @Nullable String thumbUrl) {
        if (thumbUrl == null) {
            return null;
        }
        int hostStartIndex = thumbUrl.indexOf("//");
        String[] splittedUrl = hostStartIndex != -1 ?
                thumbUrl.substring(hostStartIndex + 2).split("/") : thumbUrl.split("/");
        if (splittedUrl.length < 4) {
            logger.error("invalid url for creative id + " + creativeId + " with thumb url " + thumbUrl);
            return null;
        }
        return String.format("%s/%s", splittedUrl[2], splittedUrl[3]);
    }

    private static String extractNamespaceFromThumburl(Long creativeId, @Nullable String thumbUrl) {
        if (thumbUrl == null) {
            return null;
        }
        int hostStartIndex = thumbUrl.indexOf("//");
        String[] splittedUrl = hostStartIndex != -1 ?
                thumbUrl.substring(hostStartIndex + 2).split("/") : thumbUrl.split("/");
        if (splittedUrl.length < 4) {
            logger.error("invalid url for creative id + " + creativeId + " with thumb url " + thumbUrl);
            return null;
        }
        return splittedUrl[1];
    }
}
