package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.moderationdiag.model.DiagReasonTitleAndDescription;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonKey;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonRequest;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonsResponse;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonTextService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Language;

import static com.google.common.base.CaseFormat.LOWER_CAMEL;
import static com.google.common.base.CaseFormat.LOWER_UNDERSCORE;

@Service
@ParametersAreNonnullByDefault
public class GdUcModReasonsService {
    private static final Set<Locale> SUPPORTED_LOCALES = Set.of(
            I18NBundle.RU, I18NBundle.EN, I18NBundle.TR, I18NBundle.UA);
    static final String UC_MOD_REASON_TOKEN_PLACEHOLDER = "spam"; //https://a.yandex-team.ru/arc/trunk/arcadia/frontend/services/uac/src/services/campaign/fetchModerationRejectReasons/fetchModerationRejectReasons.ts?rev=r8108640#L100
    private static final String DIAG_TEXTS_TRANSLATIONS_BUNDLE = "ru.yandex.direct.core.entity.moderatediag.ModerateDiagTranslations";
    private static final String TOKEN_REASONS_TITLE_TRANSLATIONS_BUNDLE = "ru.yandex.direct.core.entity.moderatediag.ModerateDiagTokenTitleTranslations";
    private static final String TOKEN_REASONS_TEXT_TRANSLATIONS_BUNDLE = "ru.yandex.direct.core.entity.moderatediag.ModerateDiagTokenDescriptionTranslations";

    private final ModerationReasonService moderationReasonService;
    private final ModerationReasonTextService moderationReasonTextService;
    private final TranslationService translationService;

    @Autowired
    public GdUcModReasonsService(ModerationReasonService moderationReasonService,
                                 ModerationReasonTextService moderationReasonTextService,
                                 TranslationService translationService) {
        this.moderationReasonService = moderationReasonService;
        this.moderationReasonTextService = moderationReasonTextService;
        this.translationService = translationService;
    }

    Map<ModerationReasonKey, List<ModerationDiag>> getRejectReasonDiags(
            ClientId clientId, Set<ModerationReasonKey> moderationReasonKeys) {
        return moderationReasonService.getRejectReasonDiagsByModerationKey(clientId, moderationReasonKeys);
    }


    /**
     * Обогащаем причины из базы данными по тултипам от Модерации и переводами
     */
    Map<Long, DiagReasonTitleAndDescription> getTitleAndDescByDiagId(Map<Long, ModerationDiag> diagsToDetail,
                                                                     GdLanguage lang) {
        if (diagsToDetail.isEmpty()) {
            return Map.of();
        }
        Set<Long> diagsToFetchTooltips = EntryStream.of(diagsToDetail)
                .filterValues(t -> !isReasonWithToken(t))
                .keys()
                .toSet();
        Map<Long, DiagReasonTitleAndDescription> reasonsWithTooltips = retrieveAsTooltips(
                diagsToFetchTooltips, GdLanguage.toSource(lang));
        return EntryStream.of(diagsToDetail)
                .filterKeys(id -> !reasonsWithTooltips.containsKey(id))
                .mapValues(t -> translateTitleAndText(t, fromLang(GdLanguage.toSource(lang))))
                .nonNullValues()
                .append(reasonsWithTooltips)
                .toMap((a, b) -> a);
    }

    /**
     * Получение причин по тултипам из Докцентра
     */
    Map<Long, DiagReasonTitleAndDescription> retrieveAsTooltips(Set<Long> diagIds,
                                                                Language lang) {
        ModerationReasonRequest reasonRequest = new ModerationReasonRequest(
                diagIds.stream().map(String::valueOf).toArray(String[]::new), lang
        );

        ModerationReasonsResponse resp = moderationReasonTextService.showModReasons(reasonRequest, null);
        return EntryStream.of(resp.getReasons())
                .filterKeys(StringUtils::isNumeric)
                .mapKeys(Long::valueOf)
                .nonNullValues()
                .mapValues(ModerationDiagService::extractTitleAndDescFromReasonHtml)
                .nonNullValues()
                .toMap();
    }

    //вернуть токен как в https://a.yandex-team.ru/arc/trunk/arcadia/frontend/services/uac/src/helpers/moderation/getModerationRejectReasonTitleByToken/getModerationRejectReasonTitleByToken.ts?rev=r8099362#L5
    static String convertToken(String token) {
        if (token == null) { //на всякий
            return UC_MOD_REASON_TOKEN_PLACEHOLDER;
        }
        return LOWER_UNDERSCORE.to(LOWER_CAMEL, token.trim().replace(" ", "_").toLowerCase());
    }

    static boolean isReasonWithToken(ModerationDiag moderationDiag) {
        return StringUtils.isNotBlank(moderationDiag.getToken());
    }

    private static Locale fromLang(Language language) {
        return SUPPORTED_LOCALES
                .stream()
                .filter(t -> language.equals(Language.fromLocale(t)))
                .findFirst()
                .orElse(I18NBundle.RU);
    }

    //Получение переводов по причинам, которых нет в Докцентре
    //null, если получили ерунду
    @Nullable
    private DiagReasonTitleAndDescription translateTitleAndText(ModerationDiag diag, Locale locale) {
        if (isReasonWithToken(diag)) {
            return translateTokenReason(diag, locale);
        }
        return DiagReasonTitleAndDescription.builder()
                .setTitle(translateDiagText(diag.getShortText(), locale))
                .setDescription(translateDiagText(diag.getFullText(), locale))
                .build();
    }

    private String translateDiagText(String russianText, Locale locale) {
        return translationService.translate(DIAG_TEXTS_TRANSLATIONS_BUNDLE, russianText, locale);
    }

    //Получение переводов по причинам, у которых есть токен
    //Если получить не удалось, возвращаем null
    //В будущем будем для всех, nullability удалим
    @Nullable
    private DiagReasonTitleAndDescription translateTokenReason(ModerationDiag diag, Locale locale) {
        String title = translateTokenDiagTitle(convertToken(diag.getToken()), locale);
        String description = translateTokenDiagDescription(convertToken(diag.getToken()), locale);
        if (title == null || description == null) {
            return null;
        }
        return DiagReasonTitleAndDescription.builder()
                .setTitle(title)
                .setDescription(description)
                .build();
    }

    private String translateTokenDiagTitle(String convertedToken, Locale locale) {
        String result = translationService.translate(TOKEN_REASONS_TITLE_TRANSLATIONS_BUNDLE, convertedToken, locale);
        //перевода не нашлось, в таком случае возвращается исходное значение
        //https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs/i18n/src/main/java/ru/yandex/direct/i18n/bundle/StubTranslator.java?rev=r8134267#L23
        if (Objects.equals(convertedToken, result)) {
            return null;
        }
        return result;
    }

    private String translateTokenDiagDescription(String convertedToken, Locale locale) {
        String result = translationService.translate(TOKEN_REASONS_TEXT_TRANSLATIONS_BUNDLE, convertedToken, locale);
        //перевода не нашлось, в таком случае возвращается исходное значение
        //https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs/i18n/src/main/java/ru/yandex/direct/i18n/bundle/StubTranslator.java?rev=r8134267#L23
        if (Objects.equals(convertedToken, result)) { //перевода не нашлось
            return null;
        }
        return result;
    }
}
