package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdAdAggregatedStatusInfo;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdCampaignAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignAccess;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdUcDiagModReason;
import ru.yandex.direct.grid.processing.model.campaign.GdUcKeyword;
import ru.yandex.direct.grid.processing.model.campaign.GdUcRegion;
import ru.yandex.direct.grid.processing.model.campaign.GdUniversalCampaign;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImage;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.grid.processing.model.creative.GdUcSchemeCreativeData;
import ru.yandex.direct.grid.processing.service.banner.BannerDataService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter;

import static ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService.DEFAULT_ACCESS;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Сервис, возвращающий данные об универсальной кампании по идентификатор
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class UcCampaignGraphQlService {
    private static final Logger logger = LoggerFactory.getLogger(UcCampaignGraphQlService.class);

    private final GdUcCampaignService gdUcCampaignService;
    private final GdUcCreativeService ucCreativeService;
    private final CampaignInfoService campaignInfoService;
    private final ImageDataRepository imageDataRepository;
    private final BannerDataService bannerDataService;
    private final UcRegionDescDataLoader ucRegionDescDataLoader;
    private final UcModReasonDataLoader ucModReasonDataLoader;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;

    @Autowired
    public UcCampaignGraphQlService(GdUcCampaignService gdUcCampaignService,
                                    GdUcCreativeService gdUcCreativeService,
                                    CampaignInfoService campaignInfoService,
                                    ImageDataRepository imageDataRepository,
                                    BannerDataService bannerDataService,
                                    UcRegionDescDataLoader ucRegionDescDataLoader,
                                    UcModReasonDataLoader ucModReasonDataLoader,
                                    AggregatedStatusesViewService aggregatedStatusesViewService) {
        this.gdUcCampaignService = gdUcCampaignService;
        this.ucCreativeService = gdUcCreativeService;
        this.campaignInfoService = campaignInfoService;
        this.imageDataRepository = imageDataRepository;
        this.bannerDataService = bannerDataService;
        this.ucRegionDescDataLoader = ucRegionDescDataLoader;
        this.ucModReasonDataLoader = ucModReasonDataLoader;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
    }


    @GraphQLQuery(name = "ucCampaign")
    public GdUniversalCampaign getUcCampaign(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "campaignId") Long campaignId) {
        var campaign = gdUcCampaignService.getByCampaignId(campaignId,
                context.getOperator().getUid(),
                UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId()),
                gdClient.getInfo());
        return campaign;
    }

    @GraphQLQuery(name = "image")
    public GdImage getImage(
            @GraphQLContext GdUniversalCampaign universalCampaign) {
        if (StringUtils.isBlank(universalCampaign.getImageHash())) {
            return null;
        }
        GdClientInfo clientInfo = universalCampaign.getGdClientInfo();
        Image image = imageDataRepository.getImage(clientInfo.getShard(),
                ClientId.fromLong(clientInfo.getId()), universalCampaign.getImageHash());
        return ifNotNull(image, ClientEntityConverter::toGdImage);
    }

    @GraphQLQuery(name = "ucCreativeData")
    public GdUcSchemeCreativeData getUcCreativeData(
            @GraphQLContext GdUniversalCampaign universalCampaign) {
        if (universalCampaign.getCreativeId() == null) {
            return null;
        }
        GdClientInfo clientInfo = universalCampaign.getGdClientInfo();
        return ucCreativeService.getUcCreativeData(clientInfo.getShard(), ClientId.fromLong(clientInfo.getId()),
                universalCampaign.getCreativeId(), universalCampaign.getAdId());
    }

    @GraphQLQuery(name = "aggregatedStatusInfo")
    public GdCampaignAggregatedStatusInfo getAggregatedStatusInfo(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdUniversalCampaign universalCampaign) {
        if (universalCampaign.getId() == null) {
            return null;
        }
        ClientId clientId = ClientId.fromLong(universalCampaign.getGdClientInfo().getId());
        GdiCampaign campaign = campaignInfoService
                .getCampaignsAndWalletsAndSaveToContext(clientId, Set.of(universalCampaign.getId()))
                .get(universalCampaign.getId());
        return campaignInfoService.toAggregatedStatusInfo(campaign, context.getInstant());
    }

    @GraphQLQuery(name = "adAggregatedStatusInfo")
    public GdAdAggregatedStatusInfo getAdAggregatedStatusInfo(
            @GraphQLContext GdUniversalCampaign universalCampaign) {
        if (universalCampaign.getAdId() == null) {
            return null;
        }
        AggregatedStatusAdData aggregatedStatusAdData = aggregatedStatusesViewService
                .getAdStatusesByIds(
                        universalCampaign.getGdClientInfo().getShard(), Set.of(universalCampaign.getAdId()))
                .get(universalCampaign.getAdId());
        return bannerDataService.toGdAdAggregatedStatusInfo(aggregatedStatusAdData);
    }

    @GraphQLQuery(name = "sitelinksModReasons")
    @GraphQLNonNull
    public CompletableFuture<List<GdUcDiagModReason>> getSitelinksModReasons(
            @GraphQLContext GdUniversalCampaign universalCampaign,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        if (universalCampaign.getAdId() == null) {
            return CompletableFuture.completedFuture(List.of());
        }
        return getModReasonsForObject(universalCampaign.getAdId(), ModerationReasonObjectType.SITELINKS_SET, lang,
                null);
    }

    @GraphQLQuery(name = "bannerModReasons")
    @GraphQLNonNull
    public CompletableFuture<List<GdUcDiagModReason>> getBannerModReasons(
            @GraphQLContext GdUniversalCampaign universalCampaign,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        if (universalCampaign.getAdId() == null) {
            return CompletableFuture.completedFuture(List.of());
        }
        return getModReasonsForObject(universalCampaign.getAdId(), ModerationReasonObjectType.BANNER, lang, null);
    }

    @GraphQLQuery(name = "imageModReasons")
    @GraphQLNonNull
    public CompletableFuture<List<GdUcDiagModReason>> getImageModReasons(
            @GraphQLContext GdUniversalCampaign universalCampaign,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        if (universalCampaign.getAdId() == null) {
            return CompletableFuture.completedFuture(List.of());
        }
        return getModReasonsForObject(universalCampaign.getAdId(), ModerationReasonObjectType.IMAGE, lang, null);
    }

    @GraphQLQuery(name = "ucVideoModReasons")
    @GraphQLNonNull
    public CompletableFuture<List<GdUcDiagModReason>> getVideoAdditionModReasons(
            @GraphQLContext GdUcSchemeCreativeData ucCreative,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        if (ucCreative.getAdId() == null) {
            return CompletableFuture.completedFuture(List.of());
        }
        return getModReasonsForObject(ucCreative.getAdId(), ModerationReasonObjectType.VIDEO_ADDITION, lang, null);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "access")
    public GdCampaignAccess getAccess(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdUniversalCampaign universalCampaign) {
        if (universalCampaign.getId() == null) {
            return DEFAULT_ACCESS;
        }

        ClientId clientId = ClientId.fromLong(universalCampaign.getGdClientInfo().getId());
        Map<Long, GdiCampaign> gdiCampaignsAndWallets = campaignInfoService
                .getCampaignsAndWalletsAndSaveToContext(clientId, Set.of(universalCampaign.getId()));
        return campaignInfoService
                .getCampaignsAccess(Set.of(universalCampaign.getId()), gdiCampaignsAndWallets, context.getOperator(),
                        universalCampaign.getGdClientInfo(), context.getInstant())
                .getOrDefault(universalCampaign.getId(), DEFAULT_ACCESS);
    }

    @GraphQLQuery(name = "regionName")
    @GraphQLNonNull
    public CompletableFuture<String> getRegionName(
            @GraphQLContext GdUcRegion ucRegion,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        //noinspection ConstantConditions
        return ucRegionDescDataLoader.get()
                .load(ucRegion.getRegionId())
                .thenApply(regionDesc -> ClientGeoService.getRegionName(regionDesc, GdLanguage.toSource(lang)))
                .thenApply(text -> {
                    if (text == null) {
                        logger.error("unknown region " + ucRegion.getRegionId());
                    }
                    return nvl(text, "");
                });
    }

    //причины "в целом про фразы", а не про конкретные
    @GraphQLQuery(name = "adgroupPhrasesCommonModReasons")
    @GraphQLNonNull
    public CompletableFuture<List<GdUcDiagModReason>> getAdgroupPhrasesCommonModReasons(
            @GraphQLContext GdUniversalCampaign universalCampaign,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        if (universalCampaign.getAdGroupId() == null) {
            return CompletableFuture.completedFuture(List.of());
        }
        return getModReasonsForObject(
                universalCampaign.getAdGroupId(), ModerationReasonObjectType.PHRASES, lang, null);
    }

    @GraphQLQuery(name = "phraseModReasons")
    @GraphQLNonNull
    public CompletableFuture<List<GdUcDiagModReason>> getPhraseModReasons(
            @GraphQLContext GdUcKeyword ucKeyword,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        if (ucKeyword.getPid() == null) {
            logger.error("no pid for phrase with id " + ucKeyword.getId());
            return CompletableFuture.completedFuture(List.of());
        }
        return getModReasonsForObject(
                ucKeyword.getPid(), ModerationReasonObjectType.PHRASES, lang, ucKeyword.getId());
    }

    @Nonnull
    private CompletableFuture<List<GdUcDiagModReason>> getModReasonsForObject(
            Long id, ModerationReasonObjectType objectType, GdLanguage lang, Long subObjectId) {
        return ucModReasonDataLoader.get()
                .load(new UcModerationReasonRequestItem(id, subObjectId, objectType, lang))
                .thenApply(reasons -> nvl(reasons, List.of()));
    }
}
