package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Predicates;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesService;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RecalculationDepthEnum;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.BannerSuspendResumeService;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.uc.UcCampaignService;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstantsService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.feature.service.FeatureHelper;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.KeywordInclusionService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.uc.UcRetargetingConditionService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddUcCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdMeaningfulGoalRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationPayload;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationResult;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateUcCampaignInput;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignConverterContext;
import ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableAutoGoalsAllowedForCampaignWithStrategy;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableGoalsAllowed;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexDynamicAdGroup;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexPerformanceAdGroup;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexTextAdGroup;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexTextAdGroupWithDynamicBanner;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexTextAdGroupWithDynamicBanners;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexTextAdGroupWithSmartBanner;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexTextAdGroupWithSmartBanners;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toCoreCampaignModelChanges;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toCoreDynamicCampaignModelChanges;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toCoreSmartCampaignModelChanges;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toCoreTextCampaign;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toDynamicCampaign;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toSmartCampaign;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResultIfErrors;
import static ru.yandex.direct.grid.processing.util.GdValidationResultUtils.concatValidationResults;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getSuccessfullyResults;
import static ru.yandex.direct.utils.CollectionUtils.nullIfEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;

@Service
@ParametersAreNonnullByDefault
public class UcCampaignMutationService {
    private static final Logger logger = LoggerFactory.getLogger(UcCampaignMutationService.class);
    private static final Set<ModelProperty> UPDATING_COMPLEX_AD_GROUP_PROPERTIES = Set.of(
            ComplexTextAdGroup.COMPLEX_BID_MODIFIER, ComplexTextAdGroup.KEYWORDS,
            ComplexTextAdGroup.RETARGETING_CONDITION, ComplexTextAdGroup.TARGET_INTERESTS);
    private final AttributeResolverService attributeResolverService;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final HostingsHandler hostingsHandler;
    private final ClientGeoService clientGeoService;
    private final RecentStatisticsService recentStatisticsService;
    private final BannerModerateService bannerModerateService;
    private final UcCampaignValidationService ucCampaignValidationService;
    private final AggregatedStatusesService aggregatedStatusesService;
    private final BannerSuspendResumeService bannerSuspendResumeService;
    private final CampaignGoalsService campaignGoalsService;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final KeywordRepository keywordRepository;
    private final KeywordInclusionService keywordInclusionService;
    private final UcRetargetingConditionService ucRetargetingConditionService;
    private final UcCampaignService ucCampaignService;
    private final CampaignService campaignService;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;
    private final PpcProperty<Boolean> recalculateAggregatedStatusesProperty;
    private final PpcProperty<Boolean> useNewBannersInComplexPrevalidation;
    private final CampaignConstantsService campaignConstantsService;

    public UcCampaignMutationService(AttributeResolverService attributeResolverService,
                                     SspPlatformsRepository sspPlatformsRepository,
                                     HostingsHandler hostingsHandler,
                                     ClientGeoService clientGeoService,
                                     RecentStatisticsService recentStatisticsService,
                                     BannerModerateService bannerModerateService,
                                     UcCampaignValidationService ucCampaignValidationService,
                                     AggregatedStatusesService aggregatedStatusesService,
                                     BannerSuspendResumeService bannerSuspendResumeService,
                                     CampaignGoalsService campaignGoalsService,
                                     BannerRelationsRepository bannerRelationsRepository,
                                     CampaignTypedRepository campaignTypedRepository,
                                     KeywordRepository keywordRepository,
                                     KeywordInclusionService keywordInclusionService,
                                     UcRetargetingConditionService ucRetargetingConditionService,
                                     UcCampaignService ucCampaignService,
                                     CampaignService campaignService,
                                     FeatureService featureService,
                                     ShardHelper shardHelper,
                                     PpcPropertiesSupport ppcPropertiesSupport,
                                     EnvironmentType environmentType,
                                     CampaignConstantsService campaignConstantsService) {
        this.attributeResolverService = attributeResolverService;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.hostingsHandler = hostingsHandler;
        this.clientGeoService = clientGeoService;
        this.recentStatisticsService = recentStatisticsService;
        this.bannerModerateService = bannerModerateService;
        this.campaignGoalsService = campaignGoalsService;
        this.ucRetargetingConditionService = ucRetargetingConditionService;
        this.ucCampaignValidationService = ucCampaignValidationService;
        this.aggregatedStatusesService = aggregatedStatusesService;
        this.bannerSuspendResumeService = bannerSuspendResumeService;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.keywordRepository = keywordRepository;
        this.keywordInclusionService = keywordInclusionService;
        this.ucCampaignService = ucCampaignService;
        this.campaignService = campaignService;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.recalculateAggregatedStatusesProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.RECALCULATE_AGGREGATED_STATUSES_ON_SAVE_UC,
                        Duration.ofMinutes(5));
        this.useNewBannersInComplexPrevalidation = environmentType.isDevelopment() ?
                ppcPropertiesSupport.get(PpcPropertyNames.USE_BANNERS_ADD_UAC_TGO_UPDATE_PREVALIDATION) :
                ppcPropertiesSupport
                        .get(PpcPropertyNames.USE_BANNERS_ADD_UAC_TGO_UPDATE_PREVALIDATION, Duration.ofMinutes(1));
        this.campaignConstantsService = campaignConstantsService;
    }

    public GdUcCampaignMutationPayload addUcCampaign(GridGraphQLContext context, GdAddUcCampaignInput input) {
        ucCampaignValidationService.validateAddRequest(input);

        User client = context.getSubjectUser();
        ClientId clientId = client.getClientId();
        Long operatorUid = context.getOperator().getUid();
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        RetargetingCondition autoRetargeting = ucRetargetingConditionService.getAutoRetargetingCondition(
                clientId, input.getMetrikaCounters(), getGoalIds(input), null);
        var result = doMutation(
                () -> validateAddingGroup(input, geoTree, operatorUid, clientId, autoRetargeting),
                () -> createCampaign(input, operatorUid, client, null),
                campaignId -> createGroup(input, geoTree, operatorUid, client, campaignId, autoRetargeting)
        );
        recalculateAggregatedStatusesIfNeeded(clientId, result);
        return result;
    }

    public GdUcCampaignMutationPayload updateUcCampaign(GridGraphQLContext context, GdUpdateUcCampaignInput input) {
        ucCampaignValidationService.validateUpdateRequest(input);

        User client = context.getSubjectUser();
        ClientId clientId = client.getClientId();
        Long operatorUid = context.getOperator().getUid();
        int shard = shardHelper.getShardByClientId(clientId);
        Long campaignId = input.getId();
        TextCampaign textCampaign = StreamEx.of(campaignTypedRepository.getTypedCampaigns(shard, List.of(campaignId)))
                .select(TextCampaign.class)
                .findFirst()
                .orElse(null);
        if (textCampaign == null ||
                !nvl(textCampaign.getIsUniversal(), false) ||
                !AvailableCampaignSources.INSTANCE.isUC(textCampaign.getSource())) {
            return new GdUcCampaignMutationPayload().withValidationResult(
                    buildGridValidationResult(ValidationResult.failed(null, campaignNotFound())));
        }

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        Map<Long, List<Long>> bannerIdsToAdGroupIds = bannerRelationsRepository.getAdGroupIdsToNonArchivedBannerIds(
                shard, campaignId, Set.of(TextBanner.class));
        Long adGroupId = StreamEx.of(bannerIdsToAdGroupIds.keySet()).min(Comparator.naturalOrder()).orElse(null);
        RetargetingCondition autoRetargeting = ucRetargetingConditionService.getAutoRetargetingCondition(
                clientId, input.getMetrikaCounters(), getGoalIds(input), adGroupId);

        Supplier<ValidationResult<?, Defect>> groupValidator;
        Function<Long, Result<Long>> groupMutator;
        if (adGroupId == null) {
            logger.info("Ad group not found for campaignId={}, trying to create a new one", campaignId);
            groupValidator = () -> validateAddingGroup(input, geoTree, operatorUid, clientId, autoRetargeting);
            groupMutator = (cid) -> createGroup(input, geoTree, operatorUid, client, cid, autoRetargeting);
        } else {
            // в случае bannerId == null создатся новый баннер
            Long bannerId = StreamEx.of(bannerIdsToAdGroupIds.get(adGroupId))
                    .min(Comparator.naturalOrder())
                    .orElse(null);
            List<Keyword> existingKeywords = keywordRepository.getKeywordsByAdGroupId(shard, adGroupId);

            boolean tabletBidModifierEnabled = featureService
                    .isEnabledForClientId(clientId, FeatureName.TABLET_BIDMODIFER_ENABLED);

            ComplexTextAdGroup complexGroup = toComplexTextAdGroup(input, adGroupId, bannerId,
                    existingKeywords, autoRetargeting, tabletBidModifierEnabled);
            groupValidator = () -> validateUpdatingGroup(complexGroup, geoTree, existingKeywords,
                    operatorUid, clientId);
            groupMutator = (cid) -> updateGroup(complexGroup, geoTree, operatorUid, client);
        }

        var result = doMutation(
                groupValidator,
                () -> updateCampaign(input, textCampaign.getMinusKeywords(), operatorUid, client),
                groupMutator);

        List<Long> bannerIds = bannerRelationsRepository.getBannerIdsByCampaignIds(shard, List.of(campaignId));
        moderateBannersIfNeeded(clientId, operatorUid, bannerIds, textCampaign, input.getIsDraft());
        resumeBannersIfNeeded(clientId, operatorUid, bannerIds);
        recalculateAggregatedStatusesIfNeeded(clientId, result);
        return result;
    }

    private GdUcCampaignMutationPayload doMutation(
            Supplier<ValidationResult<?, Defect>> groupValidator,
            Supplier<Result<Long>> campaignMutator,
            Function<Long, Result<Long>> groupMutator
    ) {
        var gdValidationResult = buildGridValidationResult(groupValidator.get());
        if (gdValidationResult != null && !gdValidationResult.getErrors().isEmpty()) {
            return new GdUcCampaignMutationPayload().withValidationResult(gdValidationResult);
        }

        Result<Long> campaignResult = campaignMutator.get();

        var campaignValidationResult = buildGridValidationResult(campaignResult.getValidationResult());
        if (campaignValidationResult != null && !campaignValidationResult.getErrors().isEmpty()) {
            return new GdUcCampaignMutationPayload().withValidationResult(campaignValidationResult);
        }

        Result<Long> groupResult = groupMutator.apply(campaignResult.getResult());

        var groupValidationResult = buildGridValidationResult(groupResult.getValidationResult());

        GdValidationResult validationResult = concatValidationResults(campaignValidationResult, groupValidationResult);
        if (validationResult != null && !validationResult.getErrors().isEmpty()) {
            return new GdUcCampaignMutationPayload().withValidationResult(validationResult);
        }

        var result = new GdUcCampaignMutationResult().withCampaignId(campaignResult.getResult());
        return new GdUcCampaignMutationPayload().withResult(result).withValidationResult(validationResult);
    }

    private void recalculateAggregatedStatusesIfNeeded(ClientId clientId, GdUcCampaignMutationPayload payload) {
        if (payload.getResult() != null && payload.getResult().getCampaignId() != null
                && recalculateAggregatedStatusesProperty.getOrDefault(false)) {
            int shard = shardHelper.getShardByClientId(clientId);
            try (TraceProfile ignored = Trace.current().profile("aggregatedStatuses:fullyRecalculate")) {
                aggregatedStatusesService.fullyRecalculateStatuses(shard, LocalDateTime.now(),
                        Set.of(payload.getResult().getCampaignId()), RecalculationDepthEnum.ALL);
            }
        }
    }

    private void moderateBannersIfNeeded(ClientId clientId, Long operatorUid, List<Long> bannerIds,
                                         TextCampaign textCampaign, boolean isDraft) {
        boolean isCurrentCampaignDraft =
                nvl(textCampaign.getStatusModerate(), CampaignStatusModerate.NEW) == CampaignStatusModerate.NEW;
        //отправляем все баннеры на модерацию, если надо
        if (!isDraft && isCurrentCampaignDraft && !bannerIds.isEmpty()) {
            var vr = bannerModerateService.moderateBanners(clientId, operatorUid, bannerIds);
            logValidationResultErrors("Error while making nondraft", vr.getValidationResult());
        }
    }

    private void resumeBannersIfNeeded(ClientId clientId, Long operatorUid, List<Long> bannerIds) {
        // Возобновляем баннеры, т.к. модерация может выставить statusShow=No,
        // а после принятия модерацией этот статус не переводится в Yes. А в интерфейсе UC не предусмотрено
        // кнопки возобновления баннеров (например как для ТГО на гридах), поэтому делаем автоматом.
        if (!bannerIds.isEmpty()) {
            List<ModelChanges<BannerWithSystemFields>> bannersToResume =
                    mapList(bannerIds, id -> new ModelChanges<>(id, BannerWithSystemFields.class)
                            .process(true, BannerWithSystemFields.STATUS_SHOW));
            var resumeBannersVr = bannerSuspendResumeService.suspendResumeBanners(clientId,
                    operatorUid, bannersToResume, true).getValidationResult();
            logValidationResultErrors("Error while resume banners", resumeBannersVr, true);
        }
    }

    /**
     * Валидирует создание группы объявлений ДО самого создания
     * <p>
     * В случае Еком сценария сначала проверяет создание динамической и смарт групп для подкампаний,
     * если ошибок там не оказалось, то валадирует создание группы мастер-кампании.
     * Порядок таков, потому что у динамической и смарт групп могут быть "более специфичные" ошибки валидации
     * (например, валидация трекинговых параметров в них проходит отдельно, а в текстовой группе - скленно со ссылкой)
     */
    public ValidationResult<?, Defect> validateAddingGroup(
            GdUcCampaignMutationInput input, GeoTree geoTree,
            Long operatorUid, ClientId clientId, @Nullable RetargetingCondition autoretargetingCondition) {

        boolean tabletBidModifierEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.TABLET_BIDMODIFER_ENABLED);

        boolean multipleSegmentsInHyperGeo = featureService
                .isEnabledForClientId(clientId, FeatureName.HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA);

        ValidationResult<?, Defect> ecomGroupsResult = null;
        if (Boolean.TRUE.equals(input.getIsEcom())
                && FeatureHelper.feature(FeatureName.ECOM_UC_NEW_BACKEND_ENABLED).disabled()) {
            ecomGroupsResult = validateAddingEcomGroupsOnOldBackend(input, geoTree, clientId,
                    multipleSegmentsInHyperGeo, tabletBidModifierEnabled);
        } else if (Boolean.TRUE.equals(input.getIsEcom())) {
            ecomGroupsResult = validateAddingEcomGroupsOnNewBackend(input, geoTree, clientId, operatorUid, autoretargetingCondition, multipleSegmentsInHyperGeo, tabletBidModifierEnabled);
        }
        if (ecomGroupsResult != null) {
            return ecomGroupsResult;
        }

        var complexGroup = toComplexTextAdGroup(
                input, null, autoretargetingCondition, tabletBidModifierEnabled);

        var textGroupResult =
                ucCampaignValidationService.preValidateAddingComplexTextAdGroup(clientId,
                        operatorUid, complexGroup, geoTree, multipleSegmentsInHyperGeo);
        if (textGroupResult.hasAnyErrors()) {
            logValidationResultErrors("Error validating text ad group", textGroupResult);
        }
        return textGroupResult;
    }

    @Nullable
    private ValidationResult<?, Defect> validateAddingEcomGroupsOnOldBackend(GdUcCampaignMutationInput input,
                                                                             GeoTree geoTree,
                                                                             ClientId clientId,
                                                                             boolean multipleSegmentsInHyperGeo,
                                                                             boolean tabletBidModifierEnabled) {
        var dynamicGroupResult = ucCampaignValidationService
                .preValidateAddingComplexDynamicAdGroup(
                        clientId, input, geoTree, multipleSegmentsInHyperGeo, tabletBidModifierEnabled);
        if (dynamicGroupResult.hasAnyErrors()) {
            logValidationResultErrors("Error validating dynamic ad group", dynamicGroupResult);
            return dynamicGroupResult;
        }

        var smartGroupResult = ucCampaignValidationService
                .preValidateAddingComplexPerformanceAdGroup(
                        clientId, input, geoTree, multipleSegmentsInHyperGeo, tabletBidModifierEnabled);
        if (smartGroupResult.hasAnyErrors()) {
            logValidationResultErrors("Error validating smart ad group", smartGroupResult);
            return smartGroupResult;
        }
        return null;
    }

    @Nullable
    private ValidationResult<?, Defect> validateAddingEcomGroupsOnNewBackend(GdUcCampaignMutationInput input,
                                                                             GeoTree geoTree,
                                                                             ClientId clientId,
                                                                             Long operatorUid,
                                                                             @Nullable RetargetingCondition autoretargetingCondition,
                                                                             boolean multipleSegmentsInHyperGeo,
                                                                             boolean tabletBidModifierEnabled) {
        var groupForDynamics = toComplexTextAdGroupWithDynamicBanners(
                input, null, autoretargetingCondition, tabletBidModifierEnabled);
        var dynamicResult =
                ucCampaignValidationService.preValidateAddingComplexTextAdGroup(clientId,
                        operatorUid, groupForDynamics, geoTree, multipleSegmentsInHyperGeo);
        if (dynamicResult.hasAnyErrors()) {
            logValidationResultErrors("Error validating dynamic ad group", dynamicResult);
            return dynamicResult;
        }

        var groupForSmarts = toComplexTextAdGroupWithSmartBanners(
                input, null, autoretargetingCondition, tabletBidModifierEnabled);
        var smartResult =
                ucCampaignValidationService.preValidateAddingComplexTextAdGroup(clientId,
                        operatorUid, groupForSmarts, geoTree, multipleSegmentsInHyperGeo);
        if (smartResult.hasAnyErrors()) {
            logValidationResultErrors("Error validating smart ad group", smartResult);
            return smartResult;
        }
        return null;
    }

    public Result<Long> createCampaign(GdAddUcCampaignInput input, Long operatorUid, User client, @Nullable List<String> minusKeywords) {
        List<String> enabledMinusKeywords = null;
        if (featureService.isEnabledForClientId(client.getClientId(), FeatureName.UC_ENABLE_OPTIONAL_KEYWORDS) ||
                featureService.isEnabledForClientId(client.getClientId(), FeatureName.UAC_ENABLE_MINUS_KEYWORDS_TGO)) {
            enabledMinusKeywords = minusKeywords;
        }
        var converterContext =
                CampaignConverterContext.create(attributeResolverService, sspPlatformsRepository, hostingsHandler);
        var masterCampaign = toCoreTextCampaign(input, client.getEmail(), enabledMinusKeywords, converterContext,
                campaignConstantsService.getDefaultAttributionModel(),
                FeatureHelper.feature(FeatureName.ADVANCED_GEOTARGETING).enabled()
        );
        MassResult<Long> massResult = ucCampaignService.createCampaign(masterCampaign, input.getIsDraft(),
                operatorUid, UidAndClientId.of(client.getUid(), client.getClientId()));
        var isEcom = nvl(input.getIsEcom(), false);
        if (isEcom && massResult.isSuccessful() && FeatureHelper.feature(FeatureName.ECOM_UC_NEW_BACKEND_ENABLED).disabled()) {
            createSubCampaigns(massResult.get(0).getResult(), input, converterContext, operatorUid, client);
        }
        logValidationResultErrors(
                isEcom ? "error while creating ecom campaigns" : "error while creating campaigns",
                massResult.getValidationResult());
        return massResult.get(0);
    }

    private void createSubCampaigns(Long masterCid,
                                    GdAddUcCampaignInput input,
                                    CampaignConverterContext converterContext,
                                    Long operatorUid,
                                    User client) {
        var dynamicCampaign = toDynamicCampaign(masterCid, input, client.getEmail(), null, converterContext,
                campaignConstantsService.getDefaultAttributionModel());
        var result = ucCampaignService.createCampaign(dynamicCampaign, input.getIsDraft(),
                operatorUid, UidAndClientId.of(client.getUid(), client.getClientId()));
        logValidationResultErrors("error while creating dynamic subcampaign", result.getValidationResult());

        var counterId = getCounterIdForSmartCampaign(operatorUid, client.getClientId(), input);
        var smartCampaign = toSmartCampaign(
                masterCid,
                input,
                client.getEmail(),
                null,
                counterId,
                converterContext,
                campaignConstantsService.getDefaultAttributionModel());
        result = ucCampaignService.createCampaign(smartCampaign, input.getIsDraft(),
                operatorUid, UidAndClientId.of(client.getUid(), client.getClientId()));
        logValidationResultErrors("error while creating smart subcampaign", result.getValidationResult());
    }

    /**
     * С входными параметрами может передаваться список счётчиков, включая недоступные,
     * а для смарт-кампании нам нужен ровно один доступный счётчик. Здесь выбираем счётчик
     * по указанным целям. Доступность счётчика должна заранее проверяться в валидации.
     * Наличие хотя бы одной цели и принадлежность всех целей одному счетчику также гарантируется валидацией.
     */
    private Integer getCounterIdForSmartCampaign(Long operatorUid, ClientId clientId, GdUcCampaignMutationInput input) {
        var inputGoalIds = listToSet(nvl(getGoalIds(input), emptyList()));
        var goals = campaignGoalsService.getAvailableGoalsForCampaignType(
                operatorUid, clientId, Set.of(getSmartCampaignTypeWithCounterIds(clientId, input)), null
        );
        var goal = StreamEx.of(goals.values())
                .flatMap(Set::stream)
                .filter(Predicates.compose(inputGoalIds::contains, Goal::getId))
                .findFirst()
                .orElseThrow();
        return goal.getCounterId();
    }

    private CampaignTypeWithCounterIds getSmartCampaignTypeWithCounterIds(ClientId clientId, GdUcCampaignMutationInput input) {
        var enabledFeatures = featureService.getEnabledForClientId(clientId);
        return new CampaignTypeWithCounterIds()
                .withCampaignType(CampaignType.PERFORMANCE)
                .withCounterIds(listToSet(input.getMetrikaCounters(), Integer::longValue))
                .withUnavailableAutoGoalsAllowed(
                        isUnavailableAutoGoalsAllowedForCampaignWithStrategy(
                                CampaignSource.UAC,
                                getStrategyName(input),
                                input.getGoalId(),
                                input.getAvgCpa(),
                                input.getAvgCpa() != null || input.getCrr() != null,
                                enabledFeatures)
                )
                .withUnavailableGoalsAllowed(
                        isUnavailableGoalsAllowed(CampaignSource.UAC, enabledFeatures)
                );
    }

    private StrategyName getStrategyName(GdUcCampaignMutationInput input) {
        if (input.getCrr() != null && input.getCrr() > 0) {
            return StrategyName.AUTOBUDGET_CRR;
        } else if (input.getGoalId() != null && input.getAvgCpa() != null) {
            return StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER;
        }
        return StrategyName.AUTOBUDGET;
    }

    private Result<Long> createGroup(GdUcCampaignMutationInput input, GeoTree geoTree,
                                     Long operatorUid, User client, Long campaignId,
                                     @Nullable RetargetingCondition autoretargetingCondition) {
        ClientId clientId = client.getClientId();

        boolean tabletBidModifierEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.TABLET_BIDMODIFER_ENABLED);

        var complexGroup = toComplexTextAdGroup(
                input, campaignId, autoretargetingCondition, tabletBidModifierEnabled);

        var autoPriceParams = new ShowConditionAutoPriceParams(ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null),
                recentStatisticsService);
        MassResult<Long> massResult = ucCampaignService.createGroup(complexGroup, geoTree, autoPriceParams,
                operatorUid, UidAndClientId.of(client.getUid(), clientId));
        logValidationResultErrors("error while creating groups", massResult.getValidationResult());
        List<Long> createdAdGroupIds = getSuccessfullyResults(massResult, Function.identity());
        if (createdAdGroupIds.isEmpty()) {
            deleteCampaigns(operatorUid, clientId, List.of(campaignId));
        }
        return massResult.get(0);
    }

    private void deleteCampaigns(Long operatorUid, ClientId clientId, List<Long> campaignIds) {
        logger.info("Deleting campaigns {}", campaignIds);
        MassResult<Long> deleteCampaignResult = ucCampaignService.deleteCampaigns(operatorUid, clientId, campaignIds);
        logValidationResultErrors("Error while deleting campaign", deleteCampaignResult.getValidationResult());
        List<Long> deletedCampaignIds = getSuccessfullyResults(deleteCampaignResult, Function.identity());
        logger.info("Successfully deleted campaigns: {}", deletedCampaignIds);
    }

    private ValidationResult<?, Defect> validateUpdatingGroup(ComplexTextAdGroup complexTextAdGroup,
                                                              GeoTree geoTree, List<Keyword> existingKeywords,
                                                              Long operatorUid, ClientId clientId) {
        boolean multipleSegmentsInHyperGeo = featureService
                .isEnabledForClientId(clientId, FeatureName.HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA);

        return ucCampaignValidationService.preValidateUpdatingComplexAdGroup(clientId, operatorUid,
                complexTextAdGroup, geoTree, existingKeywords, multipleSegmentsInHyperGeo);
    }

    //поскольку в uac действия производятся пересозданием баннеров, используем bannerId==null в конвертации в ComplexTextAdGroup
    public ValidationResult<?, Defect> validateUpdatingGroupForUac(GdUpdateUcCampaignInput input,
                                                                   Long operatorUid,
                                                                   ClientId clientId) {
        var geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        boolean multipleSegmentsInHyperGeo = featureService
                .isEnabledForClientId(clientId, FeatureName.HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA);

        int shard = shardHelper.getShardByClientId(clientId);
        Long campaignId = input.getId();
        Map<Long, List<Long>> bannerIdsToAdGroupIds = bannerRelationsRepository.getAdGroupIdsToNonArchivedBannerIds(
                shard, campaignId, Set.of(TextBanner.class));
        Long adGroupId = StreamEx.of(bannerIdsToAdGroupIds.keySet())
                .min(Comparator.naturalOrder())
                .orElse(null);

        var autoRetargeting = ucRetargetingConditionService.getAutoRetargetingCondition(
                clientId, input.getMetrikaCounters(), getGoalIds(input), adGroupId);
        if (adGroupId == null) {
            return validateAddingGroup(input, geoTree, operatorUid, clientId, autoRetargeting);
        }

        Long bannerId = null;
        if (!useNewBannersInComplexPrevalidation.getOrDefault(false)) {
            bannerId = StreamEx.of(bannerIdsToAdGroupIds.get(adGroupId))
                    .min(Comparator.naturalOrder())
                    .orElseThrow(() -> new IllegalStateException("Banner not found"));
        }
        boolean tabletBidModifierEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.TABLET_BIDMODIFER_ENABLED);

        if (Boolean.TRUE.equals(input.getIsEcom()) && FeatureHelper.feature(FeatureName.ECOM_UC_NEW_BACKEND_ENABLED).disabled()) {
            var subCampaignsIds = campaignService.getSubCampaignIdsWithMasterIds(Set.of(campaignId), clientId).keySet();
            var subCampaignsIdsByType =
                    EntryStream.of(campaignTypedRepository.getClientCampaignIdsWithType(shard, clientId, subCampaignsIds))
                            .invert()
                            .toMap();

            var dynamicCampaignId = subCampaignsIdsByType.get(CampaignsType.dynamic);
            var dynamicGroupResult = validateUpdatingDynamicSubCampaign(
                    clientId,
                    dynamicCampaignId,
                    input,
                    geoTree,
                    multipleSegmentsInHyperGeo);
            if (dynamicGroupResult.hasAnyErrors()) {
                return dynamicGroupResult;
            }

            var performanceCampaignId = subCampaignsIdsByType.get(CampaignsType.performance);
            var performanceGroupResult = validateUpdatingPerformanceSubCampaign(
                    clientId,
                    performanceCampaignId,
                    input,
                    geoTree,
                    multipleSegmentsInHyperGeo);
            if (performanceGroupResult.hasAnyErrors()) {
                return performanceGroupResult;
            }
        } else if (Boolean.TRUE.equals(input.getIsEcom())) {
            var subResult = validateUpdatingEcomGroupsOnNewBackend(input, clientId, operatorUid,
                    campaignId, geoTree, DynamicBanner.class, tabletBidModifierEnabled, multipleSegmentsInHyperGeo);
            if (subResult != null) {
                logValidationResultErrors("Error validating dynamic ad group", subResult);
                return subResult;
            }

            subResult = validateUpdatingEcomGroupsOnNewBackend(input, clientId, operatorUid,
                    campaignId, geoTree, PerformanceBanner.class, tabletBidModifierEnabled, multipleSegmentsInHyperGeo);
            if (subResult != null) {
                logValidationResultErrors("Error validating smart ad group", subResult);
                return subResult;
            }
        }

        List<Keyword> existingKeywords = keywordRepository.getKeywordsByAdGroupId(shard, adGroupId);
        var complexTextAdGroup = toComplexTextAdGroup(input, adGroupId, bannerId,
                existingKeywords, autoRetargeting, tabletBidModifierEnabled);
        return ucCampaignValidationService.preValidateUpdatingComplexAdGroup(clientId, operatorUid,
                complexTextAdGroup, geoTree, existingKeywords, multipleSegmentsInHyperGeo);
    }

    @Nullable
    private ValidationResult<?, Defect> validateUpdatingEcomGroupsOnNewBackend(GdUpdateUcCampaignInput input,
                                                                               ClientId clientId,
                                                                               Long operatorUid,
                                                                               Long campaignId,
                                                                               GeoTree geoTree,
                                                                               Class<? extends Banner> bannerType,
                                                                               boolean tabletBidModifierEnabled,
                                                                               boolean multipleSegmentsInHyperGeo) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, List<Long>> bannerIdsToAdGroupIds = bannerRelationsRepository.getAdGroupIdsToNonArchivedBannerIds(
                shard, campaignId, bannerType.equals(PerformanceBanner.class) ?
                Set.of(PerformanceBanner.class, PerformanceBannerMain.class) : Set.of(bannerType));
        Long adGroupId = StreamEx.of(bannerIdsToAdGroupIds.keySet())
                .min(Comparator.naturalOrder())
                .orElseThrow(() -> new IllegalStateException("Ad group not found"));
        var autoRetargeting = ucRetargetingConditionService.getAutoRetargetingCondition(
                clientId, input.getMetrikaCounters(), getGoalIds(input), adGroupId);
        Long bannerId = null;
        if (!useNewBannersInComplexPrevalidation.getOrDefault(false)) {
            bannerId = StreamEx.of(bannerIdsToAdGroupIds.get(adGroupId))
                    .min(Comparator.naturalOrder())
                    .orElseThrow(() -> new IllegalStateException("Banner not found"));
        }
        List<Keyword> existingKeywords = keywordRepository.getKeywordsByAdGroupId(shard, adGroupId);
        var group = bannerType.equals(DynamicBanner.class) ?
                toComplexTextAdGroupWithDynamicBanner(input, adGroupId, bannerId, existingKeywords, autoRetargeting, tabletBidModifierEnabled)
                : toComplexTextAdGroupWithSmartBanner(input, adGroupId, bannerId, existingKeywords, autoRetargeting, tabletBidModifierEnabled);
        var result = ucCampaignValidationService.preValidateUpdatingComplexAdGroup(clientId, operatorUid,
                group, geoTree, existingKeywords, multipleSegmentsInHyperGeo);
        if (result.hasAnyErrors()) {
            return result;
        }
        return null;
    }

    private ValidationResult<?, Defect> validateUpdatingDynamicSubCampaign(
            ClientId clientId,
            Long dynamicCampaignId,
            GdUpdateUcCampaignInput input,
            GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo
    ) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        var bannerIdsToAdGroupIds = bannerRelationsRepository.getAdGroupIdsToNonArchivedBannerIds(
                shard, dynamicCampaignId, Set.of(DynamicBanner.class));
        var adGroupId = StreamEx.of(bannerIdsToAdGroupIds.keySet())
                .min(Comparator.naturalOrder())
                .orElseThrow(() -> new IllegalStateException("Dynamic ad group not found"));
        Long bannerId = null;
        if (!useNewBannersInComplexPrevalidation.getOrDefault(false)) {
            bannerId = StreamEx.of(bannerIdsToAdGroupIds.get(adGroupId))
                    .min(Comparator.naturalOrder())
                    .orElseThrow(() -> new IllegalStateException("Banner not found"));
        }

        boolean tabletBidModifierEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.TABLET_BIDMODIFER_ENABLED);

        var complexDynamicAdGroup = toComplexDynamicAdGroup(
                input, dynamicCampaignId, adGroupId, bannerId, tabletBidModifierEnabled);
        return ucCampaignValidationService.preValidateUpdatingComplexDynamicAdGroup(
                clientId, complexDynamicAdGroup, geoTree, multipleSegmentsInHyperGeo);
    }

    private ValidationResult<?, Defect> validateUpdatingPerformanceSubCampaign(
            ClientId clientId,
            Long performanceCampaignId,
            GdUpdateUcCampaignInput input,
            GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        var bannerIdsToAdGroupIds = bannerRelationsRepository.getAdGroupIdsToNonArchivedBannerIds(
                shard, performanceCampaignId, Set.of(PerformanceBanner.class, PerformanceBannerMain.class));
        var adGroupId = StreamEx.of(bannerIdsToAdGroupIds.keySet())
                .min(Comparator.naturalOrder())
                .orElseThrow(() -> new IllegalStateException("Performance ad group not found"));

        boolean tabletBidModifierEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.TABLET_BIDMODIFER_ENABLED);

        var complexPerformanceAdGroup = toComplexPerformanceAdGroup(
                input, performanceCampaignId, adGroupId, tabletBidModifierEnabled);
        return ucCampaignValidationService.preValidateUpdatingComplexPerformanceAdGroup(
                clientId, complexPerformanceAdGroup, geoTree, multipleSegmentsInHyperGeo);
    }

    public Result<Long> updateCampaign(GdUpdateUcCampaignInput input,
                                       @Nullable List<String> existingMinusKeywords,
                                       Long operatorUid, User client) {
        List<String> newMinusKeywords = CollectionUtils.isNotEmpty(existingMinusKeywords) ?
                keywordInclusionService.getMinusKeywordsNotIncludedInPlusKeywords(input.getKeywords(),
                        existingMinusKeywords) : null;
        var converterContext = CampaignConverterContext.create(attributeResolverService, sspPlatformsRepository, hostingsHandler);
        var uiAndClientId = UidAndClientId.of(client.getUid(), client.getClientId());
        var changes = toCoreCampaignModelChanges(input, client.getEmail(), nullIfEmpty(newMinusKeywords),
                converterContext, campaignConstantsService.getDefaultAttributionModel(),
                featureService.isEnabledForClientId(client.getClientId(), FeatureName.ADVANCED_GEOTARGETING));
        MassResult<Long> massResult = ucCampaignService.updateCampaign(changes, operatorUid, uiAndClientId);

        if (nvl(input.getIsEcom(), false) && massResult.getUnsuccessfulObjectsCount() == 0
                && FeatureHelper.feature(FeatureName.ECOM_UC_NEW_BACKEND_ENABLED).disabled()) {
            var result = updateSubCampaigns(massResult.get(0).getResult(),
                    input, converterContext, newMinusKeywords, operatorUid, client);
            if (!result.isSuccessful()) {
                return Result.broken(result.getValidationResult());
            }
        }

        logValidationResultErrors("error while updating campaigns", massResult.getValidationResult());
        return massResult.get(0);
    }

    private MassResult<Long> updateSubCampaigns(Long masterCid,
                                                GdUpdateUcCampaignInput input,
                                                CampaignConverterContext converterContext,
                                                @Nullable List<String> minusKeywords,
                                                Long operatorUid,
                                                User client) {
        var uidAndClientId = UidAndClientId.of(client.getUid(), client.getClientId());
        int shard = shardHelper.getShardByClientId(client.getClientId());
        var subCampaignIds = campaignService.getSubCampaignIdsWithMasterIds(Set.of(masterCid), client.getClientId());
        var subCampaigns = campaignTypedRepository.getTypedCampaigns(shard, subCampaignIds.keySet());
        var dynamicCampaignId = onlyCampaignId(subCampaigns, DynamicCampaign.class);
        var smartCampaignId = onlyCampaignId(subCampaigns, SmartCampaign.class);

        if (subCampaigns.size() != 2 || dynamicCampaignId == null || smartCampaignId == null) {
            // Очень плохо, в еком-сценарии подкампаний всегда должно быть 2 — динамическая и смарt
            return MassResult.brokenMassAction(null,
                    ValidationResult.failed(null, new Defect<>(DefectIds.INCONSISTENT_STATE)));
        }

        var dynamicModelChanges = toCoreDynamicCampaignModelChanges(
                dynamicCampaignId, masterCid, input, client.getEmail(), nullIfEmpty(minusKeywords), converterContext,
                campaignConstantsService.getDefaultAttributionModel());
        var result = ucCampaignService.updateCampaign(dynamicModelChanges, operatorUid, uidAndClientId);
        logValidationResultErrors("error while updating dynamic subcampaign", result.getValidationResult());

        var counterId = getCounterIdForSmartCampaign(operatorUid, client.getClientId(), input);
        var smartModelChanges = toCoreSmartCampaignModelChanges(
                smartCampaignId,
                masterCid,
                input,
                client.getEmail(),
                nullIfEmpty(minusKeywords),
                counterId,
                converterContext,
                campaignConstantsService.getDefaultAttributionModel());
        result = ucCampaignService.updateCampaign(smartModelChanges, operatorUid, uidAndClientId);
        logValidationResultErrors("error while updating smart subcampaign", result.getValidationResult());

        var resultValue = List.of(dynamicCampaignId, smartCampaignId);
        return MassResult.successfulMassAction(resultValue, ValidationResult.success(resultValue));
    }

    private static <T extends BaseCampaign> Long onlyCampaignId(List<? extends BaseCampaign> campaigns,
                                                                Class<T> campaignClass) {
        return StreamEx.of(campaigns)
                .select(campaignClass)
                .findFirst()
                .map(BaseCampaign::getId)
                .orElse(null);
    }

    private Result<Long> updateGroup(ComplexTextAdGroup complexTextAdGroup, GeoTree geoTree,
                                     Long operatorUid, User client) {
        var autoPriceParams = new ShowConditionAutoPriceParams(ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null),
                recentStatisticsService);
        MassResult<Long> massResult = ucCampaignService.updateGroup(complexTextAdGroup,
                UPDATING_COMPLEX_AD_GROUP_PROPERTIES, geoTree, autoPriceParams,
                operatorUid, UidAndClientId.of(client.getUid(), client.getClientId()));
        logValidationResultErrors("error while updating groups", massResult.getValidationResult());
        return massResult.get(0);
    }

    @Nullable
    private List<Long> getGoalIds(GdUcCampaignMutationInput input) {
        if (input.getGoalId() == null) {
            return null;
        }
        if (input.getGoalId().equals(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)) {
            return mapList(input.getMeaningfulGoals(), GdMeaningfulGoalRequest::getGoalId);
        }
        return List.of(input.getGoalId());
    }

    private GdValidationResult buildGridValidationResult(ValidationResult<?, Defect> vr) {
        return buildGridValidationResultIfErrors(vr, emptyPath(), UcCampaignConverter.PATH_NODE_CONVERTER_PROVIDER);
    }

    private void logValidationResultErrors(String logPrefix, @Nullable ValidationResult<?, Defect> vr) {
        logValidationResultErrors(logPrefix, vr, false);
    }

    private void logValidationResultErrors(String logPrefix,
                                           @Nullable ValidationResult<?, Defect> vr,
                                           boolean isWarning) {
        if (vr != null && vr.hasAnyErrors()) {
            String vrErrors = vr.flattenErrors().stream()
                    .map(DefectInfo::toString)
                    .collect(Collectors.joining("; "));
            if (isWarning) {
                logger.warn("{}: {}", logPrefix, vrErrors);
            } else {
                logger.error("{}: {}", logPrefix, vrErrors);
            }
        }
    }


}
