package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexDynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithFeedId;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.AddComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.adgroup.service.complex.UpdateComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.AddComplexBannersSubOperation;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.AddUpdateRetargetingConditionSubOperation;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.UpdateComplexBannersSubOperation;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.converter.KeywordUpdateConverter;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupValidationService;
import ru.yandex.direct.core.entity.adgroup.service.validation.types.AdGroupWithFeedIdValidator;
import ru.yandex.direct.core.entity.adgroup.service.validation.types.DynamicFeedAdGroupValidation;
import ru.yandex.direct.core.entity.adgroup.service.validation.types.PerformanceAdGroupValidation;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.service.ComplexBidModifierService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService;
import ru.yandex.direct.core.entity.keyword.container.AdGroupInfoForKeywordAdd;
import ru.yandex.direct.core.entity.keyword.container.InternalKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordsAddValidationService;
import ru.yandex.direct.core.entity.keyword.service.validation.UpdateKeywordValidationService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionOperationFactory;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddUcCampaignInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationInput;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateUcCampaignInput;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateConverter.adGroupToModelChanges;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexDynamicAdGroup;
import static ru.yandex.direct.grid.processing.service.campaign.converter.UcCampaignConverter.toComplexPerformanceAdGroup;
import static ru.yandex.direct.libs.keywordutils.parser.KeywordParser.parseWithMinuses;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachInSetNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.util.ValidationUtils.modelChangesValidationToModelValidation;

@Service
@ParametersAreNonnullByDefault
public class UcCampaignValidationService {

    private final GridValidationService gridValidationService;
    private final HyperGeoService hyperGeoService;
    private final AdGroupValidationService adGroupValidationService;
    private final DynamicFeedAdGroupValidation dynamicFeedAdGroupValidationService;
    private final PerformanceAdGroupValidation performanceAdGroupValidationService;
    private final ComplexBidModifierService complexBidModifierService;
    private final BannersAddOperationFactory bannersAddOperationFactory;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final RetargetingConditionOperationFactory retargetingConditionOperationFactory;
    private final VcardService vcardService;
    private final SitelinkSetService sitelinkSetService;
    private final KeywordsAddValidationService keywordsAddValidationService;
    private final UpdateKeywordValidationService updateKeywordValidationService;
    private final ClientLimitsService clientLimitsService;
    private final AddComplexAdGroupValidationService addComplexAdGroupValidationService;
    private final UpdateComplexAdGroupValidationService updateComplexAdGroupValidationService;
    private final FeedService feedService;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final ClientRepository clientRepository;

    public UcCampaignValidationService(GridValidationService gridValidationService,
                                       HyperGeoService hyperGeoService,
                                       AdGroupValidationService adGroupValidationService,
                                       DynamicFeedAdGroupValidation dynamicFeedAdGroupValidationService,
                                       PerformanceAdGroupValidation performanceAdGroupValidationService,
                                       ComplexBidModifierService complexBidModifierService,
                                       BannersAddOperationFactory bannersAddOperationFactory,
                                       BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                       RetargetingConditionOperationFactory retargetingConditionOperationFactory,
                                       VcardService vcardService,
                                       SitelinkSetService sitelinkSetService,
                                       KeywordsAddValidationService keywordsAddValidationService,
                                       UpdateKeywordValidationService updateKeywordValidationService,
                                       ClientLimitsService clientLimitsService,
                                       AddComplexAdGroupValidationService addComplexAdGroupValidationService,
                                       UpdateComplexAdGroupValidationService updateComplexAdGroupValidationService,
                                       FeedService feedService,
                                       ShardHelper shardHelper,
                                       CampaignRepository campaignRepository,
                                       ClientRepository clientRepository) {
        this.gridValidationService = gridValidationService;
        this.hyperGeoService = hyperGeoService;
        this.adGroupValidationService = adGroupValidationService;
        this.dynamicFeedAdGroupValidationService = dynamicFeedAdGroupValidationService;
        this.performanceAdGroupValidationService = performanceAdGroupValidationService;
        this.complexBidModifierService = complexBidModifierService;
        this.bannersAddOperationFactory = bannersAddOperationFactory;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.retargetingConditionOperationFactory = retargetingConditionOperationFactory;
        this.vcardService = vcardService;
        this.sitelinkSetService = sitelinkSetService;
        this.keywordsAddValidationService = keywordsAddValidationService;
        this.updateKeywordValidationService = updateKeywordValidationService;
        this.clientLimitsService = clientLimitsService;
        this.addComplexAdGroupValidationService = addComplexAdGroupValidationService;
        this.updateComplexAdGroupValidationService = updateComplexAdGroupValidationService;
        this.feedService = feedService;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.clientRepository = clientRepository;
    }

    public void validateAddRequest(GdAddUcCampaignInput input) {
        gridValidationService.applyValidator(this::validateAddUcCampaign, input, false);
    }

    public void validateUpdateRequest(GdUpdateUcCampaignInput input) {
        gridValidationService.applyValidator(this::validateUpdateUcCampaign, input, false);
    }

    private ValidationResult<GdAddUcCampaignInput, Defect> validateAddUcCampaign(
            GdAddUcCampaignInput input) {
        var vb = ModelItemValidationBuilder.of(input);
        checkCampaignMutationInput(input, vb);
        return vb.getResult();
    }

    private ValidationResult<GdUpdateUcCampaignInput, Defect> validateUpdateUcCampaign(
            GdUpdateUcCampaignInput input) {
        var vb = ModelItemValidationBuilder.of(input);
        vb.item(GdUpdateUcCampaignInput.ID)
                .check(notNull())
                .check(validId());
        checkCampaignMutationInput(input, vb);
        return vb.getResult();
    }

    private <T extends GdUcCampaignMutationInput> void checkCampaignMutationInput(T input,
                                                                                  ModelItemValidationBuilder<T> vb) {
        vb.item(GdUcCampaignMutationInput.NAME)
                .check(notBlank());
        vb.item(GdUcCampaignMutationInput.HREF)
                .check(notBlank())
                .check(validHref());
        vb.item(GdUcCampaignMutationInput.BUSINESS_CATEGORY)
                .check(notBlank());
        vb.item(GdUcCampaignMutationInput.BUDGET)
                .check(notNull());
        vb.item(GdUcCampaignMutationInput.BUDGET_DISPLAY_FORMAT)
                .check(notNull());
        vb.item(GdUcCampaignMutationInput.AD_TITLE)
                .check(notBlank());
        vb.item(GdUcCampaignMutationInput.AD_TEXT)
                .check(notBlank());
        vb.item(GdUcCampaignMutationInput.PERMALINK_ID)
                .check(validId());
        vb.item(GdUcCampaignMutationInput.PHONE_ID)
                .check(validId());

        vb.list(GdUcCampaignMutationInput.SITELINKS)
                .checkEach(notNull());
        vb.item(GdUcCampaignMutationInput.DEVICE_TYPES)
                .check(eachInSetNotNull());

        boolean isHyperLocalAdGroup = input.getHyperGeoId() != null;
        vb.list(GdUcCampaignMutationInput.GEO)
                .check(notNull(), When.isFalse(isHyperLocalAdGroup))
                .check(notEmptyCollection(), When.isValidAnd(When.isFalse(isHyperLocalAdGroup)))
                .check(isNull(), When.isTrue(isHyperLocalAdGroup))
                .checkEach(notNull(), When.isValid())
                .checkEach(validId(), When.isValid());
        vb.item(GdUcCampaignMutationInput.HYPER_GEO_ID)
                .check(validId());

        boolean hasGoal = input.getGoalId() != null;
        vb.item(GdUcCampaignMutationInput.METRIKA_COUNTERS)
                .check(notNull(), When.isTrue(hasGoal))
                .check(notEmptyCollection(), When.isValidAnd(When.isTrue(hasGoal)));
    }

    /**
     * Пре-валидация комплексной группы еще до создания кампании и самой группы с баннерами
     */
    public ValidationResult<ComplexTextAdGroup, Defect> preValidateAddingComplexTextAdGroup(
            ClientId clientId, Long operatorUid,
            ComplexTextAdGroup complexAdGroup,
            GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo) {
        var isTextGroupWithoutSmartBanners = StreamEx.of(complexAdGroup.getComplexBanners())
                .map(ComplexBanner::getBanner)
                .select(PerformanceBanner.class)
                .findFirst()
                .isEmpty();
        AdGroup adGroup = complexAdGroup.getAdGroup();
        ValidationResult<AdGroup, Defect> adGroupVr =
                preValidateAddingAdGroup(adGroup, clientId, geoTree, multipleSegmentsInHyperGeo);
        ValidationResult<ComplexBidModifier, Defect> bidModifierVr =
                preValidateBidModifier(complexAdGroup.getComplexBidModifier(), adGroup, clientId);
        ValidationResult<List<Keyword>, Defect> keywordsVr =
                preValidateKeywords(clientId, adGroup, complexAdGroup.getKeywords(), List.of());

        ValidationResult<ComplexTextAdGroup, Defect> vr =
                ModelItemValidationBuilder.of(complexAdGroup).getResult()
                .addSubResult(field(ComplexTextAdGroup.AD_GROUP), adGroupVr)
                .addSubResult(field(ComplexTextAdGroup.COMPLEX_BID_MODIFIER), bidModifierVr)
                .addSubResult(field(ComplexTextAdGroup.KEYWORDS), keywordsVr);
        // В еком сценарии смарт баннеры создаются по дефолтным креативам и пользовательского ввода там вообще нет
        // Соответственно, валидировать нечего
        if (isTextGroupWithoutSmartBanners) {
            ValidationResult<List<BannerWithSystemFields>, Defect> bannersVr =
                    preValidateAddingBanners(adGroup, complexAdGroup.getComplexBanners(), clientId, operatorUid);
            vr.addSubResult(field(ComplexTextAdGroup.COMPLEX_BANNERS), bannersVr);
        }
        if (complexAdGroup.getRetargetingCondition() != null) {
            ValidationResult<List<RetargetingConditionBase>, Defect> retargetingConditionVr =
                    preValidateRetargetingCondition(complexAdGroup.getRetargetingCondition(), clientId);
            vr.addSubResult(field(ComplexTextAdGroup.RETARGETING_CONDITION), retargetingConditionVr);
        }
        return vr;
    }

    public ValidationResult<ComplexDynamicAdGroup, Defect> preValidateAddingComplexDynamicAdGroup(
            ClientId clientId, GdUcCampaignMutationInput input, GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo, boolean tabletBidModifierEnabled) {
        var complexAdGroup = toComplexDynamicAdGroup(input, null, tabletBidModifierEnabled);
        var adGroup = (DynamicFeedAdGroup) complexAdGroup.getAdGroup();
        ValidationResult<AdGroup, Defect> adGroupVr =
                preValidateAddingAdGroup(adGroup, clientId, geoTree, multipleSegmentsInHyperGeo);
        ValidationResult<AdGroup, Defect> dynamicAdGroupVr = dynamicFeedAdGroupValidationService
                .validateAdGroups(clientId, List.of(adGroup))
                .getOrCreateSubValidationResult(index(0), adGroup);
        ValidationResult<ComplexBidModifier, Defect> bidModifierVr =
                preValidateBidModifier(complexAdGroup.getComplexBidModifier(), adGroup, clientId);
        ValidationResult<List<AdGroupWithFeedId>, Defect> feedsVr = validateAdGroupWithFeed(clientId, adGroup);
        adGroupVr.addSubResult(field(DynamicFeedAdGroup.FEED_ID), feedsVr);

        return ModelItemValidationBuilder.of(complexAdGroup).getResult()
                .addSubResult(field(ComplexDynamicAdGroup.AD_GROUP), adGroupVr.merge(dynamicAdGroupVr))
                .addSubResult(field(ComplexDynamicAdGroup.COMPLEX_BID_MODIFIER), bidModifierVr);
    }

    public ValidationResult<ComplexPerformanceAdGroup, Defect> preValidateAddingComplexPerformanceAdGroup(
            ClientId clientId, GdUcCampaignMutationInput input, GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo, boolean tabletBidModifierEnabled) {
        var complexAdGroup = toComplexPerformanceAdGroup(input, null, tabletBidModifierEnabled);
        var adGroup = (PerformanceAdGroup) complexAdGroup.getAdGroup();
        ValidationResult<AdGroup, Defect> adGroupVr =
                preValidateAddingAdGroup(adGroup, clientId, geoTree, multipleSegmentsInHyperGeo);
        ValidationResult<AdGroup, Defect> performanceAdGroupVr = performanceAdGroupValidationService
                .validateAdGroups(clientId, List.of(adGroup))
                .getOrCreateSubValidationResult(index(0), adGroup);
        ValidationResult<ComplexBidModifier, Defect> bidModifierVr =
                preValidateBidModifier(complexAdGroup.getComplexBidModifier(), adGroup, clientId);
        ValidationResult<List<AdGroupWithFeedId>, Defect> feedsVr = validateAdGroupWithFeed(clientId, adGroup);
        adGroupVr.addSubResult(field(PerformanceAdGroup.FEED_ID), feedsVr);

        return ModelItemValidationBuilder.of(complexAdGroup).getResult()
                .addSubResult(field(ComplexPerformanceAdGroup.AD_GROUP), adGroupVr.merge(performanceAdGroupVr))
                .addSubResult(field(ComplexPerformanceAdGroup.COMPLEX_BID_MODIFIER), bidModifierVr);
    }

    private ValidationResult<List<AdGroupWithFeedId>, Defect> validateAdGroupWithFeed(ClientId clientId, AdGroupWithFeedId adGroup) {
        var adGroups = List.of(adGroup);
        AdGroupWithFeedIdValidator<AdGroupWithFeedId> feedIdValidator =
                new AdGroupWithFeedIdValidator.Builder<>(feedService, clientId, adGroups)
                        .build();
        ListValidationBuilder<AdGroupWithFeedId, Defect> vb = ListValidationBuilder.of(adGroups);
        vb.checkEachBy(feedIdValidator);
        return vb.getResult();
    }

    public ValidationResult<AdGroup, Defect> preValidateAddingAdGroup(
            AdGroup adGroup, ClientId clientId, GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo) {
        Map<Long, HyperGeo> existingHyperGeoByIds = adGroup.getHyperGeoId() != null ?
                hyperGeoService.getHyperGeoById(clientId, List.of(adGroup.getHyperGeoId())) :
                emptyMap();
        BiFunction<AdGroup, ModelProperty, Boolean> isPropertyChanged = (ag, mp) -> true;

        return adGroupValidationService.validateAdGroup(adGroup, geoTree, isPropertyChanged,
                emptyMap(), emptySet(), existingHyperGeoByIds, true, multipleSegmentsInHyperGeo);
    }

    private ValidationResult<List<BannerWithSystemFields>, Defect> preValidateAddingBanners(
            AdGroup adGroup, List<ComplexBanner> complexBanners, ClientId clientId, Long operatorUid) {
        var banners = mapList(complexBanners, ComplexBanner::getBanner);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        ValidationResult<List<BannerWithSystemFields>, Defect> vr =
                addComplexAdGroupValidationService.validateComplexBanners(adGroup, complexBanners, banners,
                        Language.UNKNOWN, clientId, clientRegionId);
        if (vr.hasAnyErrors()) {
            return vr;
        }
        return new AddComplexBannersSubOperation(complexBanners, bannersAddOperationFactory,
                vcardService, sitelinkSetService, operatorUid, clientId, false, true, DatabaseMode.ONLY_MYSQL)
                .prepare();
    }

    /**
     * Пре-валидация комплексной группы еще до изменения кампании и самой группы с баннерами
     */
    public ValidationResult<ComplexTextAdGroup, Defect> preValidateUpdatingComplexAdGroup(
            ClientId clientId, Long operatorUid,
            ComplexTextAdGroup complexAdGroup, GeoTree geoTree,
            List<Keyword> existingKeywords,
            boolean multipleSegmentsInHyperGeo) {
        var isTextGroupWithoutSmartBanners = StreamEx.of(complexAdGroup.getComplexBanners())
                .map(ComplexBanner::getBanner)
                .select(PerformanceBanner.class)
                .findFirst()
                .isEmpty();
        AdGroup adGroup = complexAdGroup.getAdGroup();
        ValidationResult<AdGroup, Defect> adGroupVr =
                preValidateUpdatingAdGroup(adGroup, clientId, geoTree, multipleSegmentsInHyperGeo);
        ValidationResult<ComplexBidModifier, Defect> bidModifierVr =
                preValidateBidModifier(complexAdGroup.getComplexBidModifier(), adGroup, clientId);
        ValidationResult<List<Keyword>, Defect> keywordsVr =
                preValidateKeywords(clientId, adGroup, complexAdGroup.getKeywords(), existingKeywords);

        ValidationResult<ComplexTextAdGroup, Defect> vr =
                ModelItemValidationBuilder.of(complexAdGroup).getResult()
                .addSubResult(field(ComplexTextAdGroup.AD_GROUP), adGroupVr)
                .addSubResult(field(ComplexTextAdGroup.COMPLEX_BID_MODIFIER), bidModifierVr)
                .addSubResult(field(ComplexTextAdGroup.KEYWORDS), keywordsVr);

        // В еком сценарии смарт баннеры создаются по дефолтным креативам и пользовательского ввода там вообще нет
        // Соответственно, валидировать нечего
        if (isTextGroupWithoutSmartBanners) {
            ValidationResult<List<BannerWithSystemFields>, Defect> bannersVr =
                    preValidateUpdatingBanners(adGroup, complexAdGroup.getComplexBanners(), clientId, operatorUid);
            vr.addSubResult(field(ComplexTextAdGroup.COMPLEX_BANNERS), bannersVr);
        }

        if (complexAdGroup.getRetargetingCondition() != null) {
            ValidationResult<List<RetargetingConditionBase>, Defect> retargetingConditionVr =
                    preValidateRetargetingCondition(complexAdGroup.getRetargetingCondition(), clientId);
            vr.addSubResult(field(ComplexTextAdGroup.RETARGETING_CONDITION), retargetingConditionVr);
        }

        return vr;
    }

    public ValidationResult<ComplexDynamicAdGroup, Defect> preValidateUpdatingComplexDynamicAdGroup(
            ClientId clientId, ComplexDynamicAdGroup complexAdGroup, GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo
    ) {
        var adGroup = (DynamicFeedAdGroup) complexAdGroup.getAdGroup();
        ValidationResult<AdGroup, Defect> adGroupVr =
                preValidateUpdatingAdGroup(adGroup, clientId, geoTree, multipleSegmentsInHyperGeo);
        ValidationResult<AdGroup, Defect> dynamicAdGroupVr = dynamicFeedAdGroupValidationService
                .validateAdGroups(clientId, List.of(adGroup))
                .getOrCreateSubValidationResult(index(0), adGroup);
        ValidationResult<ComplexBidModifier, Defect> bidModifierVr =
                preValidateBidModifier(complexAdGroup.getComplexBidModifier(), adGroup, clientId);
        ValidationResult<List<AdGroupWithFeedId>, Defect> feedsVr = validateAdGroupWithFeed(clientId, adGroup);
        adGroupVr.addSubResult(field(DynamicFeedAdGroup.FEED_ID), feedsVr);

        return ModelItemValidationBuilder.of(complexAdGroup).getResult()
                        .addSubResult(field(ComplexDynamicAdGroup.AD_GROUP), adGroupVr.merge(dynamicAdGroupVr))
                        .addSubResult(field(ComplexDynamicAdGroup.COMPLEX_BID_MODIFIER), bidModifierVr);
    }

    public ValidationResult<ComplexPerformanceAdGroup, Defect> preValidateUpdatingComplexPerformanceAdGroup(
            ClientId clientId, ComplexPerformanceAdGroup complexAdGroup, GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo
    ) {
        var adGroup = (PerformanceAdGroup) complexAdGroup.getAdGroup();
        ValidationResult<AdGroup, Defect> adGroupVr =
                preValidateUpdatingAdGroup(adGroup, clientId, geoTree, multipleSegmentsInHyperGeo);
        ValidationResult<AdGroup, Defect> performanceAdGroupVr = performanceAdGroupValidationService
                .validateAdGroups(clientId, List.of(adGroup))
                .getOrCreateSubValidationResult(index(0), adGroup);
        ValidationResult<ComplexBidModifier, Defect> bidModifierVr =
                preValidateBidModifier(complexAdGroup.getComplexBidModifier(), adGroup, clientId);
        ValidationResult<List<AdGroupWithFeedId>, Defect> feedsVr = validateAdGroupWithFeed(clientId, adGroup);
        adGroupVr.addSubResult(field(DynamicFeedAdGroup.FEED_ID), feedsVr);

        return ModelItemValidationBuilder.of(complexAdGroup).getResult()
                .addSubResult(field(ComplexDynamicAdGroup.AD_GROUP), adGroupVr.merge(performanceAdGroupVr))
                .addSubResult(field(ComplexDynamicAdGroup.COMPLEX_BID_MODIFIER), bidModifierVr);
    }

    private ValidationResult<AdGroup, Defect> preValidateUpdatingAdGroup(
            AdGroup adGroup, ClientId clientId, GeoTree geoTree,
            boolean multipleSegmentsInHyperGeo) {
        Map<Long, HyperGeo> existingHyperGeoByIds = adGroup.getHyperGeoId() != null ?
                hyperGeoService.getHyperGeoById(clientId, List.of(adGroup.getHyperGeoId())) :
                emptyMap();

        ModelChanges<AdGroup> modelChanges = adGroupToModelChanges(adGroup);
        BiFunction<AdGroup, ModelProperty, Boolean> isPropertyChanged =
                (group, modelProperty) -> modelChanges.isPropChanged(modelProperty);

        return adGroupValidationService.validateAdGroup(adGroup, geoTree, isPropertyChanged,
                emptyMap(), emptySet(), existingHyperGeoByIds, true, multipleSegmentsInHyperGeo);
    }

    private ValidationResult<List<BannerWithSystemFields>, Defect> preValidateUpdatingBanners(
            AdGroup adGroup, List<ComplexBanner> complexBanners, ClientId clientId, Long operatorUid) {
        var banners = mapList(complexBanners, ComplexBanner::getBanner);
        var existingBannerIds = filterAndMapList(banners, t -> t.getId() != null, BannerWithSystemFields::getId);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, Language> adGroupIdToCampaignLanguageMap =
                campaignRepository.getCampaignsLangByAdGroupIds(shard, clientId, List.of(adGroup.getId()));
        ImmutableMultimap<Long, Long> existingBannerIdsByAdGroupIds =
                ImmutableMultimap.<Long, Long>builder().putAll(adGroup.getId(), existingBannerIds).build();

        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        ValidationResult<List<BannerWithSystemFields>, Defect> vr =
                updateComplexAdGroupValidationService.validateComplexBanners(adGroup, complexBanners, banners,
                        adGroupIdToCampaignLanguageMap, existingBannerIdsByAdGroupIds, clientId, clientRegionId);
        if (vr.hasAnyErrors()) {
            return vr;
        }
        return new UpdateComplexBannersSubOperation(ModerationMode.FORCE_MODERATE, complexBanners, bannersAddOperationFactory,
                bannersUpdateOperationFactory, vcardService, sitelinkSetService, operatorUid, clientId, true)
                .prepare();
    }

    private ValidationResult<ComplexBidModifier, Defect> preValidateBidModifier(
            ComplexBidModifier bidModifier, AdGroup adGroup, ClientId clientId) {
        Pair<List<BidModifier>, Multimap<Integer, Integer>> flatModifiersInfo =
                complexBidModifierService.convertFromComplexModelsForAdGroups(List.of(bidModifier));
        List<BidModifier> bidModifiers = flatModifiersInfo.getLeft();
        ValidationResult<List<BidModifier>, Defect> flatVr =
                complexBidModifierService.validateBidModifiersFlat(bidModifiers, flatModifiersInfo.getRight(),
                        flatIndex -> CampaignType.TEXT, flatIndex -> adGroup, clientId);
        ValidationResult<List<ComplexBidModifier>, Defect> vr =
                new ValidationResult<>(singletonList(bidModifier));
        complexBidModifierService.transferValidationResultFlatToComplex(vr, flatVr, flatModifiersInfo.getRight());
        return vr.getOrCreateSubValidationResult(index(0), bidModifier);
    }

    public ValidationResult<List<Keyword>, Defect> preValidateKeywords(
            ClientId clientId, AdGroup adGroup, List<Keyword> keywords, List<Keyword> existingKeywords
    ) {
        final var keywordToIndex = IntStreamEx.range(keywords.size())
                .boxed()
                .collect(Collectors.toMap(keywords::get, Function.identity()));

        Map<Boolean, List<Keyword>> keywordsForAddOrUpdate = StreamEx.of(keywords)
                .partitioningBy(k -> k.getId() == null);

        List<Keyword> keywordsForAdd = keywordsForAddOrUpdate.get(true);
        if (!keywordsForAdd.isEmpty()) {
            ValidationResult<List<Keyword>, Defect> vr = keywordsAddValidationService
                    .preValidate(keywordsForAdd, clientId);

            if (vr != null && vr.hasAnyErrors()) {
                final var currentIndexToSourceIndex = IntStreamEx.range(keywordsForAdd.size())
                        .boxed()
                        .toMap(Function.identity(), i -> keywordToIndex.get(keywordsForAdd.get(i)));

                return (ValidationResult<List<Keyword>, Defect>) vr
                        .transform(new ValidationResultPathByIndexTransformer(currentIndexToSourceIndex));
            }
        }

        List<Keyword> keywordsForUpdate = keywordsForAddOrUpdate.get(false);
        if (!keywordsForUpdate.isEmpty()) {
            List<ModelChanges<Keyword>> modelChanges = mapList(
                    keywordsForUpdate, KeywordUpdateConverter::keywordToModelChanges
            );

            Set<Long> existingKeywordIds = listToSet(existingKeywords, Keyword::getId);
            ValidationResult<List<ModelChanges<Keyword>>, Defect> vr = updateKeywordValidationService
                    .preValidate(modelChanges, existingKeywordIds, Set.of());

            if (vr != null && vr.hasAnyErrors()) {
                final var currentIndexToSourceIndex = IntStreamEx.range(keywordsForUpdate.size())
                        .boxed()
                        .toMap(Function.identity(), i -> keywordToIndex.get(keywordsForUpdate.get(i)));

                return (ValidationResult<List<Keyword>, Defect>) modelChangesValidationToModelValidation(
                        vr, emptyList(), id -> new Keyword().withId(id)
                ).transform(new ValidationResultPathByIndexTransformer(currentIndexToSourceIndex));
            }
        }

        var internalKeywords = IntStreamEx.range(0, keywords.size())
                .mapToEntry(i -> i, keywords::get)
                .mapValues(k -> new InternalKeyword(k, parseWithMinuses(k.getPhrase())))
                .toMap();
        var adGroupInfo = IntStreamEx.range(0, keywords.size())
                .mapToEntry(i -> i, keywords::get)
                .mapToValue((i, k) -> new AdGroupInfoForKeywordAdd(0, -1L, adGroup.getType()))
                .toMap();
        ValidationResult<List<Keyword>, Defect> vr = ValidationResult.success(keywords);
        keywordsAddValidationService
                .validateWithNonexistentAdGroups(vr, adGroupInfo, emptyMap(), internalKeywords, false);

        Long keywordsLimit = clientLimitsService.massGetClientLimits(singletonList(clientId))
                .iterator().next()
                .getKeywordsCountLimitOrDefault();
        keywordsAddValidationService.validateSizeWithNonexistentAdGroups(vr, adGroupInfo, keywordsLimit);
        return vr;
    }

    private ValidationResult<List<RetargetingConditionBase>, Defect> preValidateRetargetingCondition(
            RetargetingConditionBase retargetingCondition, ClientId clientId) {
        return new AddUpdateRetargetingConditionSubOperation(List.of(retargetingCondition),
                retargetingConditionOperationFactory, clientId).prepare();
    }
}
