package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.moderationdiag.model.DiagReasonTitleAndDescription;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonKey;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.campaign.GdUcDiagModReason;
import ru.yandex.direct.grid.processing.model.campaign.GdUcDiagModReasonWithDescription;
import ru.yandex.direct.grid.processing.model.campaign.GdUcDiagModReasonWithToken;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.grid.processing.service.campaign.uc.GdUcModReasonsService.UC_MOD_REASON_TOKEN_PLACEHOLDER;
import static ru.yandex.direct.grid.processing.service.campaign.uc.GdUcModReasonsService.isReasonWithToken;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Получение причин отклонения в uc-формате по разным типам объектов
 */
@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class UcModReasonDataLoader extends GridBatchingDataLoader<UcModerationReasonRequestItem, List<GdUcDiagModReason>> {
    private static final Logger logger = LoggerFactory.getLogger(UcModReasonDataLoader.class);

    private final GdUcModReasonsService gdUcModReasonsService;

    public UcModReasonDataLoader(GridContextProvider gridContextProvider,
                                 GdUcModReasonsService gdUcModReasonsService) {
        this.dataLoader = mappedDataLoader(gridContextProvider,
                getBatchLoadFunction(() -> retrieveClientId(gridContextProvider)));
        this.gdUcModReasonsService = gdUcModReasonsService;
    }

    private MappedBatchLoaderWithContext<UcModerationReasonRequestItem, List<GdUcDiagModReason>> getBatchLoadFunction(
            Supplier<ClientId> clientIdSupplier) {
        return (requestItems, environment) -> {
            if (CollectionUtils.isEmpty(requestItems)) {
                return CompletableFuture.completedFuture(Map.<UcModerationReasonRequestItem, List<GdUcDiagModReason>>of());
            }
            ClientId clientId = clientIdSupplier.get();
            if (clientId == null) {
                logger.error("clientId cannot be null");
                return CompletableFuture.completedFuture(Map.<UcModerationReasonRequestItem, List<GdUcDiagModReason>>of());
            }

            Set<GdLanguage> gdLangSet = StreamEx.of(requestItems)
                    .map(UcModerationReasonRequestItem::getGdLanguage)
                    .nonNull()
                    .toSet();

            Map<UcModerationReasonRequestItem, List<GdUcDiagModReason>> result = new HashMap<>();
            for (GdLanguage lang: gdLangSet) {
                Set<ModerationReasonKey> modReasonKeys = StreamEx.of(requestItems)
                        .filter(t -> lang.equals(t.getGdLanguage()))
                        .map(t -> new ModerationReasonKey()
                                .withObjectId(t.getObjectId())
                                .withObjectType(t.getObjectType())
                                .withSubObjectId(t.getSubObjectId())
                        )
                        .toSet();
                result.putAll(getReasonsForGivenLanguage(modReasonKeys, clientId, lang));
            }
            return CompletableFuture.completedFuture(result);
        };
    }


    /**
     * Получаем причины отклонения в нужном для отдачи лоадером формате на переданном языке
     */
    private Map<UcModerationReasonRequestItem, List<GdUcDiagModReason>> getReasonsForGivenLanguage(
            Set<ModerationReasonKey> modReasonKeys,
            ClientId clientId,
            GdLanguage lang) {

        Map<UcModerationReasonRequestItem, List<ModerationDiag>> diagsByKey = getDiagsByKey(
                gdUcModReasonsService.getRejectReasonDiags(clientId, modReasonKeys), lang
        );

        //Обогащаем причины из базы данными по тултипам от Модерации и переводами
        Map<Long, ModerationDiag> diagsToDetail = StreamEx.of(diagsByKey.values())
                .flatMap(StreamEx::of)
                .mapToEntry(ModerationDiag::getId, Function.identity())
                .toMap((a, b) -> a);
        Map<Long, DiagReasonTitleAndDescription> titleAndDescByDiagId = gdUcModReasonsService
                .getTitleAndDescByDiagId(diagsToDetail, lang);

        return EntryStream.of(diagsByKey).mapValues(t -> convert(t, titleAndDescByDiagId)).toMap();
    }

    private Map<UcModerationReasonRequestItem, List<ModerationDiag>> getDiagsByKey(
            Map<ModerationReasonKey, List<ModerationDiag>> rejectReasonDiagsByKey,
            GdLanguage lang) {
        return EntryStream.of(rejectReasonDiagsByKey)
                .mapKeys(t -> new UcModerationReasonRequestItem(t.getObjectId(),
                        t.getSubObjectId(), t.getObjectType(), lang))
                .toMap();
    }

    private static List<GdUcDiagModReason> convert(List<ModerationDiag> diag,
                                                   Map<Long, DiagReasonTitleAndDescription> titleAndDescByDiagId) {
        return mapList(diag, t -> convert(t, titleAndDescByDiagId));
    }

    private static GdUcDiagModReason convert(ModerationDiag diag,
                                             Map<Long, DiagReasonTitleAndDescription> titleAndDescByDiagId) {
        if (titleAndDescByDiagId.containsKey(diag.getId())) {
            return convertCommonFields(diag, GdUcDiagModReasonWithDescription::new)
                    .withDescription(titleAndDescByDiagId.get(diag.getId()).getDescription());
        }
        if (isReasonWithToken(diag)) {
            return convertCommonFields(diag, GdUcDiagModReasonWithToken::new)
                    .withToken(GdUcModReasonsService.convertToken(diag.getToken()));
        }
        return convertCommonFields(diag, GdUcDiagModReasonWithToken::new)
                .withToken(UC_MOD_REASON_TOKEN_PLACEHOLDER);
    }

    private static <T extends GdUcDiagModReason> T convertCommonFields(ModerationDiag diag,
                                                                       Supplier<T> newInstanceProvider) {
        T result = newInstanceProvider.get();
        result
                .withDiagId(diag.getId())
                .withDiagText(diag.getDiagText())
                .withShortText(diag.getShortText())
                .withAllowFirstAid(diag.getAllowFirstAid())
                .withBadReason(diag.getStrongReason())
                .withUnbanIsProhibited(diag.getUnbanIsProhibited())
                .withShowDetailsUrl(ModerationDiagService.needShowDetailsUrl(diag));
        return result;
    }

    //Дальше идёт костыль, так как clientId  получают разными способами.
    // Других пока нет, так что текущий костыль должен быть исчерпывающим
    private static ClientId retrieveClientId(GridContextProvider gridContextProvider) {
        ClientId clientId = ClientId.fromNullableLong(
                ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                        GridGraphQLContext::getQueriedClient),
                        GdClientInfo::getId));
        if (clientId == null) {
            clientId = ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                    GridGraphQLContext::getSubjectUser),
                    User::getClientId);
        }
        return clientId;
    }
}
