package ru.yandex.direct.grid.processing.service.campaign.uc;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.region.RegionDesc;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class UcRegionDescDataLoader extends GridBatchingDataLoader<Long, RegionDesc> {
    private static final Logger logger = LoggerFactory.getLogger(UcRegionDescDataLoader.class);

    public UcRegionDescDataLoader(GridContextProvider gridContextProvider,
                                  ClientGeoService clientGeoService) {
        this.dataLoader = mappedDataLoader(gridContextProvider,
                getBatchLoadFunction(clientGeoService, () -> retrieveOperatorClientId(gridContextProvider)));
    }

    private MappedBatchLoaderWithContext<Long, RegionDesc> getBatchLoadFunction(
            ClientGeoService clientGeoService, Supplier<ClientId> clientIdSupplier) {
        return (regionIds, environment) -> {
            if (CollectionUtils.isEmpty(regionIds)) {
                return CompletableFuture.completedFuture(Map.<Long, RegionDesc>of());
            }
            ClientId clientId = clientIdSupplier.get();
            if (clientId == null) {
                logger.error("clientId cannot be null");
                return CompletableFuture.completedFuture(Map.<Long, RegionDesc>of());
            }
            GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
            return CompletableFuture.completedFuture(
                    StreamEx.of(regionIds)
                            .mapToEntry(geoTree::getRegion)
                            .nonNullValues()
                            .mapValues(RegionDesc::fromRegion)
                            .toMap()
            );
        };
    }

    private static ClientId retrieveOperatorClientId(GridContextProvider gridContextProvider) {
        return ifNotNull(ifNotNull(
                gridContextProvider.getGridContext(), GridGraphQLContext::getOperator),
                User::getClientId);
    }
}
