package ru.yandex.direct.grid.processing.service.cashback;

import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackCardsInfo;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackCardsInfoInput;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackInfo;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackRewardsDetails;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackRewardsDetailsInput;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class CashbackGraphQLService {
    public static final String CASHBACK_INFO_RESOLVER_NAME = "cashbackInfo";
    public static final String CASHBACK_REWARDS_DETAILS_RESOLVER_NAME = "cashbackRewardsDetails";
    public static final String CASHBACK_CARDS_INFO_RESOLVER_NAME = "cashbackCardsInfo";

    private final CashbackInfoService cashbackInfoService;
    private final GridValidationService gridValidationService;

    @Autowired
    public CashbackGraphQLService(CashbackInfoService cashbackInfoService, GridValidationService gridValidationService) {
        this.cashbackInfoService = cashbackInfoService;
        this.gridValidationService = gridValidationService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CASHBACK_INFO_RESOLVER_NAME)
    public GdCashbackInfo getCashbackInfo(@GraphQLContext GdClient client) {
        var clientId = client.getInfo().getId();
        return cashbackInfoService.getCashbackInfo(ClientId.fromLong(clientId));
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CASHBACK_REWARDS_DETAILS_RESOLVER_NAME)
    public GdCashbackRewardsDetails getCashbackRewardsDetails(
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCashbackRewardsDetailsInput input) {
        gridValidationService.validateCashbackRewardsDetailsRequest(input);
        var clientId = client.getInfo().getId();
        var period = input.getPeriod();
        var language = Optional.ofNullable(input.getLanguage()).orElse(GdLanguage.RU);
        return cashbackInfoService.getCashbackRewardsDetails(ClientId.fromLong(clientId), period, language);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = CASHBACK_CARDS_INFO_RESOLVER_NAME)
    public GdCashbackCardsInfo getCashbackCardsInfo(
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCashbackCardsInfoInput input) {
        return cashbackInfoService.getCashbackCardsInfo(
                ClientId.fromLong(client.getInfo().getId()), input.getLanguage().toString());
    }
}
