package ru.yandex.direct.grid.processing.service.cashback;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.cashback.model.CashbackCardCategoryInternal;
import ru.yandex.direct.core.entity.cashback.model.CashbackCardProgramInternal;
import ru.yandex.direct.core.entity.cashback.model.CashbackCardsProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackClientInfo;
import ru.yandex.direct.core.entity.cashback.model.CashbackClientProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramDetails;
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardsDetails;
import ru.yandex.direct.core.entity.cashback.repository.CashbackCategoriesRepository;
import ru.yandex.direct.core.entity.cashback.repository.CashbackProgramsCategoriesRepository;
import ru.yandex.direct.core.entity.cashback.repository.CashbackProgramsRepository;
import ru.yandex.direct.core.entity.cashback.service.CashbackClientsService;
import ru.yandex.direct.core.entity.cashback.service.CashbackProgramsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackCardsInfo;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackCategoryInfo;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackInfo;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackProgram;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackProgramDetails;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackProgramInfo;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackRewardsDetails;
import ru.yandex.direct.grid.processing.model.cashback.GdCashbackTooltipInfo;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CashbackInfoService {

    private final CashbackClientsService cashbackClientsService;
    private final ClientService clientService;
    private final CashbackProgramsCategoriesRepository cashbackProgramsCategoriesRepository;
    private final CashbackProgramsRepository cashbackProgramsRepository;
    private final CashbackCategoriesRepository cashbackCategoriesRepository;
    private final CashbackProgramsService cashbackProgramsService;

    @Autowired
    public CashbackInfoService(CashbackClientsService cashbackClientsService,
                               ClientService clientService,
                               CashbackProgramsCategoriesRepository cashbackProgramsCategoriesRepository,
                               CashbackProgramsRepository cashbackProgramsRepository,
                               CashbackCategoriesRepository cashbackCategoriesRepository,
                               CashbackProgramsService cashbackProgramsService) {
        this.cashbackClientsService = cashbackClientsService;
        this.clientService = clientService;
        this.cashbackProgramsCategoriesRepository = cashbackProgramsCategoriesRepository;
        this.cashbackProgramsRepository = cashbackProgramsRepository;
        this.cashbackCategoriesRepository = cashbackCategoriesRepository;
        this.cashbackProgramsService = cashbackProgramsService;
    }

    public GdCashbackInfo getCashbackInfo(ClientId clientId) {
        var currency = clientService.getWorkCurrency(clientId);
        return toGdCashbackInfo(cashbackClientsService.getClientCashbackInfo(clientId), currency);
    }

    public GdCashbackRewardsDetails getCashbackRewardsDetails(ClientId clientId, int period, GdLanguage language) {
        return toGdCashbackRewardsDetails(cashbackClientsService.getClientCashbackRewardsDetails(clientId, period),
                language);
    }

    public GdCashbackCardsInfo getCashbackCardsInfo(ClientId clientId, String language) {
        var programIds = StreamEx.of(cashbackProgramsService.getClientPrograms(clientId))
                .map(CashbackProgram::getId).toList();
        var categoriesByPrograms = cashbackProgramsCategoriesRepository.getCategoriesByPrograms(programIds);
        var dbPrograms = cashbackProgramsRepository.getCardsProgramsByIds(programIds);
        Map<Long, CashbackCardCategoryInternal> categories = new HashMap<>();
        for (var programEntity : categoriesByPrograms.entrySet()) {
            for (var categoryLink : programEntity.getValue()) {
                if (!categories.containsKey(categoryLink.getCategoryId())) {
                    categories.put(categoryLink.getCategoryId(),
                            fetchCashbackCategory(categoryLink.getCategoryId(), language));
                }
                categories.get(categoryLink.getCategoryId()).addProgram(new CashbackCardProgramInternal(
                        dbPrograms.get(categoryLink.getProgramId()), categoryLink.getOrder(), language));
            }
        }
        var maxCategoryPercent = BigDecimal.ZERO;
        for (var category : categories.values()) {
            category.prepareCategory();
            maxCategoryPercent = maxCategoryPercent.max(category.getMaxPercent());
        }
        return new GdCashbackCardsInfo()
                .withMaxCategoryPercent(toIntegerPercent(maxCategoryPercent))
                .withMaxCategoryPercentDecimal(maxCategoryPercent)
                .withCategories(StreamEx.of(categories.values())
                        .map(CashbackInfoService::toGdCashbackCategoryInfo)
                        .sorted(Comparator.comparing(GdCashbackCategoryInfo::getIsGeneral)
                                .thenComparing(GdCashbackCategoryInfo::getIsNew, Comparator.reverseOrder()))
                        .toList());
    }

    private CashbackCardCategoryInternal fetchCashbackCategory(Long categoryId, String language) {
        return new CashbackCardCategoryInternal(cashbackCategoriesRepository.get(categoryId), language);
    }

    private static GdCashbackProgramInfo toGdCashbackProgramInfo(CashbackCardProgramInternal programInternal) {
        var isHaveTooltip = Objects.nonNull(programInternal.getTooltipInfo()) ||
                Objects.nonNull(programInternal.getTooltipLink()) ||
                Objects.nonNull(programInternal.getTooltipLinkText());
        return new GdCashbackProgramInfo()
                .withProgramId(programInternal.getProgramId())
                .withName(programInternal.getName())
                .withPercent(toIntegerPercent(programInternal.getPercent()))
                .withPercentDecimal(programInternal.getPercent())
                .withOrder(programInternal.getOrder())
                .withTooltip(isHaveTooltip ? new GdCashbackTooltipInfo()
                        .withInfo(programInternal.getTooltipInfo())
                        .withLinkText(programInternal.getTooltipLinkText())
                        .withLink(programInternal.getTooltipLink()) : null);

    }

    private static GdCashbackCategoryInfo toGdCashbackCategoryInfo(CashbackCardCategoryInternal categoryInternal) {
        return new GdCashbackCategoryInfo()
                .withCategoryId(categoryInternal.getId())
                .withName(categoryInternal.getName())
                .withDescription(categoryInternal.getDescription())
                .withMaxPercent(toIntegerPercent(categoryInternal.getMaxPercent()))
                .withMaxPercentDecimal(categoryInternal.getMaxPercent())
                .withIsGeneral(categoryInternal.isGeneral())
                .withIsNew(categoryInternal.isNew())
                .withIsTechnical(categoryInternal.isTechnical())
                .withButtonText(categoryInternal.getButtonText())
                .withButtonLink(categoryInternal.getButtonLink())
                .withPrograms(StreamEx.of(
                        categoryInternal.getPrograms()).map(CashbackInfoService::toGdCashbackProgramInfo).toList());
    }

    private static GdCashbackRewardsDetails toGdCashbackRewardsDetails(CashbackRewardsDetails coreDetails,
                                                                       GdLanguage language) {
        List<GdCashbackProgramDetails> gdPrograms = new ArrayList<>();
        for (var program : coreDetails.getTotalByPrograms()) {
            gdPrograms.add(toGdCashbackProgramDetails(program, language));
        }
        return new GdCashbackRewardsDetails()
                .withTotalCashback(coreDetails.getTotalCashback())
                .withTotalCashbackWithoutNds(coreDetails.getTotalCashbackWithoutNds())
                .withTotalByPrograms(gdPrograms);
    }

    private static GdCashbackProgramDetails toGdCashbackProgramDetails(CashbackProgramDetails coreDetails,
                                                                       GdLanguage language) {
        var programInfo = Objects.nonNull(coreDetails.getProgram()) ?
                toGdCashbackProgramInfo(coreDetails.getProgram(), language) : null;
        return new GdCashbackProgramDetails()
                .withProgramId(coreDetails.getProgramId())
                .withDate(coreDetails.getDate())
                .withReward(coreDetails.getReward())
                .withRewardWithoutNds(coreDetails.getRewardWithoutNds())
                .withProgram(programInfo);
    }

    private static GdCashbackInfo toGdCashbackInfo(CashbackClientInfo coreInfo, Currency currency) {
        return new GdCashbackInfo()
                .withCashbacksEnabled(coreInfo.getCashbacksEnabled())
                .withCurrency(currency.getCode())
                .withTotalCashback(coreInfo.getTotalCashback())
                .withTotalCashbackWithoutNds(coreInfo.getTotalCashbackWithoutNds())
                .withAwaitingCashback(coreInfo.getAwaitingCashback())
                .withAwaitingCashbackWithoutNds(coreInfo.getAwaitingCashbackWithoutNds())
                .withLastMonthCashback(coreInfo.getLastMonthCashback())
                .withLastMonthCashbackWithoutNds(coreInfo.getLastMonthCashbackWithoutNds())
                .withPrograms(mapList(coreInfo.getPrograms(), CashbackInfoService::toGdCashbackProgram));
    }

    private static GdCashbackProgram toGdCashbackProgram(CashbackClientProgram coreProgram) {
        return new GdCashbackProgram()
                .withProgramId(coreProgram.getProgramId())
                .withName(coreProgram.getName())
                .withDescription(coreProgram.getDescription())
                .withPercent(coreProgram.getPercent())
                .withEnabled(coreProgram.getEnabled())
                .withHasRewards(coreProgram.getHasRewards())
                .withIsTechnical(coreProgram.getIsTechnical());
    }

    private static GdCashbackProgramInfo toGdCashbackProgramInfo(CashbackCardsProgram program, GdLanguage language) {
        String programName;
        String tooltipInfo;
        String tooltipLinkText;
        switch (language) {
            case RU:
                programName = program.getNameRu();
                tooltipInfo = program.getTooltipInfoRu();
                tooltipLinkText = program.getTooltipLinkTextRu();
                break;
            case EN:
                programName = program.getNameEn();
                tooltipInfo = program.getTooltipInfoEn();
                tooltipLinkText = program.getTooltipLinkTextEn();
                break;
            default:
                throw new IllegalArgumentException(String.format("Unsupported language %s", language));
        }
        boolean isHaveTooltip = Objects.nonNull(tooltipInfo) ||
                Objects.nonNull(tooltipLinkText) ||
                Objects.nonNull(program.getTooltipLink());
        return new GdCashbackProgramInfo()
                .withProgramId(program.getId())
                .withPercentDecimal(program.getPercent())
                .withPercent(toIntegerPercent(program.getPercent()))
                .withName(programName)
                .withTooltip(isHaveTooltip ? new GdCashbackTooltipInfo()
                        .withInfo(tooltipInfo)
                        .withLink(program.getTooltipLink())
                        .withLinkText(tooltipLinkText) : null);
    }

    private static Integer toIntegerPercent(BigDecimal decimalPercent) {
        return decimalPercent.multiply(BigDecimal.valueOf(100L)).intValue();
    }
}
