package ru.yandex.direct.grid.processing.service.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmAudioAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoPinAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoproductAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmIndoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmOutdoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.McBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimits;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimitsAdvanced;
import ru.yandex.direct.core.entity.bidmodifiers.service.FeaturesProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.GdIntRange;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdAdjustmentAdType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdAgeType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdGenderType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdInventoryType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdOperationType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdOsType;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdTrafaretPosition;
import ru.yandex.direct.grid.processing.model.bidmodifier.GdWeatherType;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValues;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesABSegment;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesAdType;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesDemographics;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesDesktop;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesDesktopOnly;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesExpression;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesGeo;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesInventory;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesMobile;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesRetargeting;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesSmartTV;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesSmartTgo;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesTablet;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesTrafaretPosition;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesVideo;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValuesWeather;
import ru.yandex.direct.grid.processing.model.client.GdAllowedBidModifiersByAdGroupType;
import ru.yandex.direct.grid.processing.model.client.GdAllowedBidModifiersByCampaignType;
import ru.yandex.direct.grid.processing.model.client.GdClientLimits;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounters;
import ru.yandex.direct.grid.processing.model.client.GdDefaultCampaignMetrikaCounters;
import ru.yandex.direct.grid.processing.model.forecast.GdDeviceType;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemInput;
import ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataConverter;
import ru.yandex.direct.grid.processing.service.client.converter.ClientConstantsConverter;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.web.core.entity.inventori.service.CryptaService;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalWeb;

import static com.google.common.collect.ImmutableSet.copyOf;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static org.apache.commons.collections4.ListUtils.emptyIfNull;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_TYPES;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.CLOUDNESS_VALUES;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.MAX_TEMP_VALUE;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.MIN_TEMP_VALUE;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.PREC_STRENGTH_VALUES;
import static ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimitsAdvanced.DEFAULT_MAX;
import static ru.yandex.direct.feature.FeatureName.BRANDSAFETY_ADDITIONAL_CATEGORIES;
import static ru.yandex.direct.feature.FeatureName.GEO_RATE_CORRECTIONS_ALLOWED_FOR_DNA;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.PERFORMANCE;
import static ru.yandex.direct.grid.model.entity.adgroup.AdGroupTypeConverter.toGdAdGroupTypes;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignType;
import static ru.yandex.direct.grid.processing.model.forecast.GdDeviceType.ALL;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_18;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_INF;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemGender.FEMALE;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemGender.MALE;
import static ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataConverter.toGdBidModifierType;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.getVisibleSyntheticAdGroupTypes;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.isSynthetic;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
class ClientConstantsDataService {
    private static final long POLITICS_ID = 4294967302L;
    private static final long CHILDREN_ID = 4294967303L;
    private static final long NEWS_ID = 4294967305L;
    private static final long RELIGION_ID = 4294967307L;
    private static final long VIDEOGAMES_ID = 4294967308L;

    public static final GdTouchSocdemInput DEFAULT_UC_SOCDEM = new GdTouchSocdemInput()
            .withGenders(List.of(MALE, FEMALE))
            .withAgeLower(AGE_18)
            .withAgeUpper(AGE_INF);
    public static final Set<GdDeviceType> DEFAULT_UC_DEVICES = Set.of(ALL);

    public static final boolean DEFAULT_METRIKA_AVALIABLE = true;

    private static final Set<Long> ADDITIONAL_BRAND_SAFETY_IDS = Set.of(
            POLITICS_ID,
            CHILDREN_ID,
            NEWS_ID,
            RELIGION_ID,
            VIDEOGAMES_ID
    );

    private static final Set<GdInventoryType> ALLOWED_INVENTORY_TYPES = Set.of(
            GdInventoryType.INPAGE,
            GdInventoryType.INSTREAM_WEB,
            GdInventoryType.INAPP,
            GdInventoryType.REWARDED
    );

    private static final Set<GdTrafaretPosition> ALLOWED_TRAFARET_POSITIONS = Set.of(
            GdTrafaretPosition.ALONE,
            GdTrafaretPosition.SUGGEST
    );

    private final ClientLimitsService clientLimitsService;
    private final GridContextProvider contextProvider;
    private final FeatureService featureService;
    private final ClientService clientService;
    private final CryptaService cryptaService;

    /**
     * Список типов групп, поддерживаемых в новом интерфейсе
     * Добавлены ещё и типы кампаний, т.к. для определения допустимости корректировок нужно передать и их тоже.
     */
    private static final List<Combination> SUPPORTED_IN_GRID = asList(
            new Combination(CampaignType.TEXT, AdGroupType.BASE, TextAdGroup::new),
            new Combination(CampaignType.DYNAMIC, AdGroupType.DYNAMIC, DynamicTextAdGroup::new),
            new Combination(CampaignType.PERFORMANCE, AdGroupType.PERFORMANCE, PerformanceAdGroup::new),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_BANNER,
                    () -> new CpmBannerAdGroup().withCriterionType(CriterionType.KEYWORD)),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_VIDEO, CpmVideoAdGroup::new),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_AUDIO, CpmAudioAdGroup::new),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_OUTDOOR, CpmOutdoorAdGroup::new),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_INDOOR, CpmIndoorAdGroup::new),
            new Combination(CampaignType.MCBANNER, AdGroupType.MCBANNER, McBannerAdGroup::new),
            new Combination(CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, MobileContentAdGroup::new),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_GEOPRODUCT, CpmGeoproductAdGroup::new)
    );

    /**
     * Список типов групп, поддерживаемых в новом интерфейсе по фиче.
     */
    private static final Map<List<FeatureName>, Combination> SUPPORTED_BY_FEATURE_IN_GRID = Map.of(
            List.of(FeatureName.CONTENT_PROMOTION_VIDEO_ON_GRID, FeatureName.CONTENT_PROMOTION_COLLECTIONS_ON_GRID),
            new Combination(CampaignType.CONTENT_PROMOTION, AdGroupType.CONTENT_PROMOTION,
                    ContentPromotionAdGroup::new),
            List.of(FeatureName.CPM_GEO_PIN_PRODUCT_ENABLED),
            new Combination(CampaignType.CPM_BANNER, AdGroupType.CPM_GEO_PIN, CpmGeoPinAdGroup::new)
    );

    @Autowired
    ClientConstantsDataService(ClientLimitsService clientLimitsService,
                               GridContextProvider contextProvider, FeatureService featureService,
                               ClientService clientService, CryptaService cryptaService) {
        this.clientLimitsService = clientLimitsService;
        this.contextProvider = contextProvider;
        this.featureService = featureService;
        this.clientService = clientService;
        this.cryptaService = cryptaService;
    }

    Collection<CryptaGoalWeb> getAllowedBrandSafetyCategoriesByClient(ClientId clientId) {
        var brandSafety = cryptaService.getBrandSafety();

        if (!featureService.isEnabledForClientId(clientId, BRANDSAFETY_ADDITIONAL_CATEGORIES)) {
            return brandSafety
                    .stream()
                    .filter(this::notContainsInAdditionalCategories)
                    .collect(Collectors.toList());
        }

        return brandSafety;
    }

    Collection<CryptaGoalWeb> getAdditionalBrandSafetyCategoriesForClient(ClientId clientId) {
        return cryptaService.getAdditionalBrandSafetyCategoriesForClient(clientId);
    }

    private boolean notContainsInAdditionalCategories(CryptaGoalWeb goal) {
        return !ADDITIONAL_BRAND_SAFETY_IDS.contains(goal.getId());
    }

    GdClientLimits getClientLimits(GridGraphQLContext context) {
        ClientId clientId = context.getSubjectUser().getClientId();
        Map<ClientId, ClientLimits> clientIdClientLimitsMap = listToMap(
                clientLimitsService.massGetClientLimits(List.of(clientId)), ClientLimits::getClientId);
        return ClientConstantsConverter.toGdClientLimits(clientIdClientLimitsMap.get(clientId));
    }

    public List<GdDefaultCampaignMetrikaCounters> getDefaultMetrikaCountersForCampaignTypes(
            ClientId clientId,
            @Nullable Collection<GdCampaignType> campaignTypes) {
        List<Integer> counters = emptyIfNull(getCommonCounters(clientId));
        // Для смарт кампаний передаем только один счетчик метрики
        return StreamEx.of(nvl(campaignTypes, SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES))
                .mapToEntry(type -> new GdDefaultCampaignMetrikaCounters().withCampaignType(type))
                .mapKeyValue((type, metrikaCounters) -> metrikaCounters
                        .withCounters(copyOf(type == PERFORMANCE
                                ? StreamEx.of(counters).limit(1L).toList()
                                : counters))
                        //Этим не должны пользоваться, но чтобы не сломать, отдаем false
                        .withIsMetrikaAvailable(DEFAULT_METRIKA_AVALIABLE))
                .toList();
    }

    public GdClientMetrikaCounters getCommonMetrikaCounters(ClientId clientId) {
        List<Integer> counters = getCommonCounters(clientId);
        return new GdClientMetrikaCounters()
                .withCounters(counters == null ? emptySet() : listToSet(counters))
                //Этим не должны пользоваться, но чтобы не сломать, отдаем false
                .withIsMetrikaAvailable(DEFAULT_METRIKA_AVALIABLE);
    }

    private List<Integer> getCommonCounters(ClientId clientId) {
        Map<Long, List<Long>> commonMetrikaCountersByClientId =
                clientService.getClientsCommonMetrikaCounters(List.of(clientId));
        return mapList(commonMetrikaCountersByClientId.get(clientId.asLong()), Long::intValue);
    }

    private static class Combination {
        private final CampaignType campaignType;
        private final AdGroupType adGroupType;
        private final Supplier<AdGroup> adGroupSupplier;

        private Combination(CampaignType campaignType, AdGroupType adGroupType,
                            Supplier<AdGroup> adGroupSupplier) {
            this.campaignType = campaignType;
            this.adGroupType = adGroupType;
            this.adGroupSupplier = adGroupSupplier;
        }
    }

    List<GdAllowedBidModifiersByAdGroupType> getAllowedBidModifierTypesByAgGroupTypes() {
        ClientId clientId = ClientId.fromLong(contextProvider.getGridContext().getQueriedClient().getId());

        return StreamEx.of(SUPPORTED_IN_GRID)
                .append(EntryStream.of(SUPPORTED_BY_FEATURE_IN_GRID)
                        .filterKeys(featureNames -> isAnyFeatureEnabled(clientId, featureNames))
                        .values())
                .flatMap(combination -> {
                    CampaignType campaignType = combination.campaignType;
                    AdGroupType adGroupType = combination.adGroupType;
                    AdGroup adGroup = combination.adGroupSupplier.get().withType(adGroupType);

                    return StreamEx.of(
                            buildAllowedBidModifiers(adGroupType, campaignType, clientId,
                                    getEnabledBidModifierTypes(adGroup, campaignType, clientId)));
                })
                .toList();
    }

    private boolean isAnyFeatureEnabled(ClientId clientId, List<FeatureName> featureNames) {
        return StreamEx.of(featureNames)
                .map(featureName -> featureService.isEnabledForClientId(clientId, featureName))
                .reduce(Boolean::logicalOr)
                .orElse(false);
    }

    private List<GdAllowedBidModifiersByAdGroupType> buildAllowedBidModifiers(AdGroupType adGroupType,
                                                                              CampaignType campaignType,
                                                                              ClientId clientId,
                                                                              List<BidModifierType> enabledTypes) {
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        Set<GdAdGroupType> visibleSyntheticAdGroupTypes = getVisibleSyntheticAdGroupTypes(enabledFeatures);

        return StreamEx.of(toGdAdGroupTypes(adGroupType, toGdCampaignType(campaignType)))
                // Дополнительная фильтрация для синтетических типов групп
                .filter(gdAdGroupType -> !isSynthetic(gdAdGroupType) || visibleSyntheticAdGroupTypes.contains(gdAdGroupType))
                .map(gdAdGroupType -> new GdAllowedBidModifiersByAdGroupType()
                        .withAdGroupType(gdAdGroupType)
                        .withAllowedBidModifierTypes(
                                enabledTypes.stream()
                                        .map(BidModifierDataConverter::toGdBidModifierType)
                                        .collect(Collectors.toSet())
                        ))
                .toList();
    }

    private List<BidModifierType> getEnabledBidModifierTypes(AdGroup adGroup,
                                                             CampaignType campaignType,
                                                             ClientId clientId) {
        FeaturesProvider featureProvider = featureService::isEnabledForClientId;

        List<BidModifierType> enabledTypes = new ArrayList<>();
        for (BidModifierType bidModifierType : ALL_TYPES) {
            BidModifierLimits limits = BidModifierLimitsAdvanced.getLimits(
                    bidModifierType, campaignType, adGroup, clientId, featureProvider);
            if (limits.enabled) {
                enabledTypes.add(bidModifierType);
            }
        }
        return enabledTypes;
    }

    List<GdAllowedBidModifiersByCampaignType> getAllowedBidModifierTypesByCampaignTypes() {
        ClientId clientId = ClientId.fromLong(contextProvider.getGridContext().getQueriedClient().getId());

        Map<CampaignType, List<BidModifierType>> supported = new HashMap<>();
        FeaturesProvider featureProvider = featureService::isEnabledForClientId;

        List<Combination> allSupportedCombinations = StreamEx.of(SUPPORTED_IN_GRID)
                .append(EntryStream.of(SUPPORTED_BY_FEATURE_IN_GRID)
                        .filterKeys(featureNames -> isAnyFeatureEnabled(clientId, featureNames))
                        .values())
                .toList();

        for (Combination combination : allSupportedCombinations) {
            CampaignType campaignType = combination.campaignType;

            List<BidModifierType> enabledTypes = new ArrayList<>();
            for (BidModifierType bidModifierType : ALL_TYPES) {
                BidModifierLimits limits = BidModifierLimitsAdvanced.getLimits(
                        bidModifierType, campaignType, null, clientId, featureProvider);
                if (limits.enabled && isBidModifiersEnabledAdditional(bidModifierType, clientId)) {
                    enabledTypes.add(bidModifierType);
                }
            }
            supported.put(campaignType, enabledTypes);
        }

        return EntryStream.of(supported)
                .mapKeyValue((campaignType, bidModifierTypes) ->
                        new GdAllowedBidModifiersByCampaignType()
                                .withCampaignType(toGdCampaignType(campaignType))
                                .withAllowedBidModifierTypes(
                                        listToSet(bidModifierTypes, BidModifierDataConverter::toGdBidModifierType)
                                ))
                .toList();
    }

    List<GdAllowedBidModifierValues> getAllowedBidModifierValues() {
        FeaturesProvider featuresProvider = featureService::isEnabledForClientId;
        return ALL_TYPES.stream()
                .map(type -> getGdAllowedBidModifierValuesForType(type, featuresProvider))
                .collect(Collectors.toList());
    }

    private GdAllowedBidModifierValues getGdAllowedBidModifierValuesForType(BidModifierType type,
                                                                            FeaturesProvider featuresProvider) {
        ClientId clientId = ClientId.fromLong(contextProvider.getGridContext().getQueriedClient().getId());

        GdAllowedBidModifierValues values;
        // Если имеем дело с универсальной корректировкой, то возвращаем только базовую модель
        if (Constants.EXPRESSION_MODIFIER_TYPES.contains(type)) {
            values = new GdAllowedBidModifierValuesExpression();
        } else {
            switch (type) {
                case MOBILE_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesMobile()
                            .withOsTypes(Set.of(GdOsType.values()));
                    break;
                case TABLET_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesTablet()
                            .withOsTypes(Set.of(GdOsType.values()));
                    break;
                case DEMOGRAPHY_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesDemographics()
                            .withAges(Set.of(GdAgeType.values()))
                            .withGenders(Set.of(GdGenderType.values()));
                    break;
                case RETARGETING_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesRetargeting();
                    break;
                case GEO_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesGeo();
                    break;
                case VIDEO_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesVideo();
                    break;
                case PERFORMANCE_TGO_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesSmartTgo();
                    break;
                case AB_SEGMENT_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesABSegment();
                    break;
                case BANNER_TYPE_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesAdType()
                            .withAdTypes(Set.of(GdAdjustmentAdType.values()));
                    break;
                case INVENTORY_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesInventory()
                            .withInventoryTypes(ALLOWED_INVENTORY_TYPES);
                    break;
                case DESKTOP_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesDesktop();
                    break;
                case DESKTOP_ONLY_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesDesktopOnly();
                    break;
                case SMARTTV_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesSmartTV();
                    break;
                case WEATHER_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesWeather()
                            .withOperations(Set.of(GdOperationType.values()))
                            .withParameters(Set.of(GdWeatherType.values()))
                            .withTemperatureRange(new GdIntRange()
                                    .withMax(MAX_TEMP_VALUE)
                                    .withMin(MIN_TEMP_VALUE)
                                    .withStep(1))
                            .withCloudness(CLOUDNESS_VALUES)
                            .withPrecStrength(PREC_STRENGTH_VALUES);
                    break;
                case TRAFARET_POSITION_MULTIPLIER:
                    values = new GdAllowedBidModifierValuesTrafaretPosition()
                            .withTrafaretPositions(ALLOWED_TRAFARET_POSITIONS);
                    break;
                default:
                    throw new IllegalArgumentException("Unknown GdBidModifierType: " + type.name());
            }
        }

        BidModifierLimits bidModifierLimits = BidModifierLimitsAdvanced.getLimits(
                type, CampaignType.TEXT, null, clientId, featuresProvider);

        //todo Фронт пока фильтрует типы ОС на своей стороне по фиче,
        // тикет на возможное изменение https://st.yandex-team.ru/DIRECT-103122
//        if (type == BidModifierType.MOBILE_MULTIPLIER && !featureService.isEnabled(clientId, CPC_DEVICE_MODIFIERS)) {
//            ((GdAllowedBidModifierValuesMobile) values).setOsTypes(Collections.emptySet());
//        }
        //аналогичное поведение и с фичей MOBILE_OS_BID_MODIFIER_ENABLED, только тут надо на фронт отдавать "более
        // частое" значение, для правильной валидации с включенной фичей, фронт пока проверяет на своей стороне
        return values.withPercentRange(
                new GdIntRange()
                        .withMax(type == BidModifierType.DESKTOP_MULTIPLIER ? DEFAULT_MAX :
                                bidModifierLimits.percentMax)
                        .withMin(bidModifierLimits.percentMin)
                        .withStep(1))
                .withMaxConditions(nvl(bidModifierLimits.maxConditions, 1))
                .withType(toGdBidModifierType(type));
    }

    //Временный костыль, пока  вновом редактировании идет переходный процесс по поддержке гео-корректировок
    //Проверяет включенность фичи для геокорректировок
    private boolean isBidModifiersEnabledAdditional(BidModifierType bidModifierType, ClientId clientId) {
        if (bidModifierType != BidModifierType.GEO_MULTIPLIER) {
            return true;
        }
        return featureService.isEnabledForClientId(clientId, GEO_RATE_CORRECTIONS_ALLOWED_FOR_DNA);
    }

    public static final Set<StrategyName> STRATEGY_TYPES_FOR_CLEANING_BID_MODIFIERS = Set.of(
            StrategyName.AUTOBUDGET_AVG_CPA,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER,
            StrategyName.AUTOBUDGET_AVG_CPI,
            StrategyName.AUTOBUDGET_CRR,
            StrategyName.AUTOBUDGET_ROI,
            StrategyName.AUTOBUDGET);
}
