package ru.yandex.direct.grid.processing.service.client;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.bidmodifier.values.GdAllowedBidModifierValues;
import ru.yandex.direct.grid.processing.model.client.GdAllowedBidModifiersByAdGroupType;
import ru.yandex.direct.grid.processing.model.client.GdAllowedBidModifiersByCampaignType;
import ru.yandex.direct.grid.processing.model.client.GdBrandSafetyCategory;
import ru.yandex.direct.grid.processing.model.client.GdClientConstants;
import ru.yandex.direct.grid.processing.model.client.GdClientLimits;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounters;
import ru.yandex.direct.grid.processing.model.client.GdDefaultCampaignMetrikaCounters;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalWeb;

/**
 * Сервис, возвращающий данные о "константах" завязанные на клиентские данные
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ClientConstantsGraphQlService {
    private final ClientConstantsDataService clientConstantsDataService;

    @Autowired
    public ClientConstantsGraphQlService(ClientConstantsDataService clientConstantsDataService) {
        this.clientConstantsDataService = clientConstantsDataService;
    }

    /**
     * GraphQL ручка. Отдает провязку типов групп и применимых к ним типов корректировок ставок
     * Прямо сейчас не использует клиентские данные, но для определения применимости некоторых типов корректировок
     * требуется проверка фичей
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "allowedBidModifierTypesByAdGroupTypes")
    public List<@GraphQLNonNull GdAllowedBidModifiersByAdGroupType> getAllowedBidModifierTypesByAgGroupTypes(
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants) {
        return clientConstantsDataService.getAllowedBidModifierTypesByAgGroupTypes();
    }

    /**
     * GraphQL ручка. Отдает провязку типов кампаний и применимых к ним типов корректировок ставок
     * Прямо сейчас не использует клиентские данные, но для определения применимости некоторых типов корректировок
     * требуется проверка фичей
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "allowedBidModifierTypesByCampaignTypes")
    public List<@GraphQLNonNull GdAllowedBidModifiersByCampaignType> getAllowedBidModifierTypesByCampaignTypes(
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants) {
        return clientConstantsDataService.getAllowedBidModifierTypesByCampaignTypes();
    }

    /**
     * GraphQL ручка. Для всех типов корректировок диапазоны и наборы допустимых для клиента значений
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "allowedBidModifiersValues")
    public List<@GraphQLNonNull GdAllowedBidModifierValues> getAllowedBidModifiersValues(
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants) {
        return clientConstantsDataService.getAllowedBidModifierValues();
    }

    /**
     * GraphQL ручка. Отдает ограничения на клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "clientLimits")
    public GdClientLimits getClientLimits(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants) {
        return clientConstantsDataService.getClientLimits(context);
    }

    /**
     * Отдает доступные brand safety категории на клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "allowedBrandSafetyCategoriesByClient")
    public Collection<GdBrandSafetyCategory> getAllowedBrandSafetyCategoriesByClient(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants
    ) {

        var clientId = context.getSubjectUser().getClientId();
        var goals = clientConstantsDataService.getAllowedBrandSafetyCategoriesByClient(clientId);
        return mapGoalsToGdBrandSafetyCategoryCollection(goals);
    }

    /**
     * Отдает дополнительные brand safety категории доступные для клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "additionalBrandSafetyCategoriesForClient")
    public Collection<GdBrandSafetyCategory> getAdditionalBrandSafetyCategoriesForClient(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants
    ) {
        var clientId = context.getSubjectUser().getClientId();
        var goals = clientConstantsDataService.getAdditionalBrandSafetyCategoriesForClient(clientId);
        return mapGoalsToGdBrandSafetyCategoryCollection(goals);
    }

    /**
     * Отдает счетчики Метрики по умолчанию на клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "commonMetrikaCounters")
    public GdClientMetrikaCounters getCommonMetrikaCounters(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants) {

        var clientId = context.getSubjectUser().getClientId();
        return clientConstantsDataService.getCommonMetrikaCounters(clientId);
    }

    /**
     * Отдает счетчики Метрики по умолчанию на клиента, количественно ограниченных типом кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "defaultCampaignMetrikaCounters")
    public List<@GraphQLNonNull GdDefaultCampaignMetrikaCounters> getDefaultCampaignMetrikaCounters(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClientConstants clientConstants,
            @GraphQLArgument(name = "campaignTypes") @Nullable List<@GraphQLNonNull GdCampaignType> campaignTypes) {
        var clientId = context.getSubjectUser().getClientId();
        return clientConstantsDataService.getDefaultMetrikaCountersForCampaignTypes(clientId, campaignTypes);
    }

    private Collection<GdBrandSafetyCategory> mapGoalsToGdBrandSafetyCategoryCollection(Collection<CryptaGoalWeb> goals) {
        return goals
                .stream()
                .map(this::mapGoalToGdBrandSafetyCategory)
                .collect(Collectors.toList());
    }

    private GdBrandSafetyCategory mapGoalToGdBrandSafetyCategory(CryptaGoalWeb goal) {
        return new GdBrandSafetyCategory()
                .withId(goal.getId())
                .withName(goal.getName())
                .withDescription(goal.getDescription());
    }

}
