package ru.yandex.direct.grid.processing.service.client;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.abac.Attribute;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.adgeneration.ZenDomain;
import ru.yandex.direct.core.entity.adgeneration.ZenMetaInfoService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.agency.service.AgencyService;
import ru.yandex.direct.core.entity.banner.type.href.BannerHrefValidator;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.calltrackingsettings.service.CalltrackingSettingsService;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientMeasurerSettingsService;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneService;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.domain.SocialNetworkAccountDomainUtils;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.forecast.CpaEstimatesService;
import ru.yandex.direct.core.entity.forecast.model.CpaEstimatesContainer;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectFilter;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.geo.service.CurrentGeoService;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.core.entity.metrika.model.MetrikaCounterByDomain;
import ru.yandex.direct.core.entity.metrika.service.ForCampaignType;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsConversionService;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.page.service.PageService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCountersAndGoals;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.core.entity.uac.model.ShopInShopBusinessInfo;
import ru.yandex.direct.core.entity.uac.model.Source;
import ru.yandex.direct.core.entity.uac.service.shopinshop.ShopInShopBusinessesService;
import ru.yandex.direct.core.entity.user.model.AgencyLimRep;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.user.utils.UserUtil;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignEmailEvent;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignEmailSettings;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignNotification;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignSmsSettings;
import ru.yandex.direct.grid.model.campaign.notification.GdDefaultCampaignNotification;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.container.agency.GdAgencyInfo;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GdExceptions;
import ru.yandex.direct.grid.processing.exception.GridPublicException;
import ru.yandex.direct.grid.processing.exception.NoSuchObjectException;
import ru.yandex.direct.grid.processing.model.campaign.GdSocialNetworkAccountType;
import ru.yandex.direct.grid.processing.model.campaign.GdSuggestCampaignsSettings;
import ru.yandex.direct.grid.processing.model.campaign.GdSuggestSocialNetworkAccount;
import ru.yandex.direct.grid.processing.model.client.GdAccountScoreInfo;
import ru.yandex.direct.grid.processing.model.client.GdAgencyLimRepInfo;
import ru.yandex.direct.grid.processing.model.client.GdAvailableFields;
import ru.yandex.direct.grid.processing.model.client.GdCalltrackingOnSiteSuggest;
import ru.yandex.direct.grid.processing.model.client.GdCampaignsAvailableFields;
import ru.yandex.direct.grid.processing.model.client.GdClientAutoOverdraftInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientFeatures;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientMeasurerSystem;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounter;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounterForSuggest;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCountersWithAdditionalInformation;
import ru.yandex.direct.grid.processing.model.client.GdClientNotificationInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientSearchRequest;
import ru.yandex.direct.grid.processing.model.client.GdClientTopRegionInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientTopRegions;
import ru.yandex.direct.grid.processing.model.client.GdCoreFeatureWithDescription;
import ru.yandex.direct.grid.processing.model.client.GdCpaEstimate;
import ru.yandex.direct.grid.processing.model.client.GdInaccessibleMetrikaCounter;
import ru.yandex.direct.grid.processing.model.client.GdInternalDontShowDomainsDataPayload;
import ru.yandex.direct.grid.processing.model.client.GdInternalDontShowDomainsDataPayloadItem;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCounter;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterSource;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterStatus;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersByDomainPayload;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersFilterPayload;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersInfoPayload;
import ru.yandex.direct.grid.processing.model.client.GdShowCpmUacPromoParams;
import ru.yandex.direct.grid.processing.model.client.GdSuggestCpaGoalInfo;
import ru.yandex.direct.grid.processing.model.client.GdSuggestDataByUrl;
import ru.yandex.direct.grid.processing.model.client.GdSuggestDataByUrlPayload;
import ru.yandex.direct.grid.processing.model.client.GdSuggestMetrikaDataByUrl;
import ru.yandex.direct.grid.processing.model.client.GdSuggestMetrikaDataByUrlPayload;
import ru.yandex.direct.grid.processing.model.client.GdTextSuggest;
import ru.yandex.direct.grid.processing.model.client.GdUserFeature;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.model.client.GdZenMetaInfo;
import ru.yandex.direct.grid.processing.model.cliententity.GdClientMeasurerAccountDetails;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdClientMeasurerSettingsMediascope;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdAddress;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPhoneWithId;
import ru.yandex.direct.grid.processing.model.constants.GdCurrencyDescription;
import ru.yandex.direct.grid.processing.model.goal.GdGoal;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaCounterGoalType;
import ru.yandex.direct.grid.processing.model.goal.GdRecommendedGoalCostPerAction;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.grid.processing.service.client.converter.BusinessCategoriesService;
import ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter;
import ru.yandex.direct.grid.processing.service.client.converter.GdFeatureWithDescriptionConverterService;
import ru.yandex.direct.grid.processing.service.client.loader.ClientsAccountScoreInfoDataLoader;
import ru.yandex.direct.grid.processing.service.client.model.MetrikaCounterForSuggest;
import ru.yandex.direct.grid.processing.service.client.validation.SaveChoiceFromConversionModifiersPopupValidationService;
import ru.yandex.direct.grid.processing.service.constant.ConstantsConverter;
import ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils;
import ru.yandex.direct.grid.processing.service.goal.GoalDataService;
import ru.yandex.direct.grid.processing.service.goal.GoalMutationService;
import ru.yandex.direct.grid.processing.service.operator.OperatorClientRelationsHelper;
import ru.yandex.direct.grid.processing.service.operator.OperatorDataService;
import ru.yandex.direct.grid.processing.service.organizations.OrganizationsConverter;
import ru.yandex.direct.grid.processing.service.organizations.OrganizationsDataService;
import ru.yandex.direct.grid.processing.service.trackingphone.CalltrackingOnSiteDataService;
import ru.yandex.direct.grid.processing.service.trackingphone.Converter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.metrika.client.model.response.GoalConversionInfo;
import ru.yandex.direct.rbac.RbacAgencyLimRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.GeoTreeType;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.ListUtils;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.db.PpcPropertyNames.METRIKA_COUNTER_IDS_EXCLUDED_FROM_AUTOCOMPLETE;
import static ru.yandex.direct.common.db.PpcPropertyNames.METRIKA_COUNTER_USE_WEAK_RESTRICTIONS_FOR_SUGGESTION;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.toUnicodeDomain;
import static ru.yandex.direct.core.entity.campaign.CampaignNotificationUtils.getAvailableEmailEvents;
import static ru.yandex.direct.core.entity.campaign.CampaignNotificationUtils.getAvailableSmsFlags;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCounterSource;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.WEB_EDIT;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPI;
import static ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService.MAX_METRIKA_COUNTERS_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_UAC_IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_UAC_IS_RECOMMENDATIONS_MANAGEMENT_ENABLED;
import static ru.yandex.direct.core.entity.forecast.model.CpaEstimateAttributionModel.LAST_YANDEX_DIRECT_CLICK;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.NEW_CARD_PAY_LIMIT;
import static ru.yandex.direct.feature.FeatureName.CASHBACK_PAGE_ENABLED;
import static ru.yandex.direct.feature.FeatureName.CONVERSION_MODIFIERS_POPUP_ENABLED;
import static ru.yandex.direct.feature.FeatureName.ENABLE_ONBOARDING_WIZARD;
import static ru.yandex.direct.feature.FeatureName.PHONE_VERIFICATION_STATUS_ENABLED;
import static ru.yandex.direct.feature.FeatureName.TURBO_LANDINGS_CREATION_DISABLED;
import static ru.yandex.direct.feature.FeatureName.UAC_UNAVAILABLE_GOALS_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_USE_CATEGORICAL_CPA_SUGGEST;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_USE_HISTORICAL_CPA_SUGGEST;
import static ru.yandex.direct.grid.core.util.money.GridMoneyUtils.applyNdsSubtruction;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignType;
import static ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterSource.SPRAV;
import static ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterStatus.ACCESS_REQUESTED;
import static ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterStatus.NOT_FOUND;
import static ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterStatus.NO_RIGHTS;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toGdCampaignSmsEventInfo;
import static ru.yandex.direct.grid.processing.service.campaign.uc.GdUcCampaignService.makeFaviconLink;
import static ru.yandex.direct.grid.processing.service.client.ClientConstantsDataService.DEFAULT_UC_DEVICES;
import static ru.yandex.direct.grid.processing.service.client.ClientConstantsDataService.DEFAULT_UC_SOCDEM;
import static ru.yandex.direct.grid.processing.service.client.ClientConstantsDataService.STRATEGY_TYPES_FOR_CLEANING_BID_MODIFIERS;
import static ru.yandex.direct.grid.processing.service.client.ClientFeatureCalculator.FEATURE_CALCULATOR;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdAgencyInfo;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdAgencyLimRepInfo;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdClientMeasurerSystem;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdClientMetrikaCountersWithAdditionalInformation;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdGeotreeType;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdMetrikaCounterSource;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toGdPhoneVerificationStatus;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdGoals;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.regions.Region.BY_REGION_ID;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.regions.Region.KAZAKHSTAN_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.OptionalUtils.convertToCommonOptional;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Сервис, возвращающий базовые данные
 */
@Service
@ParametersAreNonnullByDefault
public class ClientDataService {
    private static final Logger logger = LoggerFactory.getLogger(ClientDataService.class);
    private static final int MAX_METRIKA_COUNTERS_COUNT_FOR_SUGGEST = 20;
    static final int MAX_INACCESSIBLE_METRIKA_COUNTERS_COUNT_FOR_SUGGEST = 10;

    private static final Map<GdCampaignsAvailableFields, Attribute> CAMPAIGNS_AVAILABLE_FIELDS_ATTRIBUTE_MAP = Map.of(
            GdCampaignsAvailableFields.ALLOWED_BRANDSAFETY_CATEGORIES_CPM, Attribute.CAN_SET_BRANDSAFETY_CATEGORIES_CPM,
            GdCampaignsAvailableFields.ALLOWED_BRANDSAFETY_CATEGORIES_CPC, Attribute.CAN_SET_BRANDSAFETY_CATEGORIES_CPC,
            GdCampaignsAvailableFields.ALLOWED_TURBO_APP, Attribute.CAN_ENABLE_TURBO_APPS
    );

    private static final Long[] cashbackAvailableRegions = {
            RUSSIA_REGION_ID, GLOBAL_REGION_ID, KAZAKHSTAN_REGION_ID, BY_REGION_ID
    };

    private static final List<String> DEFAULT_CPM_UAC_PROMO_COLORS = List.of(
            "#8a84ff",
            "#2e69ff"
    );

    private final ShardHelper shardHelper;
    private final ShardSupport shardSupport;
    private final ClientService clientService;
    private final GeoTreeFactory geoTreeFactory;
    private final ClientGeoService clientGeoService;
    private final UserService userService;
    private final AdGroupService adGroupService;
    private final CampaignService campaignService;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final RbacService rbacService;
    private final OperatorDataService operatorDataService;
    private final FeatureService featureService;
    private final FeatureManagingService featureManagingService;
    private final ClientNdsService clientNdsService;
    private final InternalAdsProductService internalAdsProductService;
    private final ClientMeasurerSettingsService clientMeasurerSettingsService;
    private final AttributeResolverService attributeResolverService;
    private final FreelancerService freelanceService;
    private final ClientsAccountScoreInfoDataLoader clientsAccountScoreInfoDataLoader;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final BlackboxUserService blackboxUserService;
    private final TurboLandingService turboLandingService;
    private final GdFeatureWithDescriptionConverterService gdFeatureWithDescriptionConverter;
    private final PageService pageService;
    private final GridValidationResultConversionService gridValidationResultConversionService;
    private final OrganizationsDataService organizationsDataService;
    private final MetrikaGoalsConversionService metrikaGoalsConversionService;
    private final GoalMutationService goalMutationService;
    private final ClientPhoneService clientPhoneService;
    private final CurrentGeoService currentGeoService;
    private final GeoBaseHelper geoBaseHelper;
    private final CpaEstimatesService cpaEstimatesService;
    private final NetAcl netAcl;
    private final BusinessCategoriesService businessCategoriesService;
    private final GoalDataService goalDataService;
    private final CalltrackingSettingsService calltrackingSettingsService;
    private final CalltrackingOnSiteDataService calltrackingOnSiteDataService;
    private final AgencyService agencyService;
    private final BidModifierService bidModifierService;
    private final SaveChoiceFromConversionModifiersPopupValidationService saveChoiceFromConversionModifiersPopupValidationService;
    private final ZenMetaInfoService zenMetaInfoService;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final PpcProperty<Set<Long>> countersExcludedFromAutocompleteProperty;
    private final PpcProperty<Boolean> metrikaCounterUseWeakRestrictions;
    private final BannersUrlHelper bannersUrlHelper;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final ShopInShopBusinessesService shopInShopBusinessesService;

    @SuppressWarnings("checkstyle:ParameterNumber")
    @Autowired
    public ClientDataService(ShardHelper shardHelper, ShardSupport shardSupport,
                             ClientService clientService,
                             GeoTreeFactory geoTreeFactory,
                             ClientGeoService clientGeoService,
                             UserService userService,
                             AdGroupService adGroupService,
                             CampaignService campaignService,
                             CampaignRepository campaignRepository,
                             CampaignTypedRepository campaignTypedRepository,
                             RbacService rbacService,
                             OperatorDataService operatorDataService,
                             FeatureService featureService,
                             FeatureManagingService featureManagingService,
                             ClientNdsService clientNdsService,
                             InternalAdsProductService internalAdsProductService,
                             ClientMeasurerSettingsService clientMeasurerSettingsService,
                             AttributeResolverService attributeResolverService,
                             FreelancerService freelanceService,
                             ClientsAccountScoreInfoDataLoader clientsAccountScoreInfoDataLoader,
                             CampMetrikaCountersService campMetrikaCountersService,
                             BlackboxUserService blackboxUserService,
                             TurboLandingService turboLandingService,
                             GdFeatureWithDescriptionConverterService gdFeatureWithDescriptionConverter,
                             PageService pageService,
                             GridValidationResultConversionService gridValidationResultConversionService,
                             OrganizationsDataService organizationsDataService,
                             GoalMutationService goalMutationService,
                             PpcPropertiesSupport ppcPropertiesSupport,
                             BannersUrlHelper bannersUrlHelper,
                             MetrikaGoalsConversionService metrikaGoalsConversionService,
                             ClientPhoneService clientPhoneService, CurrentGeoService currentGeoService,
                             GeoBaseHelper geoBaseHelper, CpaEstimatesService cpaEstimatesService,
                             NetAcl netAcl, BusinessCategoriesService businessCategoriesService,
                             GoalDataService goalDataService,
                             AgencyService agencyService,
                             CalltrackingSettingsService calltrackingSettingsService,
                             CalltrackingOnSiteDataService calltrackingOnSiteDataService,
                             BidModifierService bidModifierService,
                             SaveChoiceFromConversionModifiersPopupValidationService saveChoiceFromConversionModifiersPopupValidationService,
                             ZenMetaInfoService zenMetaInfoService,
                             CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
                             ShopInShopBusinessesService shopInShopBusinessesService
    ) {
        this.shardHelper = shardHelper;
        this.shardSupport = shardSupport;
        this.clientService = clientService;
        this.geoTreeFactory = geoTreeFactory;
        this.clientGeoService = clientGeoService;
        this.userService = userService;
        this.adGroupService = adGroupService;
        this.campaignService = campaignService;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.rbacService = rbacService;
        this.operatorDataService = operatorDataService;
        this.featureService = featureService;
        this.featureManagingService = featureManagingService;
        this.clientNdsService = clientNdsService;
        this.internalAdsProductService = internalAdsProductService;
        this.clientMeasurerSettingsService = clientMeasurerSettingsService;
        this.attributeResolverService = attributeResolverService;
        this.freelanceService = freelanceService;
        this.clientsAccountScoreInfoDataLoader = clientsAccountScoreInfoDataLoader;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.blackboxUserService = blackboxUserService;
        this.turboLandingService = turboLandingService;
        this.gdFeatureWithDescriptionConverter = gdFeatureWithDescriptionConverter;
        this.pageService = pageService;
        this.gridValidationResultConversionService = gridValidationResultConversionService;
        this.organizationsDataService = organizationsDataService;
        this.metrikaGoalsConversionService = metrikaGoalsConversionService;
        this.goalMutationService = goalMutationService;
        this.clientPhoneService = clientPhoneService;
        this.bannersUrlHelper = bannersUrlHelper;
        this.currentGeoService = currentGeoService;
        this.geoBaseHelper = geoBaseHelper;
        this.cpaEstimatesService = cpaEstimatesService;
        this.netAcl = netAcl;
        this.businessCategoriesService = businessCategoriesService;
        this.goalDataService = goalDataService;
        this.calltrackingSettingsService = calltrackingSettingsService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.agencyService = agencyService;
        this.zenMetaInfoService = zenMetaInfoService;

        this.countersExcludedFromAutocompleteProperty =
                ppcPropertiesSupport.get(METRIKA_COUNTER_IDS_EXCLUDED_FROM_AUTOCOMPLETE, Duration.ofMinutes(5));
        this.metrikaCounterUseWeakRestrictions =
                ppcPropertiesSupport.get(METRIKA_COUNTER_USE_WEAK_RESTRICTIONS_FOR_SUGGESTION, Duration.ofMinutes(5));
        this.calltrackingOnSiteDataService = calltrackingOnSiteDataService;
        this.bidModifierService = bidModifierService;
        this.saveChoiceFromConversionModifiersPopupValidationService =
                saveChoiceFromConversionModifiersPopupValidationService;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.shopInShopBusinessesService = shopInShopBusinessesService;
    }

    GdClientInfo getClientInfo(GridGraphQLContext context, GdClientSearchRequest searchRequests) {
        return getClientInfo(context, Collections.singletonList(searchRequests)).get(0);
    }

    /**
     * Получить базовую информацию о клиенте по переданным параметрам поиска
     *
     * @param context        контекст исполнения запроса
     * @param searchRequests параметры поиска клиента
     * @throws AccessDeniedException если оператор не имеет доступа к клиенту
     * @throws NoSuchObjectException если клиент не найден
     */
    List<GdClientInfo> getClientInfo(GridGraphQLContext context, List<GdClientSearchRequest> searchRequests) {
        Set<Long> clientIds = new HashSet<>();
        Set<Long> userIds = new HashSet<>();
        Set<String> logins = new HashSet<>();

        for (GdClientSearchRequest searchRequest : searchRequests) {
            if (searchRequest.getId() != null) {
                clientIds.add(searchRequest.getId());
            } else if (searchRequest.getUserId() != null) {
                userIds.add(searchRequest.getUserId());
            } else if (searchRequest.getLogin() != null) {
                logins.add(PassportUtils.normalizeLogin(searchRequest.getLogin()));
            }
        }

        return getClientInfo(context, clientIds, userIds, logins);
    }

    List<GdClientInfo> getClientInfo(GridGraphQLContext context, Collection<Long> clientIds,
                                     Collection<Long> userIds, Collection<String> logins) {
        Collection<Long> clientIdsByLoginsAndUids = getClientIdsByUidsAndLogins(userIds, logins);
        clientIds.addAll(clientIdsByLoginsAndUids);

        return getClientInfo(context, clientIds);
    }

    /**
     * Достать список clientIds по списку logins и userIds.
     * Если логин не найден в нашей БД, то идем в паспорт. Если сразу несколько логинов не найдено в БД, то игноируем их
     */
    protected Collection<Long> getClientIdsByUidsAndLogins(Collection<Long> userIds, Collection<String> logins) {
        Collection<Long> clientIds = new HashSet<>();

        if (!logins.isEmpty()) {
            Map<String, Long> loginToId = shardSupport
                    .getValuesMap(ShardKey.LOGIN, new ArrayList<>(logins), ShardKey.CLIENT_ID, Long.class);

            List<Long> clientIdsFromDb = filterList(loginToId.values(), Objects::nonNull);
            if (!clientIdsFromDb.isEmpty()) {
                clientIds.addAll(clientIdsFromDb);
            } else if (logins.size() == 1) {
                // если логина нет у нас, то идем в паспорт
                Optional<Long> clientId = getClientIdByPassport(logins.iterator().next());
                clientId.ifPresent(clientIds::add);
            }

        }
        if (!userIds.isEmpty()) {
            Map<Long, Long> uidToId = shardSupport
                    .getValuesMap(ShardKey.UID, new ArrayList<>(userIds), ShardKey.CLIENT_ID, Long.class);
            clientIds.addAll(uidToId.values());
        }

        return clientIds;
    }

    /**
     * В некоторых случаях (например лайтовые аккаунты), логина может не быть в нашей БД. Поэтому идем в паспорт за
     * uid для этого логина. Далее, по uid определяем clientId в нашей БД.
     */
    private Optional<Long> getClientIdByPassport(String login) {
        Optional<Long> uidFromPassport = blackboxUserService.getUidByLogin(login);
        if (uidFromPassport.isPresent()) {
            Long uid = uidFromPassport.get();
            Long clientId = shardSupport.getValue(ShardKey.UID, uid, ShardKey.CLIENT_ID, Long.class);
            return Optional.ofNullable(clientId);
        } else {
            return Optional.empty();
        }
    }

    public List<GdClientInfo> getClientInfo(GridGraphQLContext context, Collection<Long> clientIds) {
        return getClientInfo(context.getOperator(), clientIds);
    }

    public List<GdClientInfo> getClientInfo(User operator, Collection<Long> clientIds) {
        List<Long> clientIdsToRetrieve = clientIds.stream()
                .peek(id -> checkOperatorCanReadClientInfo(operator.getUid(), id))
                .collect(toList());

        if (clientIds.size() != clientIdsToRetrieve.size()) {
            throw new NoSuchObjectException("Запрошенный клиент не существует");
        }

        Map<Long, Integer> clientIdToShard = shardSupport
                .getValuesMap(ShardKey.CLIENT_ID, new ArrayList<>(clientIds), ShardKey.SHARD, Integer.class);

        List<ClientId> realClientIds = mapList(clientIdToShard.keySet(), ClientId::fromLong);
        Map<Long, Client> clients = listToMap(clientService.massGetClient(realClientIds), Client::getId);

        // Если primaryManagerUid не задан, то ищем менеджера в кампаниях
        List<Long> clientIdsToGetManagers =
                filterAndMapList(clients.values(), c -> !isValidId(c.getPrimaryManagerUid()), Client::getId);

        Map<Long, Long> clientsManagers = clientService.getClientsManagerFromCampaigns(clientIdsToGetManagers);

        /* Берем представителей агентств, которые работают с клиентами и проверяем, стоит ли у них опция
        "не показывать контактные данные" */
        List<Long> agencyUserIds = mapAndFilterList(clients.values(), Client::getAgencyUserId, CommonUtils::isValidId);

        var clientIdsToGetAgencyLimReps = filterAndMapList(clients.values(),
                client -> client.getAgencyClientId() != null, Client::getClientId);
        Map<Long, Set<AgencyLimRep>> agencyLimRepsByClientIds;
        Set<Long> displayedAgencyLimRepUids;
        if (!clientIdsToGetAgencyLimReps.isEmpty()) {
            agencyLimRepsByClientIds = agencyService.getAgencyLimRepsByClientIds(clientIdsToGetAgencyLimReps);
            displayedAgencyLimRepUids = agencyLimRepsByClientIds.values().stream()
                    .flatMap(Set::stream)
                    .filter(o -> o.getRepType() == RbacAgencyLimRepType.CHIEF || o.getRepType() == RbacAgencyLimRepType.LEGACY)
                    .map(AgencyLimRep::getUid)
                    .collect(toSet());
            if (!displayedAgencyLimRepUids.isEmpty()) {
                agencyUserIds.addAll(displayedAgencyLimRepUids);
            }
        } else {
            agencyLimRepsByClientIds = emptyMap();
            displayedAgencyLimRepUids = emptySet();
        }
        Map<Long, Boolean> showAgencyContactsByUserIds =
                userService.getUsersAgencyShowAgencyContactsWithDefaultFalse(agencyUserIds);

        Map<Long, Boolean> hiddenAgencyContactsByUserIds = EntryStream.of(showAgencyContactsByUserIds)
                .filterValues(isShowEnabled -> !isShowEnabled)
                .toMap();

        Set<Long> agencyChiefUserIds = emptySet();
        /*Для скрытых контактов, надо проверить можем ли показать данные главного представителя агентства
        или же не показывать ничего */
        if (!hiddenAgencyContactsByUserIds.isEmpty()) {
            Map<Long, Client> agencyClients =
                    clientService.massGetClientsByUids(hiddenAgencyContactsByUserIds.keySet());
            agencyChiefUserIds = listToSet(agencyClients.values(), Client::getChiefUid);
            showAgencyContactsByUserIds.putAll(
                    userService.getUsersAgencyShowAgencyContactsWithDefaultFalse(agencyChiefUserIds));
        }

        List<Long> preUserIdsToCollect = clients.values().stream()
                .flatMap(c -> Stream.of(c.getAgencyUserId(), c.getPrimaryManagerUid(), c.getChiefUid())
                        .filter(CommonUtils::isValidId))
                .collect(toList());

        Set<Long> userIdsToCollect = StreamEx.of(clientsManagers.values())
                .append(preUserIdsToCollect)
                .append(agencyChiefUserIds)
                .append(displayedAgencyLimRepUids)
                .collect(toSet());
        Map<Long, GdUserInfo> usersByIds = operatorDataService.getUsersByIds(userIdsToCollect);

        List<ClientId> relatedClientIds = StreamEx.of(preUserIdsToCollect)
                .map(usersByIds::get)
                .nonNull()
                .map(GdUserInfo::getClientId)
                .map(ClientId::fromLong)
                .toList();

        Map<Long, Client> relatedClientsByIds = new HashMap<>();
        List<ClientId> relatedClientIdsToRetrieve = new ArrayList<>();
        for (ClientId id : relatedClientIds) {
            Client client = clients.get(id.asLong());
            if (client == null) {
                relatedClientIdsToRetrieve.add(id);
            } else {
                relatedClientsByIds.put(id.asLong(), client);
            }
        }
        relatedClientsByIds.putAll(listToMap(clientService.massGetClient(relatedClientIdsToRetrieve), Client::getId));

        // Получаем фактический НДС для всех клиентов (его может не оказаться в БД)
        Map<Long, ClientNds> clientNdsMap =
                listToMap(clientNdsService.massGetEffectiveClientNds(clients.values()), ClientNds::getClientId);

        Map<Long, GeoTreeType> clientToGeoTreeType =
                listToMap(clients.values(), Client::getId, client -> geoTreeFactory
                        .getTranslocalGeoTree(client.getCountryRegionId()).getGeoTreeType());

        List<ClientId> internalAdProductClientIds = rbacService.filterInternalAdProducts(realClientIds);
        Map<ClientId, InternalAdsProduct> internalAdsProducts =
                listToMap(internalAdsProductService.getProducts(internalAdProductClientIds),
                        InternalAdsProduct::getClientId, identity());

        FreelancerProjectFilter freelancerProjectFilter = new FreelancerProjectFilter().withIsActive(true);
        Set<Long> clientIdsUnderFreelancer = freelanceService
                .massGetClientProjects(new ArrayList<>(clientIds), freelancerProjectFilter)
                .stream()
                .map(FreelancerProject::getClientId)
                .collect(toSet());

        Map<ClientId, Set<String>> enabledFeaturesByClientId = featureService.getEnabled(
                mapSet(clientIdToShard.keySet(), ClientId::fromLong));

        boolean useNewAutopayCardLimit = ppcPropertiesSupport.find(NEW_CARD_PAY_LIMIT.getName())
                .orElse("0").equals("1");
        var defaultCpmFrontpagePriceMap = cpmYndxFrontpageCurrencyService.getDefaultCpmFrontpagePrice();

        return EntryStream.of(clientIdToShard)
                .peekValues(s -> {
                    if (s == null) {
                        throw new NoSuchObjectException("Запрошенный клиент не существует");
                    }
                })
                .mapKeyValue((clientId, shard) -> {
                    Client client = clients.get(clientId);
                    ClientId realClientId = ClientId.fromLong(clientId);

                    boolean isConversionMultipliersPopupEnabled =
                            isConversionMultipliersPopupEnabled(shard, client,
                                    enabledFeaturesByClientId.get(realClientId));

                    PhoneVerificationStatus phoneVerificationStatus =
                            getPhoneVerificationStatus(client.getPhoneVerificationStatus(), realClientId);

                    return createClientInfo(shard, client, usersByIds,
                            clientsManagers.get(clientId), relatedClientsByIds,
                            showAgencyContactsByUserIds, clientNdsMap.get(client.getId()),
                            clientToGeoTreeType.get(clientId),
                            internalAdsProducts.get(realClientId), clientIdsUnderFreelancer,
                            enabledFeaturesByClientId.get(realClientId),
                            useNewAutopayCardLimit,
                            defaultCpmFrontpagePriceMap,
                            agencyLimRepsByClientIds,
                            isConversionMultipliersPopupEnabled,
                            phoneVerificationStatus,
                            isTurboLandingCreationEnabled(realClientId, enabledFeaturesByClientId.get(realClientId))
                    );
                })
                .collect(toList());
    }

    public static GdClientInfo createClientInfo(
            int shard,
            Client client,
            Map<Long, GdUserInfo> usersByIds,
            @Nullable Long otherManager,
            Map<Long, Client> relatedClientsByIds,
            Map<Long, Boolean> showAgencyContactsByUserIds,
            @Nullable ClientNds clientNds,
            GeoTreeType geoTreeType,
            @Nullable InternalAdsProduct internalAdsProduct,
            Set<Long> clientIdsUnderFreelancer,
            Set<String> availableFeatures,
            boolean useNewAutopayCardLimit,
            Map<CurrencyCode, BigDecimal> defaultCpmFrontpagePriceMap,
            Map<Long, Set<AgencyLimRep>> agencyLimRepsByClientId,
            boolean isConversionMultipliersPopupEnabled,
            PhoneVerificationStatus phoneVerificationStatus,
            boolean isTurboLandingCreationEnabled) {
        boolean isRecommendationsAllowed = client.getRole() == RbacRole.CLIENT;

        Long managerUserId = Optional.ofNullable(client.getPrimaryManagerUid())
                .filter(CommonUtils::isValidId)
                .or(() -> Optional.ofNullable(otherManager))
                .orElse(null);
        List<GdUserInfo> managers = Optional.ofNullable(managerUserId)
                .map(usersByIds::get)
                .map(Collections::singletonList)
                .orElse(emptyList());

        GdUserInfo chiefUser = usersByIds.get(client.getChiefUid());

        Long agencyUserId = isValidId(client.getAgencyUserId()) ? client.getAgencyUserId() : null;

        List<GdAgencyLimRepInfo> agencyLimRepsInfo;
        var agencyLimReps = agencyLimRepsByClientId.getOrDefault(client.getClientId(), Set.of());
        if (!agencyLimReps.isEmpty()) {
            var filteredLimRepUids = agencyLimReps.stream()
                    .filter(o -> o.getRepType() == RbacAgencyLimRepType.CHIEF || o.getRepType() == RbacAgencyLimRepType.LEGACY)
                    .collect(Collectors.toSet());
            agencyLimRepsInfo = filteredLimRepUids.stream()
                    .map(o -> toGdAgencyLimRepInfo(o, usersByIds.get(o.getUid()),
                            showAgencyContactsByUserIds.getOrDefault(o.getUid(), false)))
                    .sorted(Comparator.comparing(GdAgencyLimRepInfo::getName).thenComparing(GdAgencyLimRepInfo::getLogin))
                    .collect(Collectors.toList());
        } else {
            agencyLimRepsInfo = null;
        }

        GdAgencyInfo gdAgencyInfo = null;

        GdUserInfo gdAgencyUserInfo = ifNotNull(agencyUserId, id -> usersByIds.get(agencyUserId));
        //Если есть агентство
        if (gdAgencyUserInfo != null) {
            Client agencyClient = relatedClientsByIds.get(gdAgencyUserInfo.getClientId());
            //Показывать ли контакты представителя агентства работающего с клиентом
            Boolean showAgencyContacts = showAgencyContactsByUserIds.getOrDefault(agencyUserId, false);
            if (!showAgencyContacts) {
                GdUserInfo agencyChief = usersByIds.get(agencyClient.getChiefUid());
                //Показывать ли контакты главного представителя агентства
                Boolean showAgencyChiefContacts =
                        showAgencyContactsByUserIds.getOrDefault(agencyChief.getUserId(), false);
                gdAgencyInfo = toGdAgencyInfo(agencyClient, agencyChief, gdAgencyUserInfo, agencyLimRepsInfo,
                        showAgencyChiefContacts);
            } else {
                gdAgencyInfo = toGdAgencyInfo(agencyClient, gdAgencyUserInfo, null, agencyLimRepsInfo,
                        showAgencyContacts);
            }
        }

        GdClientAutoOverdraftInfo autoOverdraftInfo = createClientAutoOverdraftInfo(
                client, shard, clientNds);

        String internalAdProductName = ifNotNull(internalAdsProduct, InternalAdsProduct::getName);

        GdCurrencyDescription gdCurrencyDescription =
                ConstantsConverter.currencyDescription(availableFeatures,
                        useNewAutopayCardLimit,
                        client.getWorkCurrency().name(),
                        client.getWorkCurrency().getCurrency(), defaultCpmFrontpagePriceMap);

        return new GdClientInfo()
                .withId(client.getId())
                .withChiefUserId(client.getChiefUid())
                .withChiefUser(chiefUser)
                .withShard(shard)
                .withCountryRegionId(client.getCountryRegionId())
                .withWorkCurrency(client.getWorkCurrency())
                .withWorkCurrencyDescription(gdCurrencyDescription)
                .withNds(ifNotNull(clientNds, x -> x.getNds().asPercent()))
                .withCashBackBonus(nvl(client.getCashBackBonus(), BigDecimal.ZERO))
                .withCashBackAwaitingBonus(nvl(client.getCashBackAwaitingBonus(), BigDecimal.ZERO))
                .withManagerUserId(managerUserId)
                .withManagersInfo(managers)
                .withAgencyClientId(isValidId(client.getAgencyClientId()) ? client.getAgencyClientId() : null)
                .withAgencyUserId(agencyUserId)
                .withAgencyInfo(gdAgencyInfo)
                .withNonResident(client.getNonResident())
                .withAutoOverdraftInfo(autoOverdraftInfo)
                .withTranslocalGeotreeType(toGdGeotreeType(geoTreeType))
                .withAutoVideo(client.getAutoVideo())
                .withInternalAdProductName(internalAdProductName)
                .withMetrikaCountersNum(chiefUser.getMetrikaCountersNum())
                .withPhoneVerificationStatus(toGdPhoneVerificationStatus(phoneVerificationStatus))
                .withIsUnderFreelancer(clientIdsUnderFreelancer.contains(client.getId()))
                .withIsRecommendationsAllowed(isRecommendationsAllowed)
                .withIsProStategyViewEnabled(client.getIsProStrategyViewEnabled())
                .withVideohintsEnabled(!nvl(client.getVideohintsDisabled(), Boolean.FALSE))
                .withIsCashbackAvailable(hasCashbackAvailable(client, availableFeatures))
                .withIsBrand(client.getIsBrand())
                .withIsConversionMultipliersPopupEnabled(isConversionMultipliersPopupEnabled)
                .withIsNewPaymentWorkflowEnabled(isNewPaymentWorkflowEnabled(client))
                .withIsOnboardingWidgetEnabled(isOnboardingWidgetEnabled(client, availableFeatures))
                .withIsTurboLandingCreationEnabled(isTurboLandingCreationEnabled);
    }

    private boolean isTurboLandingCreationEnabled(ClientId clientId, Set<String> enabledFeatures) {
        return !enabledFeatures.contains(TURBO_LANDINGS_CREATION_DISABLED.getName()) || hasTurbolandings(clientId);
    }

    /**
     * Проверяет наличие у клиента активных турболендингов.
     * Если у нас в базе лендингов нет, значит, их нет в принципе.
     * Иначе запрашиваются лендинги в ТурбоКонструкторе.
     * Если Конструктор отвечает ошибкой, то считаем, что активные лендинги есть.
     */
    private boolean hasTurbolandings(ClientId clientId) {
        if (turboLandingService.getTurbolandigsByClientId(clientId).isEmpty()) {
            return false;
        }
        var activeTurbolandings = turboLandingService.externalGetActiveTurboLandingsOrNull(clientId);
        return activeTurbolandings == null || !activeTurbolandings.isEmpty();
    }

    private static boolean isOnboardingWidgetEnabled(Client client, Set<String> enabledFeatures) {
        return enabledFeatures.contains(ENABLE_ONBOARDING_WIZARD.getName()) &&
                LocalDateTime.now().minusDays(30).isBefore(client.getCreateDate());
    }

    private boolean isConversionMultipliersPopupEnabled(int shard, Client client, Set<String> availableFeatures) {
        return availableFeatures.contains(CONVERSION_MODIFIERS_POPUP_ENABLED.getName()) &&
                !client.getIsConversionMultipliersPopupDisabled() &&
                hasNotArchivedConversionCampaignIds(shard, client);
    }

    private boolean hasNotArchivedConversionCampaignIds(int shard, Client client) {
        return !getNotArchivedConversionCampaignIds(shard, ClientId.fromLong(client.getClientId())).isEmpty();
    }

    private static boolean isNewPaymentWorkflowEnabled(Client client) {
        return client.getNonResident().equals(false) &&
                client.getCountryRegionId().equals(RUSSIA_REGION_ID) &&
                client.getWorkCurrency().equals(CurrencyCode.RUB) &&
                client.getAgencyClientId() == null;
    }

    /**
     * Проверяет, доступна ли клиенту программа лояльности. Программа доступна, если
     * <ul>
     *     <li>Для клиента включена фича {@link FeatureName#CASHBACK_PAGE_ENABLED}</li>
     *     <li>Для клиента включена фича {@link FeatureName#CASHBACK_PAGE_ENABLED_FOR_BY_AND_KZ}</li>
     *     <li>Рабочая валюта клиента —
     *         {@link CurrencyCode#RUB рубль},
     *         {@link CurrencyCode#BYN белорусский рубль},
     *         {@link CurrencyCode#KZT тенге}
     *     </li>
     *     <li>Клиент из России, Белоруси, Казахстана или имеет глобальный регион</li>
     * </ul>
     * <p>
     * По данному значению интерфейс понимает, нужно ли показывать клиенту ссылки на визард кешбэков
     */
    private static boolean hasCashbackAvailable(Client client, Set<String> enabledFeatures) {
        // При правках здесь надо поправить has_cashback_available в perl/protected/Client.pm
        CurrencyCode currency = client.getWorkCurrency();
        Long regionId = client.getCountryRegionId();
        return enabledFeatures.contains(CASHBACK_PAGE_ENABLED.getName()) &&
                (regionId == GLOBAL_REGION_ID
                        || (regionId == RUSSIA_REGION_ID && currency == CurrencyCode.RUB)
                        || (regionId == KAZAKHSTAN_REGION_ID && currency == CurrencyCode.KZT)
                        || (regionId == BY_REGION_ID && currency == CurrencyCode.BYN)) &&
                !(client.getAgencyClientId() != null && client.getNonResident());
    }

    public List<GdClientMeasurerSystem> getMeasurerAccounts(@GraphQLNonNull GdClientInfo clientInfo) {
        var measurerSettingsList = clientMeasurerSettingsService.getByClientId(clientInfo.getId());
        return measurerSettingsList.stream()
                .filter(ClientDataService::filterExpiredMeasurerSettings)
                .map(s -> toGdClientMeasurerSystem(s.getClientMeasurerSystem()))
                .collect(toList());
    }

    public List<GdClientMeasurerAccountDetails> getMeasurerAccountDetails(User operator, GdClientInfo clientInfo) {
        if (!canEditUserSettings(operator, clientInfo.getId())) {
            return emptyList();
        }

        var measurerSettingsList = clientMeasurerSettingsService.getByClientId(clientInfo.getId());
        return measurerSettingsList.stream()
                .map(ClientDataConverter::toGdClientMeasurerAccountDetails)
                .filter(Objects::nonNull)
                .collect(toList());
    }

    private static boolean filterExpiredMeasurerSettings(ClientMeasurerSettings clientMeasurerSettings) {
        if (clientMeasurerSettings.getClientMeasurerSystem() == ClientMeasurerSystem.MEDIASCOPE) {
            var settings = fromJson(clientMeasurerSettings.getSettings(), GdClientMeasurerSettingsMediascope.class);
            return settings.getExpiresAt() > Instant.now().getEpochSecond();
        }

        return true;
    }

    public boolean canEditUserSettings(User operator, Long clientId) {
        boolean operatorIsFreelancer = freelanceService.isFreelancer(operator.getClientId());
        boolean operatorIsMcc = rbacService.isOperatorMccForClient(operator.getUid(), clientId);
        return OperatorClientRelationsHelper.calculateCanEditUserSettings(operator, clientId,
                operatorIsFreelancer || operatorIsMcc);
    }

    GdAvailableFields getAvailableFields(ClientId clientId) {
        return new GdAvailableFields()
                .withCampaignFields(getGdCampaignsAvailableFields());
    }

    private Set<GdCampaignsAvailableFields> getGdCampaignsAvailableFields() {
        return StreamEx.ofKeys(CAMPAIGNS_AVAILABLE_FIELDS_ATTRIBUTE_MAP)
                .filter(field -> attributeResolverService.resolve(CAMPAIGNS_AVAILABLE_FIELDS_ATTRIBUTE_MAP.get(field)))
                .toSet();
    }

    private static GdClientAutoOverdraftInfo createClientAutoOverdraftInfo(
            Client client,
            int shard,
            @Nullable ClientNds clientNds
    ) {
        GdClientAutoOverdraftInfo autoOverdraftInfo = new GdClientAutoOverdraftInfo()
                .withDebt(client.getDebt())
                .withStatusBalanceBanned(client.getStatusBalanceBanned())
                .withOverdraftLimit(client.getOverdraftLimit())
                .withAutoOverdraftLimit(client.getAutoOverdraftLimit())
                .withClientId(client.getId())
                .withShard(shard);

        autoOverdraftInfo.setOverdraftLimitWithNds(autoOverdraftInfo.getOverdraftLimit());

        // Убираем НДС с денежных значений, как и в других местах везде это делается перед запуском бизнес-логики
        // Логика для автоовердрафтов работает и так, и так (там используются только сложения, вычитания и min-max)
        // По-хорошему лучше всё сделать так, как было в перле: бизнес логика работает по полным суммам (с НДС),
        // а там, где нужно вывести пользователю без НДС, точечно выполняется вычитание НДС (в отдельных полях)
        if (clientNds != null) {
            applyNdsSubtruction(autoOverdraftInfo::getDebt, autoOverdraftInfo::setDebt, clientNds);
            applyNdsSubtruction(autoOverdraftInfo::getOverdraftLimit, autoOverdraftInfo::setOverdraftLimit, clientNds);
            applyNdsSubtruction(autoOverdraftInfo::getAutoOverdraftLimit, autoOverdraftInfo::setAutoOverdraftLimit,
                    clientNds);
        }

        // Без НДС
        autoOverdraftInfo.setOverdraftRest(calcOverdraftRest(autoOverdraftInfo));

        return autoOverdraftInfo;
    }

    @Nullable
    private static BigDecimal calcOverdraftRest(GdClientAutoOverdraftInfo autoOverdraftInfo) {
        if (autoOverdraftInfo.getOverdraftLimit() == null || autoOverdraftInfo.getDebt() == null) {
            return null;
        }
        return autoOverdraftInfo.getOverdraftLimit().subtract(autoOverdraftInfo.getDebt());
    }

    void checkOperatorCanReadClientInfo(Long operatorUserId, @Nullable Long longClientId) {
        if (longClientId == null) {
            throw new GridPublicException(GdExceptions.ACCESS_DENIED, "Client id is not defined",
                    SecurityTranslations.INSTANCE.accessDenied());
        }

        ClientId clientId = ClientId.fromLong(longClientId);
        long clientChiefUid = rbacService.getChiefByClientId(clientId);
        if (!rbacService.canRead(operatorUserId, clientChiefUid)
                && !isLimitedAgencyRepresentativeRequestsAgencyInfo(operatorUserId, clientChiefUid)) {
            throw new GridPublicException(GdExceptions.ACCESS_DENIED,
                    String.format("Operator with uid %s cannot access client with id %s", operatorUserId, clientId),
                    SecurityTranslations.INSTANCE.accessDenied());
        }
    }

    boolean isLimitedAgencyRepresentativeRequestsAgencyInfo(Long operatorUserId, Long clientChiefUid) {
        //Если клиент — агентство, и у оператора совпадает clientId, то разрешаем получить clientInfo
        return rbacService.getUidRole(clientChiefUid) == AGENCY
                && rbacService.getChief(operatorUserId) == clientChiefUid;
    }

    public GdClientFeatures getClientFeatures(ClientId clientId, User operator,
                                              Collection<GdiBaseCampaign> campaignsWithoutWallets) {
        return getClientFeatures(clientId, operator, () -> FEATURE_CALCULATOR.apply(campaignsWithoutWallets));
    }

    /**
     * Получить GdClientFeatures без заполненных полей из набора
     * {@link ru.yandex.direct.grid.processing.service.client.ClientFeatureCalculator#FEATURES_DEPENDS_ON_CAMPAIGNS}.
     */
    GdClientFeatures getClientFeaturesIndependentOfCampaigns(ClientId clientId, User operator) {
        return getClientFeatures(clientId, operator, GdClientFeatures::new);
    }

    @SuppressWarnings({"checkstyle:methodlength", "checkstyle:linelength"})
    private GdClientFeatures getClientFeatures(ClientId clientId, User operator,
                                               Supplier<GdClientFeatures> gdClientFeaturesCreator) {
        // Получаем список используемых клиентом фич
        Set<String> availableFeatures = featureService.getEnabledForClientId(clientId);

        boolean isSuper = UserUtil.hasOneOfRoles(operator, SUPER, SUPERREADER, MANAGER, PLACER);
        boolean operatorHasInternalAdRole = UserUtil.hasOneOfRoles(operator,
                INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER);

        List<GdiRecommendationType> availableTypes = EnumSet.allOf(GdiRecommendationType.class).stream()
                .filter(type -> isSuper || type.getFeature() == null || availableFeatures.contains(type.getFeature().getName()))
                .sorted(Comparator.comparing(GdiRecommendationType::getId))
                .collect(toList());

        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);
        boolean ucMainFeatureEnabled =
                availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_ENABLED_FOR_UAC.getName());
        boolean ucDogfoodingDisabledByFeature =
                availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED.getName());
        boolean operatorIsClient = Objects.equals(operator.getClientId(), clientId);
        boolean ucDogfoodingEnabled = operatorIsClient && !ucDogfoodingDisabledByFeature && requestFromInternalNetwork;
        boolean ucEnabled = ucMainFeatureEnabled || ucDogfoodingEnabled;

        GdClientFeatures clientFeatures = gdClientFeaturesCreator.get()
                .withIsSocialAdvertising(availableFeatures.contains(FeatureName.SOCIAL_ADVERTISING.getName()))
                .withIsSocialAdvertisingPayable(availableFeatures.contains(FeatureName.SOCIAL_ADVERTISING_PAYABLE.getName()))
                .withIsVideoConstructorEnabled(availableFeatures.contains(FeatureName.VIDEO_CONSTRUCTOR_ENABLED.getName()))
                .withIsVideoConstructorCreateFromScratchEnabled(availableFeatures.contains(FeatureName.VIDEO_CONSTRUCTOR_CREATE_FROM_SCRATCH_ENABLED.getName()))
                .withIsVideoConstructorFeedEnabled(availableFeatures.contains(FeatureName.VIDEO_CONSTRUCTOR_FEED_ENABLED.getName()))
                .withIsRetargetingAndAudienceImprovementsEnabled(availableFeatures.contains(FeatureName.RETARGETING_AND_AUDIENCE_IMPROVEMENTS.getName()))
                .withIsVideoConstructorBeruruTemplateEnabled(availableFeatures.contains(FeatureName.VIDEO_CONSTRUCTOR_BERURU_TEMPLATE_ENABLED.getName()))
                .withIsShowOldHeaderLinksForDna(availableFeatures.contains(FeatureName.SHOW_OLD_HEADER_LINKS_FOR_DNA.getName()))
                .withIsShowPublicFeaturesForDna(availableFeatures.contains(FeatureName.SHOW_PUBLIC_FEATURES_FOR_DNA.getName()))
                .withIsGridEnabled(availableFeatures.contains(FeatureName.GRID.getName()))
                .withIsUserechoEnabled(availableFeatures.contains(FeatureName.USERECHO_ENABLE_FOR_DNA.getName()))
                .withIsCpcVideoAllowed(availableFeatures.contains(FeatureName.CPC_VIDEO_BANNER.getName()))
                .withIsExperimentRetargetingConditionsCreatingOnTextCampaignModifyEnabled(availableFeatures.contains(FeatureName.EXPERIMENT_RET_CONDITIONS_CREATING_ON_TEXT_CAMPAIGNS_MODIFY_IN_JAVA_FOR_DNA.getName()))
                .withIsFirstExpandedSidebarForDna(availableFeatures.contains(FeatureName.FIRST_EXPANDED_SIDEBAR_FOR_DNA.getName()))
                .withIsSmartScrollGridForDna(availableFeatures.contains(FeatureName.SMART_SCROLL_GRID_FOR_DNA.getName()))
                .withIsReplacedButtonViewToButtonColumnsInGrid(availableFeatures.contains(FeatureName.REPLACED_BUTTON_VIEW_TO_BUTTON_COLUMNS_IN_GRID_FOR_DNA.getName()))
                .withIsHeaderInfoMenuVisible(availableFeatures.contains(FeatureName.HEADER_INFO_MENU_VISIBLE_FOR_DNA.getName()))
                .withIsSmartSortInGrid(availableFeatures.contains(FeatureName.SMART_SORT_IN_GRID_FOR_DNA.getName()))
                .withIsImageGalleryWithServicePanel(availableFeatures.contains(FeatureName.IMAGE_GALLERY_WITH_SERVICE_PANEL_FOR_DNA.getName()))
                .withIsSmartStrategyContextOnly(availableFeatures.contains(FeatureName.SMART_STRATEGY_CONTEXT_ONLY_FOR_DNA.getName()))
                .withIsPayForConversionVisibleInFilterOptimizationEntity(availableFeatures.contains(FeatureName.PAY_FOR_CONVERSION_VISIBLE_IN_FILTER_OPTIMIZATION_ENTITY_FOR_DNA.getName()))
                .withIsMassEditRegionsEnabled(availableFeatures.contains(FeatureName.MASS_EDIT_REGIONS_IN_JAVA_FOR_DNA.getName()))
                .withIsTagsOnGroupEnabled(availableFeatures.contains(FeatureName.TAGS_ON_GROUP_ALLOWED_IN_DNA.getName()))
                .withRelevanceMatchForNewGroupsEnabled(
                        availableFeatures.contains(FeatureName.RELEVANCE_MATCH_FOR_NEW_GROUPS_ENABLED.getName()))
                .withIsInternalAdsAllowed(internalAdsProductService.clientCanHaveInternalAdCampaigns(clientId))
                .withIsBannerUpdateAllowed(availableFeatures.contains(FeatureName.BANNER_UPDATE_IN_JAVA_FOR_DNA.getName()))
                .withEnabledRecommendationTypes(availableTypes)
                .withIsMinusWordsLibEnabled(!operatorHasInternalAdRole
                        && availableFeatures.contains(FeatureName.MINUS_WORDS_LIB.getName()))
                .withEnableInterfaceGridCache(availableFeatures.contains(FeatureName.ENABLE_INTERFACE_GRID_CACHE.getName()))
                .withIsNewBannerPreviewForDna(availableFeatures.contains(FeatureName.NEW_BANNER_PREVIEW_FOR_DNA.getName()))
                .withIsShowCashBackBonus(availableFeatures.contains(FeatureName.SHOW_CASHBACK_BONUS.getName()))
                .withIsAdGroupIndivisibleDraftStatus(availableFeatures.contains(FeatureName.ADGROUP_INDIVISIBLE_DRAFT_STATUS.getName()))
                .withIsAutotargetingKeywordPrefixAllowed(availableFeatures.contains(FeatureName.AUTOTARGETING_KEYWORD_PREFIX_ALLOWED.getName()))
                .withIsBannerPricesEnabled(availableFeatures.contains(FeatureName.BANNER_PRICES.getName()))
                .withIsBannerPricesOnGridEnabled(availableFeatures.contains(FeatureName.BANNER_PRICES_ON_GRID.getName()))
                .withIsTurboForSerpWizardAllowed(availableFeatures.contains(FeatureName.TURBO_FOR_SERP_WIZARD.getName()))
                .withIsGoalsStatEnabled(availableFeatures.contains(FeatureName.SHOW_GOALS_STAT_IN_GRID.getName()))
                .withIsCpmCampaignsEnabled(availableFeatures.contains(FeatureName.SHOW_CPM_BANNER_CAMPAIGNS_IN_GRID.getName()))
                .withIsCpmGeoproductEnabled(availableFeatures.contains(FeatureName.CPM_GEOPRODUCT_ENABLED.getName()))
                .withIsCpmGeoPinProductEnabled(availableFeatures.contains(FeatureName.CPM_GEO_PIN_PRODUCT_ENABLED.getName()))
                .withIsCpmGeoPinReachRequestEnabled(availableFeatures.contains(FeatureName.CPM_GEO_PIN_REACH_REQUEST_ENABLED.getName()))
                .withIsCpmGeoPinTaximeterEnabled(availableFeatures.contains(FeatureName.CPM_GEO_PIN_TAXIMETER_ENABLED.getName()))
                .withIsEmptyRetCondInCpmGroupEnabled(availableFeatures.contains(FeatureName.CPM_BANNER_ENABLE_EMPTY_RET_COND_JSON.getName()))
                .withIsCpmDealsEnabled(availableFeatures.contains(FeatureName.CPM_DEALS.getName()))
                .withIsContentPromotionVideoEnabled(availableFeatures.contains(FeatureName.CONTENT_PROMOTION_VIDEO.getName()))
                .withIsContentPromotionCollectionEnabled(availableFeatures.contains(FeatureName.CONTENT_PROMOTION_COLLECTION.getName()))
                .withIsConversionCenterEnabled(availableFeatures.contains(FeatureName.ENABLE_CONVERSION_CENTER.getName()))
                .withIsBidCorrectionSearchEnabled(availableFeatures.contains((FeatureName.BID_CORRECTION_SEARCH_ENABLED.getName())))
                .withIsTextCampaign8SitelinksEnabled(availableFeatures.contains((FeatureName.TEXT_CAMPAIGN_8_SITELINKS.getName())))
                .withIsB2bBalanceCartEnabled(availableFeatures.contains(FeatureName.B2B_BALANCE_CART.getName()))
                .withIsCpmVideoSeveralPixelsEnabled(availableFeatures.contains(FeatureName.CPM_VIDEO_SEVERAL_PIXELS_ENABLED.getName()))
                .withIsTycoonOrganizationsEnabled(availableFeatures.contains(FeatureName.TYCOON_ORGANIZATIONS.getName()))
                .withIsShowAutoAssignedOrganizationsEnabled(availableFeatures.contains(FeatureName.SHOW_AUTO_ASSIGNED_ORGANIZATIONS.getName()))
                .withIsCampaignDefaultOrganizationsEnabled(availableFeatures.contains(FeatureName.CAMPAIGN_DEFAULT_ORGANIZATIONS.getName()))
                .withIsCpmIndoorSegmentsEnabled(availableFeatures.contains(FeatureName.INDOOR_SEGMENTS.getName()))
                .withHasAbSegments(availableFeatures.contains(FeatureName.AB_SEGMENTS.getName()))
                .withIsDefaultAutobudgetAvgCpaEnabled(availableFeatures.contains(FeatureName.DEFAULT_AUTOBUDGET_AVG_CPA.getName()))
                .withIsAdmetricaMeasurerEnabled(availableFeatures.contains(FeatureName.ADMETRICA_MEASURER.getName()))
                .withIsAdlooxMeasurerEnabled(availableFeatures.contains(FeatureName.ADLOOX_MEASURER.getName()))
                .withIsAdriverMeasurerEnabled(availableFeatures.contains(FeatureName.ADRIVER_MEASURER.getName()))
                .withIsDvMeasurerEnabled(availableFeatures.contains(FeatureName.DV_MEASURER.getName()))
                .withIsIntegralAdScienceMeasurerEnabled(availableFeatures.contains(FeatureName.INTEGRAL_AD_SCIENCE_MEASURER.getName()))
                .withIsIasMeasurerEnabled(availableFeatures.contains(FeatureName.IAS_MEASURER.getName()))
                .withIsMediascopeMeasurerEnabled(availableFeatures.contains(FeatureName.MEDIASCOPE_MEASURER.getName()))
                .withIsTelegramEnabled(availableFeatures.contains(FeatureName.TELEGRAM_ENABLED.getName()))
                .withIsMoatMeasurerEnabled(availableFeatures.contains(FeatureName.MOAT_MEASURER.getName()))
                .withIsMoatCampMeasurerEnabled(availableFeatures.contains(FeatureName.MOAT_MEASURER_CAMP.getName()))
                .withIsSizmekMeasurerEnabled(availableFeatures.contains(FeatureName.SIZMEK_MEASURER.getName()))
                .withIsWeboramaMeasurerEnabled(availableFeatures.contains(FeatureName.WEBORAMA_MEASURER.getName()))
                .withIsOmiMeasurerEnabled(availableFeatures.contains(FeatureName.OMI_MEASURER.getName()))
                .withIsTextBannerInterestsRetCondEnabled(availableFeatures.contains(FeatureName.TEXT_BANNER_INTERESTS_RET_COND_ENABLED.getName()))
                .withIsShowCampLinkByCellHoverEnabledForDna(availableFeatures.contains(FeatureName.SHOW_CAMP_LINK_BY_CELL_HOVER_ENABLED_FOR_DNA.getName()))
                .withIsShowCampLinkInPopupEnabledForDna(availableFeatures.contains(FeatureName.SHOW_CAMP_LINK_IN_POPUP_ENABLED_FOR_DNA.getName()))
                .withIsShowCampLinkInGridCellEnabledForDna(availableFeatures.contains(FeatureName.SHOW_CAMP_LINK_IN_GRID_CELL_ENABLED_FOR_DNA.getName()))
                .withIsShowCampLinkByNameClickEnabledForDna(availableFeatures.contains(FeatureName.SHOW_CAMP_LINK_BY_NAME_CLICK_ENABLED_FOR_DNA.getName()))
                .withIsTnsEnabled(availableFeatures.contains(FeatureName.TNS_ENABLED.getName()))
                .withIsBrandLiftAllowed(availableFeatures.contains(FeatureName.BRAND_LIFT.getName()))
                .withIsBrandLiftMultiCampAllowed(true)
                .withIsBrandLiftCpmYndxFrontpageAllowed(availableFeatures.contains(FeatureName.BRAND_LIFT_CPM_YNDX_FRONTPAGE.getName()))
                .withIsTouchDirectEnabled(availableFeatures.contains(FeatureName.TOUCH_DIRECT_ENABLED.getName()))
                .withIsHyperlocalGeoForTouchCampaignsEnabled(availableFeatures.contains(FeatureName.HYPERLOCAL_GEO_FOR_TOUCH_CAMPAIGNS_ENABLED.getName()))
                .withIsHyperlocalGeoForDaasCampaignsEnabled(availableFeatures.contains(FeatureName.HYPERLOCAL_GEO_FOR_DAAS_CAMPAIGNS_ENABLED.getName()))
                .withIsHyperlocalGeoForDesktopCampaignsEnabled(availableFeatures.contains(FeatureName.HYPERLOCAL_GEO_FOR_DESKTOP_CAMPAIGNS_ENABLED.getName()))
                .withIsOrganizationsForTouchCampaignsEnabled(availableFeatures.contains(FeatureName.ORGANIZATIONS_FOR_TOUCH_CAMPAIGNS_ENABLED.getName()))
                .withIsTouchTurboQuickStartEnabled(availableFeatures.contains(FeatureName.TOUCH_TURBO_QUICK_START_ENABLED.getName()))
                .withIsImpressionStandardTimeAllowed(availableFeatures.contains(FeatureName.IMPRESSION_STANDARD_TIME.getName()))
                .withIsEshowsSettingsExtendedAllowed(availableFeatures.contains(FeatureName.ESHOWS_SETTINGS_EXTENDED.getName()))
                .withIsContentDurationBidModifierAllowed(availableFeatures.contains(FeatureName.CONTENT_DURATION_BID_MODIFIER.getName()))
                .withIsInAppEventsInRmpEnabled(availableFeatures.contains(FeatureName.IN_APP_EVENTS_IN_RMP_ENABLED.getName()))
                .withIsTargetTagsAllowed(availableFeatures.contains(FeatureName.TARGET_TAGS_ALLOWED.getName()))
                .withIsAloneTrafaretOptionEnabled(availableFeatures.contains(FeatureName.ALONE_TRAFARET_OPTION_ENABLED.getName()))
                .withHasTurboSmarts(availableFeatures.contains(FeatureName.TURBO_SMARTS.getName()))
                .withIsRequireFiltrationByDontShowDomainsOptionEnabled(availableFeatures.contains(FeatureName.CAN_REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS.getName()))
                .withIsCpmYndxFrontpageOnGridEnabled(availableFeatures.contains(FeatureName.CPM_YNDX_FRONTPAGE_ON_GRID.getName()))
                .withIsCpmAudioDisable(availableFeatures.contains(FeatureName.DISABLE_CPM_AUDIO.getName()))
                .withIsMediascopeMeasurerEnabled(availableFeatures.contains(FeatureName.MEDIASCOPE_MEASURER.getName()))
                .withIsContentPromotionVideoInGridSupported(availableFeatures.contains(FeatureName.CONTENT_PROMOTION_VIDEO_ON_GRID.getName()))
                .withIsContentPromotionCollectionInGridSupported(availableFeatures.contains(FeatureName.CONTENT_PROMOTION_COLLECTIONS_ON_GRID.getName()))
                .withIsContentPromotionCampaignInDnaSupported(isAnyFeatureEnabled(clientId,
                        asList(FeatureName.CONTENT_PROMOTION_VIDEO_ON_GRID,
                                FeatureName.CONTENT_PROMOTION_COLLECTIONS_ON_GRID)))
                .withIsCpaPayForConversionsStrategyMobileAppsEnabled(availableFeatures.contains(FeatureName.CPA_PAY_FOR_CONVERSIONS_MOBILE_APPS_ALLOWED.getName()))
                .withIsOverlayCreativesEnabled(availableFeatures.contains(FeatureName.OVERLAY_CREATIVES.getName()))
                .withIsZeroSpeedPageForGeoproductEnabled(availableFeatures.contains(FeatureName.ZERO_SPEED_PAGE_ENABLED_FOR_GEOPRODUCT.getName()))
                .withIsBrandSafetyBaseCategoriesCpcEnabled(availableFeatures.contains(FeatureName.BRANDSAFETY_BASE_CATEGORIES_CPC.getName()))
                .withIsBrandSafetyBaseCategoriesCpmEnabled(availableFeatures.contains(FeatureName.BRANDSAFETY_BASE_CATEGORIES_CPM.getName()))
                .withIsBrandSafetyAdditionalCategoriesEnabled(availableFeatures.contains(FeatureName.BRANDSAFETY_ADDITIONAL_CATEGORIES.getName()))
                .withIsBrandSafetyBaseCategoriesStatEnabled(availableFeatures.contains(FeatureName.BRANDSAFETY_BASE_CATEGORIES_STAT.getName()))
                .withHideOldTextCampaignEditForDna(availableFeatures.contains(FeatureName.HIDE_OLD_TEXT_CAMPAIGN_EDIT_FOR_DNA.getName()))
                .withIsLandingsWizardAllowed(availableFeatures.contains(FeatureName.LANDINGS_WIZARD_ALLOWED.getName()))
                .withIsHonestAddTextAdGroupAllowed(availableFeatures.contains(FeatureName.HONEST_ADD_TEXT_ADGROUP_ALLOWED.getName()))
                .withIsAllGoalsOptimizationForDnaDisabled(availableFeatures.contains(FeatureName.DISABLE_ALL_GOALS_OPTIMIZATION_FOR_DNA.getName()))
                .withIsGeoRateCorrectionsAllowedForDna(availableFeatures.contains(FeatureName.GEO_RATE_CORRECTIONS_ALLOWED_FOR_DNA.getName()))
                .withIsDisableAnyDomainsAllowed(availableFeatures.contains(FeatureName.DISABLE_ANY_DOMAINS_ALLOWED.getName()))
                .withIsDisableMailRuDomainsAllowed(true)
                .withIsDisableNumberIdAndShortBundleIdAllowed(availableFeatures.contains(FeatureName.DISABLE_NUMBER_ID_AND_SHORT_BUNDLE_ID_ALLOWED.getName()))
                .withIsDefaultMobilePreviewEnabledForDna(availableFeatures.contains(FeatureName.DEFAULT_MOBILE_PREVIEW_ENABLED_FOR_DNA.getName()))
                .withIsUniversalCampaignsEnabled(availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_ENABLED.getName()))
                .withIsUniversalCampaignsEnabledForUac(ucEnabled)
                .withShowConversionsForecastForUniversalCampaigns(availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_SHOW_CONVERSIONS_FORECAST.getName()))
                .withIsAimsInTextCampaignsEnabled(availableFeatures.contains(FeatureName.AIMS_IN_TEXT_CAMPAIGNS.getName()))
                .withIsNewGeoDefaultAndSuggestEnabled(availableFeatures.contains(FeatureName.NEW_GEO_DEFAULT_AND_SUGGEST.getName()))
                .withIsInterfaceControlGroupEnabled(availableFeatures.contains(FeatureName.INTERFACE_CONTROL_GROUP.getName()))
                .withIsTurboAppAllowed(availableFeatures.contains(FeatureName.TURBO_APP_ALLOWED.getName()))
                .withIsCollectingVerifiedPhonesEnabled(
                        availableFeatures.contains(FeatureName.COLLECTING_VERIFIED_PHONES_FOR_OLD_CLIENTS.getName()) ||
                                availableFeatures.contains(FeatureName.COLLECTING_VERIFIED_PHONES_FOR_NEW_CLIENTS.getName()))
                .withIsDefaultAutobudgetAvgClickWithWeekBudgetEnabled(availableFeatures.contains(FeatureName.DEFAULT_AUTOBUDGET_AVG_CLICK_WITH_WEEK_BUDGET.getName()))
                .withIsDefaultAutobudgetRoiEnabled(availableFeatures.contains(FeatureName.DEFAULT_AUTOBUDGET_ROI.getName()))
                .withIsAutoBudgetMeaningfulGoalsValuesFromMetrikaEnabled(availableFeatures.contains(FeatureName.AUTO_BUDGET_MEANINGFUL_GOALS_VALUES_FROM_METRIKA.getName()))
                .withIsDemographyBidModifierUnknownAgeAllowed(availableFeatures.contains(FeatureName.DEMOGRAPHY_BID_MODIFIER_UNKNOWN_AGE_ALLOWED.getName()))
                .withIsCpmYndxFrontpageProfileEnabled(availableFeatures.contains(FeatureName.CPM_YNDX_FRONTPAGE_PROFILE.getName()))
                .withIsMobileOsBidModifierEnabled(true)
                .withIsCpcDeviceModifiersAllowed(true)
                .withIsEditCpmYndxFrontpageInDnaSupported(availableFeatures.contains(FeatureName.EDIT_CPM_YNDX_FRONTPAGE_IN_DNA.getName()))
                .withIsMobileAppGoalsForTextCampaignAllowed(availableFeatures.contains(FeatureName.MOBILE_APP_GOALS_FOR_TEXT_CAMPAIGN_ALLOWED.getName()))
                .withIsMobileAppGoalsForTextCampaignStatisticsEnabled(availableFeatures.contains(FeatureName.MOBILE_APP_GOALS_FOR_TEXT_CAMPAIGN_STATISTICS_ENABLED.getName()))
                .withIsMobileAppGoalsForTextCampaignStrategyEnabled(availableFeatures.contains(FeatureName.MOBILE_APP_GOALS_FOR_TEXT_CAMPAIGN_STRATEGY_ENABLED.getName()))
                .withIsCpaPayForConversionsStrategyExtendedModeEnabled(availableFeatures.contains(FeatureName.CPA_PAY_FOR_CONVERSIONS_EXTENDED_MODE.getName()))
                .withIsMetrikaCountersAccessValidationOnSaveCampaignEnabled(availableFeatures.contains(FeatureName.METRIKA_COUNTERS_ACCESS_VALIDATION_ON_SAVE_CAMPAIGN_ENABLED.getName()))
                .withIsAddingOrganizationsCountersToCampaignOnAddingOrganizationsToAdsEnabled(availableFeatures.contains(FeatureName.ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS.getName()))
                .withUcDesignForDnaEditEnabled(availableFeatures.contains(FeatureName.UC_DESIGN_FOR_DNA_EDIT_ENABLED.getName()))
                .withUcDesignForDnaGridEnabled(availableFeatures.contains(FeatureName.UC_DESIGN_FOR_DNA_GRID_ENABLED.getName()))
                // данный ключ используется фронтом для отрисовки пункта меню "Интерактивный шаблон" при создании
                // креатива
                // в открывающемся попапе можно выбрать более чем из одного шаблона
                // каждая из перечисленных фич отвечает за свой шаблон
                // но пункт "Интерактивный шаблон" общий для всех шаблонов
                .withBannerstorageTemplate1013Enabled(
                        availableFeatures.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1013_ENABLED.getName())
                                || availableFeatures.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1042_ENABLED.getName())
                                || availableFeatures.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1058_ENABLED.getName())
                                || availableFeatures.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1060_ENABLED.getName())
                                || availableFeatures.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1062_ENABLED.getName()))
                .withIsEnablePreferVCardOverPermalink(availableFeatures.contains(FeatureName.IS_ENABLE_PREFER_V_CARD_OVER_PERMALINK.getName()))
                .withIsChangeBannerOrganizationOnDefaultCampaignOrganizationChange(availableFeatures.contains(FeatureName.CHANGE_BANNER_ORGANIZATION_ON_DEFAULT_CAMPAIGN_ORGANIZATION_CHANGE.getName()))
                .withIsCpmPriceCampaignEnabled(availableFeatures.contains(FeatureName.CPM_PRICE_CAMPAIGN.getName()))
                .withIsSupportChatEnabled(availableFeatures.contains(FeatureName.SUPPORT_CHAT.getName()))
                .withIsCpmPriceBannerAdditionalHrefsAllowed(availableFeatures.contains(FeatureName.CPM_PRICE_BANNER_ADDITIONAL_HREFS_ALLOWED.getName()))
                .withIsMassEditCampaignMetrikaCountersEnabled(availableFeatures
                        .contains(FeatureName.MASS_EDIT_METRIKA_COUNTERS_IN_JAVA_FOR_DNA.getName()))
                .withPreloadGridDataEnabled(availableFeatures
                        .contains(FeatureName.PRELOAD_GRID_DATA_ENABLED_FOR_DNA.getName()))
                .withPreloadGridDataOnlyForOneObjectEnabled(availableFeatures.contains(FeatureName.PRELOAD_GRID_DATA_ONLY_FOR_ONE_OBJECT_ENABLED_FOR_DNA.getName()))
                .withIsLalSegmentsEnabled(availableFeatures.contains(FeatureName.LAL_SEGMENTS_ENABLED.getName()))
                .withIsRetargetingOnlyLalEnabled(availableFeatures.contains(FeatureName.RETARGETING_ONLY_LAL_ENABLED.getName()))
                .withIsCreateAndEditRetargetingConditionsInGridEnabled(availableFeatures.contains(FeatureName.CREATE_AND_EDIT_RETARGETING_CONDITIONS_IN_GRID_ENABLED.getName()))
                .withTargetingIsNotRequiredForCpmGeoproductGroup(availableFeatures.contains(FeatureName.TARGETING_IS_NOT_REQUIRED_FOR_CPM_GEOPRODUCT_GROUP.getName()))
                .withIsContentCategoryTargetingCpcEnabled(availableFeatures.contains(FeatureName.CONTENT_CATEGORY_TARGETING_CPC.getName()))
                .withIsContentCategoryTargetingCpmExtendedEnabled(availableFeatures.contains(FeatureName.CONTENT_CATEGORY_TARGETING_CPM_EXTENDED.getName()))
                .withIsContentCategoryTargetingShowChildrenEnabled(availableFeatures.contains(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_CHILDREN.getName()))
                .withIsContentCategoryTargetingCpmBannerEnabled(availableFeatures.contains(FeatureName.CONTENT_CATEGORY_TARGETING_CPM_BANNER.getName()))
                .withIsTelephonyAllowed(availableFeatures.contains(FeatureName.TELEPHONY_ALLOWED.getName()))
                .withIsAbcNumberTelephonyAllowed(availableFeatures.contains(FeatureName.ABC_NUMBER_TELEPHONY_ALLOWED.getName()))
                .withIsCalltrackingOnSiteAllowed(availableFeatures.contains(FeatureName.CALLTRACKING_ON_SITE_ALLOWED.getName()))
                .withUseSeparateCounterBlock(availableFeatures.contains(FeatureName.SEPARATE_COUNTER_BLOCK.getName()))
                .withIsGoalsFromOrphanedOrgsAllowed(availableFeatures.contains(FeatureName.GOALS_FROM_ORPHANED_ORGS_ALLOWED.getName()))
                .withIsGoalsFromAllOrgsAllowed(availableFeatures.contains(FeatureName.GOALS_FROM_ALL_ORGS_ALLOWED.getName()))
                .withIsNewSpravIframeEnabled(availableFeatures.contains(FeatureName.NEW_SPRAV_IFRAME_ENABLED.getName()))
                .withIsTgoPlayableEnabled(true)
                .withIsCanvasRangeRatioCpcEnabled(availableFeatures.contains(FeatureName.CANVAS_RANGE_RATIO_CPC.getName()))
                .withIsCanvasRangeRatioEnabled(availableFeatures.contains(FeatureName.CANVAS_RANGE_RATIO.getName()))
                .withIsCpaTurbolandingsModerationRequired(availableFeatures.contains(FeatureName.CPA_PAY_FOR_CONVERSIONS_TURBOLANDINGS_MODERATION_REQUIRED.getName()))
                .withIsNewTurboPromoEnabled(availableFeatures.contains(FeatureName.NEW_TURBO_PROMO_ENABLED.getName()))
                .withIsAssetCpmVideoTextEnabled(availableFeatures.contains(FeatureName.ASSET_CPM_VIDEO_TEXT.getName()))
                .withIsAssetCpmVideoLogoEnabled(availableFeatures.contains(FeatureName.ASSET_CPM_VIDEO_LOGO.getName()))
                .withIsAssetCpmVideoButtonEnabled(availableFeatures.contains(FeatureName.ASSET_CPM_VIDEO_BUTTON.getName()))
                .withIsAssetGoTextEnabled(availableFeatures.contains(FeatureName.ASSET_GO_TEXT.getName()))
                .withIsNewGoalsInterfaceInGridEnabledForDna(availableFeatures.contains(FeatureName.NEW_GOALS_INTERFACE_IN_GRID_ENABLED_FOR_DNA.getName()))
                .withIsAssetGoLogoEnabled(availableFeatures.contains(FeatureName.ASSET_GO_LOGO.getName()))
                .withOrganizationPreviewAvailableForDna(availableFeatures.contains(FeatureName.ORGANIZATION_PREVIEW_AVAILABLE_FOR_DNA.getName()))
                .withSkipBrandSurveyConditionsCheck(availableFeatures.contains(FeatureName.SKIP_BRAND_SURVEY_CONDITIONS_CHECK.getName()))
                .withIsAssetGoButtonEnabled(availableFeatures.contains(FeatureName.ASSET_GO_BUTTON.getName()))
                .withIsAssetTgoLogoEnabled(availableFeatures.contains(FeatureName.ASSET_TGO_LOGO.getName()))
                .withIsAssetTgoButtonEnabled(availableFeatures.contains(FeatureName.ASSET_TGO_BUTTON.getName()))
                .withIsAssetBannerNameRegularEnabled(availableFeatures.contains(FeatureName.ASSET_BANNER_NAME_REGULAR.getName()))
                .withIsAssetBigKingImageEnabled(availableFeatures.contains(FeatureName.ASSET_BIG_KING_IMAGE.getName()))
                .withIsAssetBigKingCompanyNameEnabled(availableFeatures.contains(FeatureName.ASSET_BIG_KING_COMPANY_NAME.getName()))
                .withIsAssetBigKingBackgroundColorEnabled(availableFeatures.contains(FeatureName.ASSET_BIG_KING_BACKGROUND_COLOR.getName()))
                .withIsAssetBigKingExpandedPreviewEnabled(availableFeatures.contains(FeatureName.ASSET_BIG_KING_EXPANDED_PREVIEW.getName()))
                .withIsAssetBannerNameUnconditionalEnabled(availableFeatures.contains(FeatureName.ASSET_BANNER_NAME_UNCONDITIONAL.getName()))
                .withIsRequireFiltrationByDontShowDomainsInCpm(availableFeatures.contains(FeatureName.CAN_REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS_IN_CPM.getName()))
                .withCanHaveInternalDontShowDomains(availableFeatures.contains(FeatureName.CAN_HAVE_INTERNAL_DONT_SHOW_DOMAINS.getName()))
                .withShowInternalDontShowDomainsWarning(availableFeatures.contains(FeatureName.SHOW_INTERNAL_DONT_SHOW_DOMAINS_WARNING.getName()))
                .withIsModerationOfferEnabledForDna(availableFeatures.contains(FeatureName.MODERATION_OFFER_ENABLED_FOR_DNA.getName()))
                .withUseAggressiveOrganizationSuggestForDna(availableFeatures.contains(FeatureName.USE_AGGRESSIVE_ORGANIZATION_SUGGEST_FOR_DNA.getName()))
                .withIsTurboLandingAutoLinkingEnabled(availableFeatures.contains(FeatureName.TURBOLANDING_AUTOLINKING_ENABLED.getName()))
                .withIsSkipTypeSelectionOnFirstCampaignForDna(availableFeatures.contains(FeatureName.SKIP_TYPE_SELECTION_ON_FIRST_CAMPAIGN_DNA.getName()))
                .withIsFaceliftDisabledForDna(availableFeatures.contains(FeatureName.FACELIFT_DISABLED_FOR_DNA.getName()))
                .withIsNewFeedbackFormEnabled(availableFeatures.contains(FeatureName.NEW_FEEDBACK_FORM.getName()))
                .withIsOrderByStatusEnabled(availableFeatures.contains(FeatureName.ORDER_BY_STATUS.getName()))
                .withIsNewCampaignsStatusFilterEnabled(availableFeatures.contains(FeatureName.NEW_CAMPAIGNS_STATUS_FILTER.getName()))
                .withIsHideAddButtonOnFirstCreateEnabled(availableFeatures.contains(FeatureName.HIDE_ADD_BUTTON_ON_FIRST_CREATE.getName()))
                .withIsNewFeedbackFormAutoOpeningEnabled(availableFeatures.contains(FeatureName.NEW_FEEDBACK_FORM_AUTO_OPENING.getName()))
                .withIsFeedbackAnswerCheckmarkEnabled(availableFeatures.contains(FeatureName.FEEDBACK_ANSWER_CHECKMARK.getName()))
                .withIsSupportChatNotificationEnabled(availableFeatures.contains(FeatureName.SUPPORT_CHAT_NOTIFICATION.getName()))
                .withIsGridBannersNewPreviewEnabled(availableFeatures.contains(FeatureName.GRID_BANNERS_NEW_PREVIEW.getName()))
                .withIsInBannerEnabled(availableFeatures.contains(FeatureName.IN_BANNER_CREATIVES_SUPPORT.getName()))
                .withCpaStrategyInCpmBannerCampaignEnabled(availableFeatures.contains(FeatureName.CPA_STRATEGY_IN_CPM_BANNER_CAMPAIGN_ENABLED.getName()))
                .withCpvStrategiesEnabled(availableFeatures.contains(FeatureName.CPV_STRATEGIES_ENABLED.getName()))
                .withIsWebsiteOnCampaignEnabledForDna(availableFeatures.contains(FeatureName.WEBSITE_ON_CAMPAIGN_ENABLED_FOR_DNA.getName()))
                .withIsWebsiteOnCampaignAllEnabledForDna(availableFeatures.contains(FeatureName.WEBSITE_ON_CAMPAIGN_ALL_ENABLED_FOR_DNA.getName()))
                .withIsSupportChatFaceliftEnabled(availableFeatures.contains(FeatureName.SUPPORT_CHAT_FACELIFT_FOR_DNA.getName()))
                .withIsStopCampaignsFromGridEnabled(availableFeatures.contains(FeatureName.STOP_CAMPAIGNS_FROM_GRID.getName()))
                .withIsStopObjectsFromGridEnabled(availableFeatures.contains(FeatureName.STOP_OBJECTS_FROM_GRID.getName()))
                .withIsTurboPromoForMobileEnabledForDna(availableFeatures.contains(FeatureName.TURBO_PROMO_FOR_MOBILE_ENABLED_FOR_DNA.getName()))
                .withIsDynamicCampaignsEditAllowedForDna(availableFeatures.contains(FeatureName.DYNAMIC_CAMPAIGNS_EDIT_ALLOWED_FOR_DNA.getName()))
                .withIsDynamicGroupsEditAllowedForDna(availableFeatures.contains(FeatureName.DYNAMIC_GROUPS_EDIT_ALLOWED_FOR_DNA.getName()))
                .withIsDynamicAdsEditAllowedForDna(availableFeatures.contains(FeatureName.DYNAMIC_ADS_EDIT_ALLOWED_FOR_DNA.getName()))
                .withIsDynamicOnlyForDna(availableFeatures.contains(FeatureName.DYNAMIC_ONLY_FOR_DNA.getName()))
                .withEnableSidebarOptimize(availableFeatures.contains(FeatureName.ENABLE_SIDEBAR_OPTIMIZE.getName()))
                .withIsDynamicGroupsFeedGridEnabledForDna(availableFeatures.contains(FeatureName.DYNAMIC_GROUPS_FEED_GRID_ENABLED_FOR_DNA.getName()))
                .withIsMobileContentCampaignsEditAllowedForDna(availableFeatures.contains(FeatureName.MOBILE_CONTENT_CAMPAIGNS_EDIT_ALLOWED_FOR_DNA.getName()))
                .withIsMobileContentGroupsEditAllowedForDna(availableFeatures.contains(FeatureName.MOBILE_CONTENT_GROUPS_EDIT_ALLOWED_FOR_DNA.getName()))
                .withIsMobileContentAdsEditAllowedForDna(availableFeatures.contains(FeatureName.MOBILE_CONTENT_ADS_EDIT_ALLOWED_FOR_DNA.getName()))
                .withIsMobileContentOnlyForDna(availableFeatures.contains(FeatureName.MOBILE_CONTENT_ONLY_FOR_DNA.getName()))
                .withIsShowSkadnetworkOnNewIosEnabled(availableFeatures.contains(FeatureName.SHOW_SKADNETWORK_ON_NEW_IOS_ENABLED.getName()))
                .withIsMobileContentCampaignsRoiStrategyEnabled(availableFeatures.contains(FeatureName.MOBILE_CONTENT_CAMPAIGNS_ROI_STRATEGY.getName()))
                .withIsMobileContentCampaignsOptimizeCpiStrategyEnabled(availableFeatures.contains(FeatureName.MOBILE_CONTENT_CAMPAIGNS_OPTIMIZE_CPI_STRATEGY.getName()))
                .withIsShowSkadnetworkOnNewIosSlotsEnabled(availableFeatures.contains(FeatureName.SHOW_SKADNETWORK_ON_NEW_IOS_SLOTS_ENABLED.getName()))
                .withIsShowSkadnetworkOnUac(availableFeatures.contains(FeatureName.SHOW_SKADNETWORK_ON_UAC.getName()))
                .withIsPostViewAttributionOnUac(availableFeatures.contains(FeatureName.POST_VIEW_ATTRIBUTION_ON_UAC.getName()))
                .withIsOrganizationsRedesignEnabledForDna(availableFeatures.contains(FeatureName.ORGANIZATIONS_REDESIGN_ENABLED_FOR_DNA.getName()))
                .withIsNewPhoneEditorEnabledForDna(availableFeatures.contains(FeatureName.NEW_PHONE_EDITOR_ENABLED_FOR_DNA.getName()))
                .withIsNewTextsInEditBannersEnabledForDna(availableFeatures.contains(FeatureName.NEW_TEXTS_IN_EDIT_BANNERS_ENABLED_FOR_DNA.getName()))
                .withIsSimplifiedStrategyViewEnabled(availableFeatures.contains(FeatureName.SIMPLIFIED_STRATEGY_VIEW_ENABLED.getName()))
                .withIsSavingSearchAndNetworkStrategyInSimplifiedViewEnabled(availableFeatures.contains(FeatureName.SAVING_SEARCH_AND_NETWORK_STRATEGY_IN_SIMPLIFIED_VIEW_ENABLED.getName()))
                .withIsStrategyProModeRedesignEnabled(availableFeatures.contains(FeatureName.STRATEGY_PRO_MODE_REDESIGN_ENABLED.getName()))
                .withIsPayStepInCampaignCreationEnabled(availableFeatures.contains(FeatureName.PAY_STEP_IN_CAMPAIGN_CREATION_ENABLED_FOR_DNA.getName()))
                .withIsConversionStrategyLearningStatusEnabled(availableFeatures.contains(FeatureName.CONVERSION_STRATEGY_LEARNING_STATUS_ENABLED.getName()))
                .withIsCpmAudioGroupsEditForDna(availableFeatures.contains(FeatureName.CPM_AUDIO_GROUPS_EDIT_FOR_DNA.getName()))
                .withIsCpmIndoorGroupsEditForDna(availableFeatures.contains(FeatureName.CPM_INDOOR_GROUPS_EDIT_FOR_DNA.getName()))
                .withEnabledChangeOfferForClientsFromTurkey(availableFeatures.contains(FeatureName.CHANGE_OFFER_FOR_CLIENTS_FROM_TURKEY.getName()))
                .withIsCpmOutdoorGroupsEditForDna(availableFeatures.contains(FeatureName.CPM_OUTDOOR_GROUPS_EDIT_FOR_DNA.getName()))
                .withIsRmpImpressionsLinkEnabled(availableFeatures.contains(FeatureName.RMP_IMPRESSIONS_LINK_ENABLED.getName()))
                .withIsNewCampaignInfoEnabled(availableFeatures.contains(FeatureName.NEW_CAMPAIGN_INFO_ENABLED.getName()))
                .withIsNewCampaignPageEnabled(availableFeatures.contains(FeatureName.NEW_CAMPAIGN_PAGE_ENABLED.getName()))
                .withIsMetrikaGoalsCreationAvailableViaDirectInterface(availableFeatures.contains(FeatureName.CREATE_METRIKA_GOALS_VIA_DIRECT_INTERFACE.getName()))
                .withIsShowWeekBudgetRecommendationByConversionPrice(availableFeatures.contains(FeatureName.SHOW_WEEK_BUDGET_RECOMMENDATION_BY_CONVERSION_PRICE.getName()))
                //саджесты uc
                .withIsUcSuggestCampaignInfoStepSiteEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CAMPAIGN_INFO_STEP_SITE.getName()))
                .withIsUcSuggestCampaignInfoStepCompanyNameEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CAMPAIGN_INFO_STEP_COMPANY_NAME.getName()))
                .withIsUcSuggestCampaignInfoStepBusinessCategoryEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CAMPAIGN_INFO_STEP_BUSINESS_CATEGORY.getName()))
                .withIsUcSuggestCampaignInfoStepMetrikaCounterEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CAMPAIGN_INFO_STEP_METRIKA_COUNTER.getName()))
                .withIsUcSuggestCampaignInfoStepGoalEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CAMPAIGN_INFO_STEP_GOAL.getName()))
                .withIsUcSuggestCampaignInfoStepGoalPriceEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CAMPAIGN_INFO_STEP_GOAL_PRICE.getName()))
                .withIsUcSuggestAudienceStepPhrasesEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_PHRASES.getName()))
                .withIsUcSuggestAudienceStepGeoEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_GEO.getName()))
                .withIsUcSuggestAudienceStepMinusPhrasesEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_MINUS_PHRASES.getName()))
                .withIsUcSuggestAudienceStepSocdemEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_SOCDEM.getName()))
                .withIsUcSuggestAudienceStepDevicesEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_DEVICES.getName()))
                .withIsUcSuggestAudienceStepRetargetingEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_RETARGETING.getName()))
                .withIsUcSuggestAdsStepExamplesEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_ADS_STEP_EXAMPLES.getName()))
                .withIsUcSuggestCategoriesByCompetitorsEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_CATEGORIES_BY_COMPETITORS.getName()))
                .withIsUcSuggestPhrasesByCompetitorsEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_PHRASES_BY_COMPETITORS.getName()))
                .withIsUcSuggestAdsStepSnippetEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_ADS_STEP_SNIPPET.getName()))
                .withIsUcSuggestAdsStepSitelinksEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_ADS_STEP_SITELINKS.getName()))
                .withIsUcSuggestAdsStepCalloutsEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_ADS_STEP_CALLOUTS.getName()))
                .withIsUcSuggestAdsStepImagesEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_ADS_STEP_IMAGES.getName()))
                .withIsUcSuggestAudienceStepForecastEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_AUDIENCE_STEP_FORECAST.getName()))
                .withIsUcSuggestBudgetStepForecastEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_BUDGET_STEP_FORECAST.getName()))
                .withIsUcSuggestLastBannerUrlEnabled(availableFeatures.contains(FeatureName.UC_SUGGEST_LAST_BANNER_URL.getName()))

                .withIsUcKotlinBackendEnabled(availableFeatures.contains(FeatureName.UC_KOTLIN_BACKEND_ENABLED.getName()))

                .withAddWithTotalsToGroupQuery(availableFeatures.contains(FeatureName.ADD_WITH_TOTALS_TO_GROUP_QUERY.getName()))
                .withAddWithTotalsToBannerQuery(availableFeatures.contains(FeatureName.ADD_WITH_TOTALS_TO_BANNER_QUERY.getName()))
                .withAddWithTotalsToShowConditionQuery(availableFeatures.contains(FeatureName.ADD_WITH_TOTALS_TO_SHOW_CONDITION_QUERY.getName()))
                .withAddWithTotalsToRetargetingConditionQuery(availableFeatures.contains(FeatureName.ADD_WITH_TOTALS_TO_BIDS_RETARGETING_QUERY.getName()))

                .withIsHolidayFeature1Enabled(availableFeatures.contains(FeatureName.HOLIDAY_FEATURE1.getName()))

                .withIsClientAllowedToRemoderate(availableFeatures.contains(FeatureName.CLIENT_ALLOWED_TO_REMODERATE.getName()))
                .withIsYouMoneyNonResidentWarningShowEnabled(availableFeatures.contains(FeatureName.YOU_MONEY_NON_RESIDENT_WARNING_SHOW.getName()))
                .withIsTgoSocialDemoInUserProfile(availableFeatures.contains(FeatureName.TGO_SOCIAL_DEMO_IN_USER_PROFILE.getName()))
                .withIsTgoFamilyAndBehaviorsInUserProfile(availableFeatures.contains(FeatureName.TGO_FAMILY_AND_BEHAVIORS_IN_USER_PROFILE.getName()))
                .withIsTgoAllInterestsInUserProfile(availableFeatures.contains(FeatureName.TGO_ALL_INTERESTS_IN_USER_PROFILE.getName()))
                .withIsTgoMetrikaAndAudienceInUserProfile(availableFeatures.contains(FeatureName.TGO_METRIKA_AND_AUDIENCE_IN_USER_PROFILE.getName()))
                .withIsCpmFixedDataInMolEnabled(availableFeatures.contains(FeatureName.CPM_FIXED_DATA_IN_MOL.getName()))
                .withIsSuggestionForRecommendedGoalsEnabled(availableFeatures.contains(FeatureName.ENABLE_SUGGESTION_FOR_RECOMMENDED_GOALS.getName()))
                .withIsNewTgoCalltrackingDesignEnabled(availableFeatures.contains(FeatureName.NEW_TGO_CALLTRACKING_DESIGN_ENABLED.getName()))
                .withIsCrrStrategyAllowed(availableFeatures.contains(FeatureName.CRR_STRATEGY_ALLOWED.getName()))
                .withIsOrderPhraseLengthPrecedenceAllowed(availableFeatures.contains(FeatureName.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ALLOWED.getName()))
                .withInAppMobileTargeting(availableFeatures.contains(FeatureName.IN_APP_MOBILE_TARGETING.getName()))
                .withEnableServerPushes(availableFeatures.contains(FeatureName.SEND_XIVA_PUSHES.getName()))
                .withIsExpandInventoryBidModifiersEnabled(availableFeatures.contains(FeatureName.EXPAND_INVENTORY_BID_MODIFIERS.getName()))
                .withIsCampaignAllowedOnAdultContentEnabled(availableFeatures.contains(FeatureName.CAMPAIGN_ALLOWED_ON_ADULT_CONTENT.getName()))
                .withIsUniversalCampaignsEcomEnabled(availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_ECOM_ENABLED.getName()))
                .withIsSearchRetargetingEnabled(availableFeatures.contains(FeatureName.SEARCH_RETARGETING_ENABLED.getName()))
                .withIsAdvancedGeoTargetingEnabled(availableFeatures.contains(FeatureName.ADVANCED_GEOTARGETING.getName()));

        Set<GdCoreFeatureWithDescription> coreFeaturesWithDescription =
                gdFeatureWithDescriptionConverter.convertToCore(availableFeatures);

        return clientFeatures
                .withAvailableClientCoreFeatures(coreFeaturesWithDescription);
    }

    Set<GdCoreFeatureWithDescription> getClientPublicCoreFeatures(ClientId clientId) {
        Set<String> publicFeatures = featureService.getPublicForClientId(clientId);
        return gdFeatureWithDescriptionConverter.convertToCore(publicFeatures);
    }

    private boolean isAnyFeatureEnabled(ClientId clientId, Collection<FeatureName> featureNames) {
        return StreamEx.of(featureNames)
                .anyMatch(featureName -> featureService.isEnabledForClientId(clientId, featureName));
    }

    public GdCampaignNotification getDefaultCampaignNotificationForTextCampaign(User operator, User client,
                                                                                boolean hasWallet) {
        return getDefaultCampaignNotification(operator, client, hasWallet, GdCampaignType.TEXT);
    }

    List<GdDefaultCampaignNotification> getDefaultCampaignNotificationsForCampaignTypes(User operator, User client,
                                                                                        boolean hasWallet,
                                                                                        List<GdCampaignType> campaignTypes) {
        return StreamEx.of(nvl(campaignTypes, SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES))
                .map(campaignType -> getDefaultCampaignNotification(operator, client, hasWallet, campaignType))
                .toList();
    }

    static GdDefaultCampaignNotification getDefaultCampaignNotification(User operator, User client, boolean hasWallet,
                                                                        GdCampaignType gdCampaignType) {
        checkState(SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES.contains(gdCampaignType));

        CampaignType coreCampaignType = toCampaignType(gdCampaignType);

        var availableEmailEvents = getAvailableEmailEvents(hasWallet, coreCampaignType, operator.getRole() == MANAGER);
        GdCampaignEmailSettings emailSettings = new GdCampaignEmailSettings()
                .withEmail(client.getEmail())
                .withAllowedEvents(mapSet(availableEmailEvents, GdCampaignEmailEvent::fromSource))
                .withXlsReady(true)
                .withSendAccountNews(true)
                .withStopByReachDailyBudget(true)
                .withWarningBalance(CampaignConstants.DEFAULT_CAMPAIGN_WARNING_BALANCE)
                .withCheckPositionInterval(null);
        Set<SmsFlag> availableSmsFlags = getAvailableSmsFlags(hasWallet, coreCampaignType);
        GdCampaignSmsSettings smsSettings = new GdCampaignSmsSettings()
                .withSmsTime(DefaultValuesUtils.DEFAULT_SMS_TIME_INTERVAL)
                .withEvents(mapSet(availableSmsFlags, flag -> toGdCampaignSmsEventInfo(flag, false)));

        if (CampaignTypeKinds.INTERNAL.contains(coreCampaignType)) {
            emailSettings
                    .withEmail(CampaignConstants.INTERNAL_CAMPAIGN_EMAIL)
                    .withWarningBalance(CampaignConstants.DEFAULT_INTERNAL_CAMPAIGN_WARNING_BALANCE);
        }

        return new GdDefaultCampaignNotification()
                .withCampaignType(gdCampaignType)
                .withEmailSettings(emailSettings)
                .withSmsSettings(smsSettings);
    }

    GdClientNotificationInfo getNotificationInfo(Long chiefUserId) {
        String smsPhone = userService.getSmsPhone(chiefUserId);

        return new GdClientNotificationInfo()
                .withSmsPhone(smsPhone);
    }

    /**
     * Получить список используемых гео клиента, взятый по всем группам, и отсортированный по частоте встречаемости
     *
     * @param clientInfo клиент
     * @param input      параметры запроса
     * @return отсортированный список гео клиента
     */
    List<GdClientTopRegionInfo> getAdGroupTopRegions(GdClientInfo clientInfo, GdClientTopRegions input) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        GeoTree geoTree = geoTreeFactory.getTranslocalGeoTree(clientInfo.getCountryRegionId());

        Set<Long> campaignIds = getClientCampaignIds(clientId, input.getCampaignIds());

        Map<Long, List<AdGroupSimple>> adGroupsByCampaignId =
                adGroupService.getSimpleAdGroupsByCampaignIds(clientId, campaignIds);

        List<List<Long>> regions = EntryStream.of(adGroupsByCampaignId)
                .flatMapValues(List::stream)
                .mapValues(AdGroupSimple::getGeo)
                .filterValues(Objects::nonNull)
                .mapKeyValue((campaignId, geo) -> clientGeoService.convertForWeb(geo, geoTree))
                .toList();

        Map<List<Integer>, Long> regionsWithFrequencyCount = regions.stream()
                .collect(groupingBy(ListUtils::longToIntegerList, Collectors.counting()));

        return EntryStream.of(regionsWithFrequencyCount)
                .mapKeyValue((regionIds, count) -> new GdClientTopRegionInfo()
                        .withCount(count.intValue())
                        .withRegionIds(regionIds))
                .reverseSorted(Comparator.comparingLong(GdClientTopRegionInfo::getCount))
                .skip(input.getLimitOffset().getOffset())
                .limit(input.getLimitOffset().getLimit())
                .toList();
    }

    private Set<Long> getClientCampaignIds(ClientId clientId, @Nullable List<Long> campaignIds) {
        if (CollectionUtils.isEmpty(campaignIds)) {
            List<CampaignSimple> campaigns = campaignService.searchCampaignsByClientIdAndTypes(clientId, WEB_EDIT);
            return listToSet(campaigns, CampaignSimple::getId);
        }

        return campaignService.getExistingCampaignIds(clientId, campaignIds);
    }

    public CompletableFuture<GdAccountScoreInfo> getAccountScoreInfo(ClientId clientId) {
        return clientsAccountScoreInfoDataLoader.get().load(clientId.asLong());
    }

    // Пока что (для эксперимента) значение тащится не из базы, а вычисляется эвристиками по некоторым допущениям
    // и работает только под фичёй MODERATION_OFFER_ENABLED_FOR_DNA
    public Boolean isOfferAccepted(GridGraphQLContext context) {
        Long uid = Optional.ofNullable(context.getSubjectUser())
                .or(() -> Optional.of(context.getOperator()))
                .map(User::getUid)
                .orElseThrow(() -> new IllegalStateException("User not set"));
        return userService.getIsOfferAccepted(uid);
    }

    public Set<GdUserFeature> getUserFeatures(User subjectUser) {
        Set<GdUserFeature> userFeatures = new HashSet<>();
        if (subjectUser.getStatusBlocked()) {
            userFeatures.add(GdUserFeature.BLOCKED);
        }
        return userFeatures;
    }

    public GdClientMetrikaCountersWithAdditionalInformation getMetrikaCountersWithAdditionInformation(
            @Nullable Long operatorUid,
            ClientId clientId) {
        return getMetrikaCountersWithAdditionInformation(operatorUid, clientId,
                CampMetrikaCountersService.CounterWithAdditionalInformationFilter.defaultFilter());
    }

    public GdMetrikaCountersFilterPayload getMetrikaCountersFilter(
            @Nullable Long operatorUid,
            ClientId clientId,
            @Nullable Set<Long> counterIds,
            @Nullable Set<Long> campaignIds,
            @Nullable String nameOrId) {
        try {
            var filter = CampMetrikaCountersService
                    .CounterWithAdditionalInformationFilter.defaultFilter();
            if ((counterIds == null || counterIds.isEmpty())
                    && (campaignIds == null || campaignIds.isEmpty())
            ) {
                filter.withLimit(MAX_METRIKA_COUNTERS_COUNT);
            }
            if (nameOrId != null && !nameOrId.isEmpty()) {
                try {
                    Long counterId = Long.parseLong(nameOrId);
                    filter.withCounterIds(singletonList(counterId));
                } catch (NumberFormatException ex) {
                    //may be not number;
                }

                filter.withCounterNamePrefix(nameOrId);
            }
            if (campaignIds != null && !campaignIds.isEmpty()) {
                filter.withCampaignIds(campaignIds);
            }
            if (counterIds != null && !counterIds.isEmpty()) {
                filter.withCounterIds(counterIds);
            }
            CampMetrikaCountersService.CountersContainer countersContainer =
                    campMetrikaCountersService.getAvailableExtendedCounters(
                            clientId,
                            operatorUid,
                            filter);
            boolean allowedSourceForTurboCounters = featureService.isEnabledForClientId(clientId,
                    FeatureName.ALLOWED_SOURCE_FOR_TURBO_COUNTERS);

            if (!allowedSourceForTurboCounters) {
                StreamEx.of(countersContainer.getClientAvailableCounters())
                        .filterBy(MetrikaCounterWithAdditionalInformation::getSource, MetrikaCounterSource.TURBO)
                        .forEach(counter -> counter.setSource(null));
            }

            return new GdMetrikaCountersFilterPayload()
                    .withCounters(toGdClientMetrikaCountersWithAdditionalInformation(
                            countersContainer.getClientAvailableCounters(),
                            countersContainer.getOperatorPermissionByCounterId()))
                    .withIsMetrikaAvailable(true)
                    .withHasMoreCounters(countersContainer.getHasMoreCounters());
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdMetrikaCountersFilterPayload()
                    .withCounters(null)
                    .withIsMetrikaAvailable(false)
                    .withHasMoreCounters(false);
        }
    }

    public GdClientMetrikaCountersWithAdditionalInformation getMetrikaCountersWithAdditionInformation(
            @Nullable Long operatorUid,
            ClientId clientId,
            CampMetrikaCountersService.CounterWithAdditionalInformationFilter counterFilter) {
        try {
            CampMetrikaCountersService.CountersContainer countersContainer =
                    campMetrikaCountersService.getAvailableCountersByClientAndOperatorUid(
                            clientId,
                            operatorUid,
                            counterFilter);
            boolean allowedSourceForTurboCounters = featureService.isEnabledForClientId(clientId,
                    FeatureName.ALLOWED_SOURCE_FOR_TURBO_COUNTERS);

            if (!allowedSourceForTurboCounters) {
                StreamEx.of(countersContainer.getClientAvailableCounters())
                        .filterBy(MetrikaCounterWithAdditionalInformation::getSource, MetrikaCounterSource.TURBO)
                        .forEach(counter -> counter.setSource(null));
            }

            return new GdClientMetrikaCountersWithAdditionalInformation()
                    .withCounters(toGdClientMetrikaCountersWithAdditionalInformation(
                            countersContainer.getClientAvailableCounters(),
                            countersContainer.getOperatorPermissionByCounterId()))
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdClientMetrikaCountersWithAdditionalInformation()
                    .withCounters(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    public GdMetrikaCountersByDomainPayload getMetrikaCountersByUrl(Long operatorUid, User subjectUser,
                                                                    @Nullable String url,
                                                                    boolean weakRestrictions) {
        try {
            return doGetMetrikaCountersByUrl(operatorUid, subjectUser, url, weakRestrictions);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            return new GdMetrikaCountersByDomainPayload()
                    .withIsMetrikaAvailable(false);
        }
    }

    private GdMetrikaCountersByDomainPayload doGetMetrikaCountersByUrl(Long operatorUid, User subjectUser,
                                                                       @Nullable String url,
                                                                       boolean weakRestrictions) {
        ClientId clientId = subjectUser.getClientId();

        GdClientMetrikaCountersWithAdditionalInformation accessibleCountersPayload =
                getMetrikaCountersWithAdditionInformation(operatorUid, clientId);

        if (!accessibleCountersPayload.getIsMetrikaAvailable()) {
            return new GdMetrikaCountersByDomainPayload()
                    .withIsMetrikaAvailable(false);
        }

        String domain = ifNotNull(bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(url),
                d -> toUnicodeDomain(d.toLowerCase()));
        Set<GdClientMetrikaCounter> accessibleCounters = accessibleCountersPayload.getCounters();
        Collection<MetrikaCounterByDomain> metrikaCountersFromUrl =
                getUrlMetrikaCounters(operatorUid, subjectUser, url, domain, accessibleCounters, weakRestrictions);
        Set<Long> metrikaCounterIdsFromUrl = listToSet(metrikaCountersFromUrl, MetrikaCounterByDomain::getCounterId);
        Set<Long> accessibleCounterIds = mapSet(accessibleCounters, GdClientMetrikaCounter::getId);

        Set<GdClientMetrikaCounter> accessibleMetrikaCountersFromUrl =
                StreamEx.of(accessibleCounters)
                        .filter(accessibleCounter -> metrikaCounterIdsFromUrl.contains(accessibleCounter.getId()))
                        .toSet();

        Set<GdInaccessibleMetrikaCounter> inaccessibleMetrikaCountersFromUrl =
                getInaccessibleCountersByDomain(subjectUser, domain, metrikaCountersFromUrl, accessibleCounterIds);

        return new GdMetrikaCountersByDomainPayload()
                .withIsMetrikaAvailable(true)
                .withAccessibleCountersByDomain(accessibleMetrikaCountersFromUrl)
                .withInaccessibleCountersByDomain(inaccessibleMetrikaCountersFromUrl)
                .withAllAccessibleCounters(accessibleCounters);
    }

    private List<MetrikaCounterByDomain> getUrlMetrikaCounters(Long operatorUid, User subjectUser, @Nullable String url,
                                                               @Nullable String domain,
                                                               Set<GdClientMetrikaCounter> accessibleCounters,
                                                               boolean weakRestrictions) {
        if (StringUtils.isEmpty(url)) {
            return emptyList();
        }

        ClientId clientId = subjectUser.getClientId();
        List<MetrikaCounterByDomain> metrikaCountersByUrl;
        boolean isEnableCounterSuggestionByShopInShopBusiness = featureService.isEnabledForClientId(clientId,
                FeatureName.ENABLE_FEED_AND_COUNTER_SUGGESTION_BY_SHOP_IN_SHOP);
        ShopInShopBusinessInfo businessInfo;
        if (isEnableCounterSuggestionByShopInShopBusiness
                && (businessInfo = shopInShopBusinessesService.getBusinessInfoByUrl(url)) != null) {
            // Если url является ссылкой на бизнес в маркетплейсе - возвращаем счетчик из таблицы
            // shop_in_shop_businesses
            metrikaCountersByUrl = List.of(new MetrikaCounterByDomain()
                    .withCounterId(businessInfo.getCounterId())
                    .withSource(businessInfo.getSource() == Source.MARKET ? MetrikaCounterSource.MARKET : null));
        } else if (bannersUrlHelper.isYandexMarketDomain(domain)) {
            metrikaCountersByUrl = getMarketMetrikaCounters(accessibleCounters);
        } else if (turboLandingService.isTurboLandingUrl(url)) {
            metrikaCountersByUrl = getTurboLandingMetrikaCounters(clientId, url);
        } else if (ZenDomain.isZenDomain(url)) {
            metrikaCountersByUrl = getZenMetrikaCounters(operatorUid, subjectUser, url);
        } else {
            metrikaCountersByUrl =
                    campMetrikaCountersService.getMetrikaCountersByDomain(clientId, domain, weakRestrictions);
        }

        Set<Long> counterIdsToExclude = countersExcludedFromAutocompleteProperty.getOrDefault(emptySet());
        return StreamEx.of(metrikaCountersByUrl)
                .remove(counter -> counterIdsToExclude.contains(counter.getCounterId()))
                .toList();
    }

    private List<MetrikaCounterByDomain> getZenMetrikaCounters(Long operatorUid, User subjectUser, String url) {
        final var optionalCounter = zenMetaInfoService.getZenCounterByUrl(url);
        if (optionalCounter.isPresent()) {
            final var counter = optionalCounter.get();

            if (Objects.nonNull(counter.getCounterId())) {
                return List.of(new MetrikaCounterByDomain().withCounterId(counter.getCounterId()));
            }
        }
        return List.of();
    }

    private List<MetrikaCounterByDomain> getTurboLandingMetrikaCounters(ClientId clientId, String url) {
        TurboLanding turbo = turboLandingService.externalFindTurboLandingByUrl(url);
        if (turbo == null || !hasAccessToTurboLanding(clientId, turbo)) {
            return emptyList();
        }

        TurboLandingMetrikaCountersAndGoals turboMetrikaCountersAndGoals =
                new TurboLandingMetrikaCountersAndGoals(turbo.getMetrikaCounters());

        return mapList(turboMetrikaCountersAndGoals.getCountersWithGoals(),
                turboLandingMetrikaCounter ->
                        new MetrikaCounterByDomain()
                                .withCounterId(turboLandingMetrikaCounter.getId())
                                .withSource(turboLandingMetrikaCounter.getIsUserCounter() ?
                                        MetrikaCounterSource.UNKNOWN : MetrikaCounterSource.TURBO));
    }

    private boolean hasAccessToTurboLanding(ClientId clientId, TurboLanding turboLanding) {
        return Objects.equals(turboLanding.getClientId(), clientId.asLong());
    }

    private List<MetrikaCounterByDomain> getMarketMetrikaCounters(Set<GdClientMetrikaCounter> accessibleCounters) {
        return StreamEx.of(accessibleCounters)
                .filter(c -> c.getSource() == GdMetrikaCounterSource.MARKET)
                .limit(1)
                .map(c -> new MetrikaCounterByDomain()
                        .withCounterId(c.getId())
                        .withSource(MetrikaCounterSource.MARKET))
                .toList();
    }

    /**
     * Извлечь недоступные для клиента счетчики.
     * Возвращает пустой список, если кол-во недоступных счетчиков превышает
     * {@link #MAX_INACCESSIBLE_METRIKA_COUNTERS_COUNT_FOR_SUGGEST} или если домен в блэклисте.
     * А также не возвращаются удаленные счетчики,
     * счетчики у которых в Метрике не разрешено использование без доступа
     * и счетчики, для которых можно найти домен из блэклиста по таблице счетчиков по доменам.
     */
    private Set<GdInaccessibleMetrikaCounter> getInaccessibleCountersByDomain(
            User clientUser, @Nullable String domain,
            Collection<MetrikaCounterByDomain> metrikaCountersFromUrl,
            Set<Long> accessibleCounterIds) {
        boolean inaccessibleCountersNotAllowed =
                campMetrikaCountersService.isDomainNotAllowedForUnavailableCounters(domain);
        if (inaccessibleCountersNotAllowed) {
            logger.info("Domain {} is in black list, inaccessible counters not allowed", domain);
            return emptySet();
        }
        List<MetrikaCounterByDomain> inaccessibleCountersFromUrl = StreamEx.of(metrikaCountersFromUrl)
                .remove(metrikaCounterFromUrl ->
                        accessibleCounterIds.contains(metrikaCounterFromUrl.getCounterId()))
                .toList();
        Set<GetExistentCountersResponseItem> allowedInaccessibleCounters =
                getAllowedInaccessibleCounters(inaccessibleCountersFromUrl);
        if (allowedInaccessibleCounters.isEmpty()) {
            return emptySet();
        } else if (allowedInaccessibleCounters.size() > MAX_INACCESSIBLE_METRIKA_COUNTERS_COUNT_FOR_SUGGEST) {
            logger.info("Found {} inaccessible counters on domain {}, but max number is {}",
                    allowedInaccessibleCounters.size(), domain, MAX_INACCESSIBLE_METRIKA_COUNTERS_COUNT_FOR_SUGGEST);
            return emptySet();
        }

        Map<Long, Boolean> hasEcommerceByCounterId = StreamEx.of(allowedInaccessibleCounters)
                .mapToEntry(GetExistentCountersResponseItem::getCounterId,
                        GetExistentCountersResponseItem::getEcommerce)
                .nonNullValues()
                .toMap();
        Map<Long, GdMetrikaCounterSource> sourceByCounterId = StreamEx.of(allowedInaccessibleCounters)
                .mapToEntry(GetExistentCountersResponseItem::getCounterId,
                        counter -> convertCounterSource(counter.getCounterSource()))
                .nonNullValues()
                .toMap();
        Set<Long> allowedInaccessibleCounterIds =
                mapSet(allowedInaccessibleCounters, GetExistentCountersResponseItem::getCounterId);
        Map<Long, Boolean> grantAccessRequestedByCounterId =
                campMetrikaCountersService.getGrantAccessRequestedByCounterId(
                        clientUser.getUid(), allowedInaccessibleCounterIds);
        return StreamEx.of(inaccessibleCountersFromUrl)
                .filter(c -> allowedInaccessibleCounterIds.contains(c.getCounterId()))
                .map(counter -> new GdInaccessibleMetrikaCounter()
                        .withId(counter.getCounterId())
                        .withSource(nullableNvl(toGdMetrikaCounterSource(counter.getSource()),
                                sourceByCounterId.get(counter.getCounterId())))
                        .withAccessRequested(grantAccessRequestedByCounterId.get(counter.getCounterId()))
                        .withHasEcommerce(hasEcommerceByCounterId.get(counter.getCounterId()))
                )
                .toSet();
    }

    private Set<GetExistentCountersResponseItem> getAllowedInaccessibleCounters(
            List<MetrikaCounterByDomain> inaccessibleCounters) {
        if (inaccessibleCounters.isEmpty()) {
            return emptySet();
        }
        Set<Long> inaccessibleCounterIds = listToSet(inaccessibleCounters, MetrikaCounterByDomain::getCounterId);
        return campMetrikaCountersService.getAllowedInaccessibleCounters(inaccessibleCounterIds);
    }

    public GdMetrikaCountersInfoPayload getMetrikaCountersInfo(Long uid, List<Long> counterIds) {
        try {
            Map<Long, GetExistentCountersResponseItem> existentCounterById = getExistentCounterById(counterIds);
            Set<Long> existentCounterIds = existentCounterById.keySet();
            Set<Long> nonExistentCounterIds = filterToSet(counterIds,
                    counterId -> !existentCounterIds.contains(counterId));
            Map<Long, Boolean> grantAccessRequestedByCounterId =
                    campMetrikaCountersService.getGrantAccessRequestedByCounterId(uid, existentCounterIds);

            Set<GdMetrikaCounter> existentCountersPayload = listToSet(existentCounterById.values(),
                    existentCounter -> new GdMetrikaCounter()
                            .withId(existentCounter.getCounterId())
                            .withSource(convertCounterSource(existentCounter.getCounterSource()))
                            .withStatus(getCounterStatus(existentCounter.getCounterId(),
                                    grantAccessRequestedByCounterId)));
            Set<GdMetrikaCounter> nonExistentCountersPayload = mapSet(nonExistentCounterIds,
                    counterId -> new GdMetrikaCounter()
                            .withId(counterId)
                            .withStatus(NOT_FOUND));

            return new GdMetrikaCountersInfoPayload()
                    .withIsMetrikaAvailable(true)
                    .withCounters(Sets.union(existentCountersPayload, nonExistentCountersPayload));
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            return new GdMetrikaCountersInfoPayload()
                    .withIsMetrikaAvailable(false)
                    .withCounters(emptySet());
        }
    }

    private Map<Long, GetExistentCountersResponseItem> getExistentCounterById(Collection<Long> counterIds) {
        List<GetExistentCountersResponseItem> responseItems =
                campMetrikaCountersService.getExistentCounters(counterIds);

        return StreamEx.of(responseItems)
                .mapToEntry(GetExistentCountersResponseItem::getCounterId, identity())
                .distinctKeys()
                .toMap();
    }

    private GdMetrikaCounterStatus getCounterStatus(Long counterId,
                                                    Map<Long, Boolean> grantAccessRequestedByCounterId) {
        Boolean grantAccessRequested = grantAccessRequestedByCounterId.get(counterId);

        if (grantAccessRequested == null) {
            return NOT_FOUND;
        }

        return grantAccessRequested ? ACCESS_REQUESTED : NO_RIGHTS;
    }

    private GdMetrikaCounterSource convertCounterSource(String counterSource) {
        return toGdMetrikaCounterSource(toMetrikaCounterSource(counterSource));
    }

    public GdInternalDontShowDomainsDataPayload getInternalDontShowDomainsData(GdClientInfo clientInfo) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Set<Long> campaignIds = campaignRepository.getNonArchivedNonDeletedCampaignIdsByClientIds(shard,
                Set.of(clientId));

        var campaignsByDomain = StreamEx.of(campaignService.getCampaigns(clientId, campaignIds))
                .mapToEntry(this::getInternalDontShowDomains, identity())
                .flatMapKeys(Collection::stream)
                .grouping();
        var rowset = EntryStream.of(campaignsByDomain)
                .mapKeyValue((domain, campaigns) -> new GdInternalDontShowDomainsDataPayloadItem()
                        .withDomain(domain)
                        .withCampaigns(campaigns))
                .toList();

        return new GdInternalDontShowDomainsDataPayload()
                .withRowset(rowset);
    }

    private Set<String> getInternalDontShowDomains(Campaign campaign) {
        Set<String> disabledDomains = nvl(campaign.getDisabledDomains(), emptySet());
        List<String> disabledVideoDomains = nvl(campaign.getDisabledVideoPlacements(), emptyList());

        return filterToSet(Iterables.concat(disabledDomains, disabledVideoDomains), pageService::isInternalPageDomain);
    }

    public GdSuggestDataByUrlPayload getSuggestDataByUrl(GridGraphQLContext context, GdSuggestDataByUrl input) {
        User subjectUser = context.getSubjectUser();
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = subjectUser.getClientId();
        String url = input.getUrl();
        CurrencyCode currencyCode = clientService.getWorkCurrency(clientId).getCode();

        ValidationResult<String, Defect> validationResult = (new BannerHrefValidator()).apply(url);
        if (validationResult.hasAnyErrors()) {
            var gridValidationResult = gridValidationResultConversionService.buildGridValidationResult(validationResult,
                    path(field(GdSuggestDataByUrl.URL)));

            return new GdSuggestDataByUrlPayload()
                    .withValidationResult(gridValidationResult)
                    .withCounterIds(emptySet())
                    .withCounters(emptySet())
                    .withGoals(emptySet());
        }
        if (ZenDomain.isZenDomain(url)) {
            var zenSuggestData = getZenSuggestDataByUrl(operatorUid, subjectUser, url);
            if (zenSuggestData.isPresent()) {
                return zenSuggestData.get()
                        .withWeekBudget(Currencies.getCurrency(currencyCode).getUcDefaultWeekBudget())
                        .withDeviceTypes(DEFAULT_UC_DEVICES)
                        .withCampaignSettings(getGdSuggestCampaignsSettings(null))
                        .withSocdem(DEFAULT_UC_SOCDEM);
            }
        }

        var organizationForSuggest = getOrganizationForSuggest(url, input.getLanguage());
        var phones = getPhonesForSuggest(subjectUser, organizationForSuggest);
        Set<MetrikaCounterForSuggest> countersForSuggest =
                getMetrikaCounterIdsForSuggest(operatorUid, subjectUser, url);
        Set<GdGoal> goalsForSuggest = getGoalsForSuggest(operatorUid, subjectUser, countersForSuggest);
        String businessCategoryForSuggest = getBusinessCategoryForSuggest(organizationForSuggest);
        Set<Long> regionIdsForSuggest = getRegionIdsForSuggest(organizationForSuggest);
        List<GdCpaEstimate> cpaEstimatesForSuggest = getCpaEstimates(subjectUser, operatorUid, currencyCode, null,
                convertGoals(goalsForSuggest), regionIdsForSuggest, businessCategoryForSuggest, url);
        var calltrackingOnSiteForSuggest = getCalltrackingOnSiteForSuggest(clientId, operatorUid, url);
        var socialNetworkAccount = extractSocialNetworkAccount(url);

        var result = new GdSuggestDataByUrlPayload()
                .withOrganization(organizationForSuggest)
                .withGoals(goalsForSuggest)
                .withBusinessCategory(businessCategoryForSuggest)
                .withCounterIds(mapSet(countersForSuggest, MetrikaCounterForSuggest::getId))
                .withCounters(mapSet(countersForSuggest, ClientDataConverter::toGdMetrikaCounterForSuggest))
                .withWeekBudget(Currencies.getCurrency(currencyCode).getUcDefaultWeekBudget())
                .withEcomWeekBudget(Currencies.getCurrency(currencyCode).getEcomUcDefaultWeekBudget())
                .withFaviconLink(makeFaviconLink(url))
                .withDeviceTypes(DEFAULT_UC_DEVICES)
                .withSocdem(DEFAULT_UC_SOCDEM)
                .withPhones(phones)
                .withCpa(getExpectedCpa(businessCategoryForSuggest))
                .withRegionIds(regionIdsForSuggest)
                .withCpaEstimates(cpaEstimatesForSuggest)
                .withCampaignSettings(getGdSuggestCampaignsSettings(socialNetworkAccount))
                .withCalltrackingOnSite(calltrackingOnSiteForSuggest);
        logger.info("SuggestDataByUrl response: {}", result);
        return result;
    }

    private GdSuggestCampaignsSettings getGdSuggestCampaignsSettings(
            @Nullable GdSuggestSocialNetworkAccount socialNetworkAccount
    ) {
        return new GdSuggestCampaignsSettings()
                .withIsRecommendationsManagementEnabled(DEFAULT_UAC_IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .withIsPriceRecommendationsManagementEnabled(DEFAULT_UAC_IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .withSocialNetworkAccount(socialNetworkAccount);
    }

    private Optional<GdSuggestDataByUrlPayload> getZenSuggestDataByUrl(Long operatorUid,
                                                                       User subjectUser,
                                                                       String url) {

        final var zenSuggest = zenMetaInfoService.getZenMetaInfoWithGeneratedTextByUrl(url);
        if (zenSuggest.isPresent()) {
            final var counter = zenSuggest.get().getCounter();
            Set<GdGoal> goals = Set.of();
            Set<GdClientMetrikaCounterForSuggest> counters = Set.of();
            Set<Long> counterIds = Set.of();

            if (Objects.nonNull(counter.getCounterId())) {
                var gdCounter =
                        getGdClientMetrikaCounter(subjectUser, operatorUid, counter.getCounterId());
                if (gdCounter.isPresent()) {
                    counterIds = Set.of(counter.getCounterId());
                    goals = toGdGoals(counter.getGoals());
                    counters = Set.of(new GdClientMetrikaCounterForSuggest().withId(counter.getCounterId())
                            .withIsAccessible(true)
                            .withIsCalltrackingOnSiteCompatible(isCalltrackingOnSiteCompatible(gdCounter.get().getSource()))
                            .withIsEditableByOperator(gdCounter.get().getIsEditableByOperator()));
                } else if (!campMetrikaCountersService.getAllowedInaccessibleCounterIds(Set.of(counter.getCounterId())).isEmpty()) {
                    counterIds = Set.of(counter.getCounterId());
                    goals = toGdGoals(counter.getGoals());
                    counters = Set.of(new GdClientMetrikaCounterForSuggest().withId(counter.getCounterId())
                            .withIsAccessible(false)
                            .withIsCalltrackingOnSiteCompatible(false)
                            .withIsEditableByOperator(false));
                }
            }
            final var textSuggest = new GdTextSuggest()
                    .withAlgVersion(zenSuggest.get().getTextSuggest().getAlgVersion())
                    .withTitles(zenSuggest.get().getTextSuggest().getTitles())
                    .withBodies(zenSuggest.get().getTextSuggest().getBodies());
            final var zenMeta = new GdZenMetaInfo()
                    .withPublisherId(zenSuggest.get().getPublisherId())
                    .withPublisherItemId(zenSuggest.get().getPublisherItemId())
                    .withImages(zenSuggest.get().getImages())
                    .withTextSuggest(textSuggest);

            return Optional.of(new GdSuggestDataByUrlPayload()
                    .withZenMeta(zenMeta)
                    .withCounterIds(counterIds)
                    .withGoals(goals)
                    .withCounters(counters));

        }
        return Optional.empty();
    }

    @NotNull
    private Optional<GdClientMetrikaCounter> getGdClientMetrikaCounter(User subjectUser, Long operatorUid,
                                                                       Long counterId) {
        var countersWithAdditionInformation =
                getMetrikaCountersWithAdditionInformation(operatorUid, subjectUser.getClientId());
        Optional<GdClientMetrikaCounter> gdCounter = Optional.empty();
        if (countersWithAdditionInformation.getCounters() != null) {
            gdCounter = countersWithAdditionInformation.getCounters()
                    .stream().filter(c -> Objects.equals(counterId, c.getId())).findAny();
        }
        return gdCounter;
    }

    @Nullable
    private Set<GdPhoneWithId> getPhonesForSuggest(
            @Nullable User subjectUser,
            @Nullable GdOrganization gdOrganization
    ) {
        if (gdOrganization == null) {
            return null;
        }
        Set<GdPhoneWithId> phones = null;
        var metrikaCounterId = gdOrganization.getCounterId();
        var organizationInfo = OrganizationsConverter.fromGdOrganizationPartial(gdOrganization);
        if (subjectUser != null && organizationInfo.getPermalinkId() != null
                && !organizationInfo.getPhones().isEmpty() && metrikaCounterId != null) {
            var clientPhones = clientPhoneService.getAndSaveOrganizationPhones(
                    subjectUser.getClientId(),
                    organizationInfo,
                    metrikaCounterId
            );
            phones = new HashSet<>();
            for (var phone : clientPhones) {
                phones.add(Converter.toGdPhoneWithId(phone.getPhoneNumber(), phone.getId()));
            }
        }
        return phones;
    }

    @Nullable
    private GdOrganization getOrganizationForSuggest(String url, LanguageOuterClass.Language language) {
        try {
            var organizationInfo = organizationsDataService.getMainOrganizationInfoByUrl(url, language);
            return organizationInfo == null ?
                    null : OrganizationsConverter.toGdOrganization(organizationInfo);
        } catch (RuntimeException e) {
            logger.error("Error while getting organization by url. Do not suggest organization.");
            return null;
        }
    }

    private Set<MetrikaCounterForSuggest> getMetrikaCounterIdsForSuggest(Long operatorUid,
                                                                         User subjectUser,
                                                                         String url) {
        return getMetrikaCounterIdsForSuggest(operatorUid, subjectUser, url, null);
    }

    private Set<MetrikaCounterForSuggest> getMetrikaCounterIdsForSuggest(Long operatorUid,
                                                                         User subjectUser,
                                                                         String url,
                                                                         @Nullable Set<Long> additionalCounterIds) {
        boolean weakRestrictions = metrikaCounterUseWeakRestrictions.getOrDefault(false);
        GdMetrikaCountersByDomainPayload metrikaCountersByDomain =
                getMetrikaCountersByUrl(operatorUid, subjectUser, url, weakRestrictions);

        List<MetrikaCounterForSuggest> accessibleCountersForSuggest =
                StreamEx.of(convertFromAccessibleCounters(metrikaCountersByDomain.getAccessibleCountersByDomain()))
                        .sortedByLong(MetrikaCounterForSuggest::getId)
                        .toList();
        List<MetrikaCounterForSuggest> inaccessibleCountersForSuggest =
                StreamEx.of(convertFromInaccessibleCounters(metrikaCountersByDomain.getInaccessibleCountersByDomain()))
                        .sortedByLong(MetrikaCounterForSuggest::getId)
                        .toList();
        List<MetrikaCounterForSuggest> additionalCountersForSuggest =
                StreamEx.of(getAdditionalCountersForSuggest(additionalCounterIds, metrikaCountersByDomain, subjectUser))
                        .sortedByLong(MetrikaCounterForSuggest::getId)
                        .toList();

        Set<MetrikaCounterForSuggest> newCountersForSuggest = StreamEx.of(accessibleCountersForSuggest)
                .append(inaccessibleCountersForSuggest)
                .limit(MAX_METRIKA_COUNTERS_COUNT_FOR_SUGGEST)
                .toSet();
        return StreamEx.of(additionalCountersForSuggest)
                .append(newCountersForSuggest)
                .limit(MAX_METRIKA_COUNTERS_COUNT)
                .toSet();
    }

    private boolean needToSuggestAllGoals(User subjectUser) {
        if (featureService.isEnabledForClientId(subjectUser.getClientId(), UAC_UNAVAILABLE_GOALS_ALLOWED)) {
            return true;
        }
        boolean ucBetaDisabled =
                featureService.isEnabledForClientId(subjectUser.getClientId(), UNIVERSAL_CAMPAIGNS_BETA_DISABLED);
        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);
        return requestFromInternalNetwork && !ucBetaDisabled;
    }

    private static Set<MetrikaCounterForSuggest> convertFromAccessibleCounters(Set<GdClientMetrikaCounter> accessibleCountersByDomain) {
        if (isEmpty(accessibleCountersByDomain)) {
            return emptySet();
        }

        return StreamEx.of(accessibleCountersByDomain)
                .map(counter -> new MetrikaCounterForSuggest()
                        .withId(counter.getId())
                        .withHasEcommerce(counter.getHasEcommerce())
                        .withSource(counter.getSource())
                        .withEditableByOperator(counter.getIsEditableByOperator())
                        .withCalltrackingOnSiteCompatible(isCalltrackingOnSiteCompatible(counter.getSource()))
                        .withAccessible(true)
                )
                .toSet();
    }

    private static Set<MetrikaCounterForSuggest> convertFromInaccessibleCounters(Set<GdInaccessibleMetrikaCounter> inaccessibleCountersByDomain) {
        if (isEmpty(inaccessibleCountersByDomain)) {
            return emptySet();
        }

        return StreamEx.of(inaccessibleCountersByDomain)
                .map(counter -> new MetrikaCounterForSuggest()
                        .withId(counter.getId())
                        .withSource(counter.getSource())
                        .withEditableByOperator(false)
                        .withCalltrackingOnSiteCompatible(false)
                        .withAccessible(false)
                        .withAccessRequested(nvl(counter.getAccessRequested(), false))
                        .withHasEcommerce(nvl(counter.getHasEcommerce(), false))
                )
                .toSet();
    }

    /**
     * @param source тип счетчика Метрики
     * @return {@code true}, если счетчик подходит для коллтрекинга на сайте, то есть счетчик является пользовательским
     */
    private static boolean isCalltrackingOnSiteCompatible(@Nullable GdMetrikaCounterSource source) {
        return source == null;
    }

    private Set<MetrikaCounterForSuggest> getAdditionalCountersForSuggest(
            @Nullable Set<Long> additionalCounterIds,
            GdMetrikaCountersByDomainPayload metrikaCountersByDomain,
            User subjectUser) {
        if (isEmpty(additionalCounterIds)) {
            return emptySet();
        }

        Set<GdClientMetrikaCounter> allAccessibleCounters = metrikaCountersByDomain.getAllAccessibleCounters();
        Set<Long> accessibleCounterIds = mapSet(allAccessibleCounters,
                GdClientMetrikaCounter::getId);

        Map<Long, GetExistentCountersResponseItem> existentCounterById = getExistentCounterById(additionalCounterIds);
        Map<Long, GdMetrikaCounterSource> sourceById = EntryStream.of(existentCounterById)
                .mapValues(counter -> convertCounterSource(counter.getCounterSource()))
                .nonNullValues()
                .toMap();
        Map<Long, Boolean> hasEcommerceById = EntryStream.of(existentCounterById)
                .mapValues(GetExistentCountersResponseItem::getEcommerce)
                .nonNullValues()
                .toMap();

        Map<Long, Boolean> isEditableByOperatorById = StreamEx.of(allAccessibleCounters)
                .mapToEntry(GdClientMetrikaCounter::getId, GdClientMetrikaCounter::getIsEditableByOperator)
                .nonNullValues()
                .toMap();

        Map<Long, Boolean> isCalltrackingOnSiteCompatible = StreamEx.of(allAccessibleCounters)
                .mapToEntry(GdClientMetrikaCounter::getId, c -> isCalltrackingOnSiteCompatible(c.getSource()))
                .nonNullValues()
                .toMap();

        Map<Long, Boolean> inaccessibleCounterIdToAccessRequested =
                getInaccessibleCounterIdToAccessRequested(additionalCounterIds, accessibleCounterIds,
                        metrikaCountersByDomain.getInaccessibleCountersByDomain(), subjectUser);

        return StreamEx.of(additionalCounterIds)
                .map(counterId -> {
                    boolean isAccessible = accessibleCounterIds.contains(counterId);
                    return new MetrikaCounterForSuggest()
                            .withId(counterId)
                            .withSource(sourceById.get(counterId))
                            .withHasEcommerce(hasEcommerceById.get(counterId))
                            .withEditableByOperator(isEditableByOperatorById.getOrDefault(counterId, false))
                            .withCalltrackingOnSiteCompatible(isCalltrackingOnSiteCompatible.getOrDefault(counterId,
                                    false))
                            .withAccessible(isAccessible)
                            .withAccessRequested(isAccessible ? null
                                    : inaccessibleCounterIdToAccessRequested.getOrDefault(counterId, false));
                })
                .toSet();
    }

    private Map<Long, Boolean> getInaccessibleCounterIdToAccessRequested(
            Set<Long> additionalCounterIds,
            Set<Long> allAccessibleCounterIds,
            Set<GdInaccessibleMetrikaCounter> inaccessibleCountersByDomain,
            User subjectUser) {
        Map<Long, Boolean> inaccessibleCounterIdToAccessRequested = StreamEx.of(inaccessibleCountersByDomain)
                .mapToEntry(GdInaccessibleMetrikaCounter::getId, GdInaccessibleMetrikaCounter::getAccessRequested)
                .nonNullValues()
                .toMap();
        Set<Long> inaccessibleCounterIdsToGetAccessRequested = StreamEx.of(additionalCounterIds)
                .remove(allAccessibleCounterIds::contains)
                .remove(inaccessibleCounterIdToAccessRequested::containsKey)
                .toSet();
        Map<Long, Boolean> grantAccessRequestedByCounterId =
                campMetrikaCountersService.getGrantAccessRequestedByCounterId(subjectUser.getUid(),
                        inaccessibleCounterIdsToGetAccessRequested);
        return EntryStream.of(inaccessibleCounterIdToAccessRequested)
                .append(grantAccessRequestedByCounterId).toMap();
    }

    private Set<GdGoal> getGoalsForSuggest(Long operatorUid,
                                           User subjectUser,
                                           Set<MetrikaCounterForSuggest> countersToFetchGoals) {
        Set<MetrikaCounterForSuggest> counters = StreamEx.of(countersToFetchGoals)
                .filter(counter -> counter.getSource() != SPRAV)
                .toSet();
        if (counters.isEmpty()) {
            return emptySet();
        }

        var counterIds = mapSet(counters, MetrikaCounterForSuggest::getId);
        var unavailableEcommerceCounterIds = filterAndMapToSet(counters,
                c -> !c.getAccessible() && nvl(c.getHasEcommerce(), false), MetrikaCounterForSuggest::getId);
        ClientId clientId = subjectUser.getClientId();
        boolean returnUnavailableGoals = needToSuggestAllGoals(subjectUser);
        Set<Goal> goals = goalMutationService.doGetMetrikaGoalsByCounterIds(operatorUid,
                clientId, counterIds, unavailableEcommerceCounterIds, Map.of(), new ForCampaignType(CampaignType.TEXT),
                !returnUnavailableGoals);
        Map<Long, GoalConversionInfo> metrikaConversionsByGoalId = getMetrikaConversionsByGoalId(counterIds);
        // setZeroConversionsForNonCombinedGoalsWithoutStats = false,
        // потому что в саджесте не хотим отдавать красные цели
        var conversionsByGoalId = goalDataService.enrichMetrikaGoalsConversions(clientId,
                goals, metrikaConversionsByGoalId, false);
        goalDataService.populateGoalsByConversionVisits(goals, conversionsByGoalId);

        Map<Long, Boolean> counterIdToAccessible = StreamEx.of(counters)
                .mapToEntry(MetrikaCounterForSuggest::getId, MetrikaCounterForSuggest::getAccessible)
                .distinctKeys()
                .toMap();
        var availableCounterIds = EntryStream.of(counterIdToAccessible)
                .filterValues(Boolean.TRUE::equals)
                .keys()
                .toSet();
        return goalDataService.toGdGoalsWithRemoveConversionsForUnavailable(
                goals, availableCounterIds, returnUnavailableGoals);
    }

    private Map<Long, GoalConversionInfo> getMetrikaConversionsByGoalId(Set<Long> counterIds) {
        try {
            return metrikaGoalsConversionService.getGoalsConversion(counterIds);
        } catch (MetrikaClientException ex) {
            logger.error(ex.getMessage());
            return emptyMap();
        }
    }

    @Nullable
    private String getBusinessCategoryForSuggest(@Nullable GdOrganization organizationForSuggest) {
        if (organizationForSuggest == null) {
            return null;
        }

        return businessCategoriesService.convertSpravBusinessCategory(organizationForSuggest.getRubric());
    }

    private BigDecimal getExpectedCpa(@Nullable String businessCategoryForSuggest) {
        if (businessCategoryForSuggest == null) {
            return null;
        }

        // Заглушка, честная реализация будет позже
        return BigDecimal.valueOf(50);
    }

    private Set<Long> getRegionIdsForSuggest(@Nullable GdOrganization organizationForSuggest) {
        Optional<Long> organizationRegionId = Optional.ofNullable(organizationForSuggest)
                .filter(organization -> !nvl(organization.getIsOnline(), true))
                .map(GdOrganization::getAddress)
                .map(GdAddress::getGeoId)
                .map(Integer::longValue);

        organizationRegionId.ifPresent(regionId -> logger.info("Fetched geo from organization {}: {}",
                organizationForSuggest.getPermalinkId(), regionId));

        Optional<Long> regionIdOptional = organizationRegionId
                .or(currentGeoService::getCurrentRegionId)
                .flatMap(geoBaseHelper::convertToDirectRegionId);

        return regionIdOptional.map(Set::of).orElse(emptySet());
    }

    public List<GdCpaEstimate> getCpaEstimates(User subjectUser,
                                               Long operatorUid,
                                               @Nullable Long campaignId,
                                               Set<GdSuggestCpaGoalInfo> goals,
                                               @Nullable Collection<Long> regionIds,
                                               @Nullable String businessCategory,
                                               @Nullable String url) {
        CurrencyCode currencyCode = clientService.getWorkCurrency(subjectUser.getClientId()).getCode();

        return getCpaEstimates(subjectUser, operatorUid, currencyCode,
                campaignId, goals, regionIds, businessCategory, url);
    }

    private List<GdCpaEstimate> getCpaEstimates(User subjectUser,
                                                Long operatorUid,
                                                CurrencyCode currencyCode,
                                                @Nullable Long campaignId,
                                                Set<GdSuggestCpaGoalInfo> goals,
                                                @Nullable Collection<Long> regionIds,
                                                @Nullable String businessCategory,
                                                @Nullable String url) {
        List<GdCpaEstimate> cpaEstimatesComputed = getCpaEstimatesComputed(subjectUser, operatorUid, campaignId,
                goals, regionIds, businessCategory, url);
        List<GdCpaEstimate> validCpaEstimatesComputed =
                getCpaEstimatesWithValidConversionValue(currencyCode, cpaEstimatesComputed);
        List<GdCpaEstimate> cpaEstimatesFallbacked =
                getCpaEstimatesFallbacked(currencyCode, goals, validCpaEstimatesComputed);

        return StreamEx.of(validCpaEstimatesComputed, cpaEstimatesFallbacked)
                .flatMap(Collection::stream)
                .toList();
    }

    private List<GdCpaEstimate> getCpaEstimatesComputed(User subjectUser,
                                                        Long operatorUid,
                                                        @Nullable Long campaignId,
                                                        Set<GdSuggestCpaGoalInfo> goals,
                                                        @Nullable Collection<Long> regionIds,
                                                        @Nullable String businessCategory,
                                                        @Nullable String url) {
        ClientId clientId = subjectUser.getClientId();

        boolean historicalCpaSuggestEnabled =
                featureService.isEnabledForClientId(subjectUser.getClientId(),
                        UNIVERSAL_CAMPAIGNS_USE_HISTORICAL_CPA_SUGGEST);
        Map<Long, GdRecommendedGoalCostPerAction> cpaEstimateByGoalId = historicalCpaSuggestEnabled ?
                getHistoricalCpaEstimates(operatorUid, clientId, campaignId, goals, url) : emptyMap();

        boolean categoricalCpaSuggestEnabled =
                featureService.isEnabledForClientId(subjectUser.getClientId(),
                        UNIVERSAL_CAMPAIGNS_USE_CATEGORICAL_CPA_SUGGEST);
        Map<MetrikaCounterGoalType, CpaEstimatesContainer> cpaEstimateByGoalType = categoricalCpaSuggestEnabled ?
                getСategoricalCpaEstimates(regionIds, businessCategory) : emptyMap();

        return StreamEx.of(goals)
                .map(goal -> getCpaEstimateForGoal(goal, cpaEstimateByGoalId, cpaEstimateByGoalType))
                .nonNull()
                .toList();
    }

    private Map<Long, GdRecommendedGoalCostPerAction> getHistoricalCpaEstimates(Long operatorUid,
                                                                                ClientId clientId,
                                                                                @Nullable Long campaignId,
                                                                                @Nullable Set<GdSuggestCpaGoalInfo> goals,
                                                                                @Nullable String url) {
        if (isEmpty(goals)) {
            return emptyMap();
        }

        List<Long> goalIds = mapList(goals, GdSuggestCpaGoalInfo::getGoalId);

        boolean unavailableGoalsEnabled = featureService.isEnabledForClientId(clientId, UAC_UNAVAILABLE_GOALS_ALLOWED);
        List<GdRecommendedGoalCostPerAction> historicalCpaEstimate;
        if (campaignId == null) {
            historicalCpaEstimate = goalDataService.getGoalsCostPerActionRecommendations(
                    operatorUid, clientId, goalIds, url, unavailableGoalsEnabled);
        } else {
            CurrencyCode clientCurrencyCode =
                    Optional.ofNullable(clientService.getClient(clientId))
                            .map(cl -> cl.getWorkCurrency().getCurrency().getCode())
                            .orElse(CurrencyCode.RUB);

            var cpaEstimatesByCampaignIds =
                    goalDataService.getCampaignsGoalsCostPerActionRecommendations(operatorUid, clientId,
                            Map.of(campaignId, goalIds), clientCurrencyCode,
                            Map.of(campaignId, nvl(url, "")), unavailableGoalsEnabled);

            historicalCpaEstimate = StreamEx.of(cpaEstimatesByCampaignIds)
                    .flatMap(cpaEstimatesByCampaignId -> cpaEstimatesByCampaignId.getRecommendedGoalsCostPerAction().stream())
                    .toList();
        }

        return StreamEx.of(historicalCpaEstimate)
                .mapToEntry(GdRecommendedGoalCostPerAction::getId, identity())
                .toMap();
    }

    private Map<MetrikaCounterGoalType, CpaEstimatesContainer> getСategoricalCpaEstimates(@Nullable Collection<Long> regionIds,
                                                                                          @Nullable String businessCategory) {
        return cpaEstimatesService.getAvgCpaByGoalType(businessCategory, mapList(regionIds, identity()),
                LAST_YANDEX_DIRECT_CLICK);
    }

    @Nullable
    private GdCpaEstimate getCpaEstimateForGoal(GdSuggestCpaGoalInfo goal,
                                                Map<Long, GdRecommendedGoalCostPerAction> cpaEstimateByGoalId,
                                                Map<MetrikaCounterGoalType, CpaEstimatesContainer> cpaEstimateByGoalType) {
        Long goalId = goal.getGoalId();

        GdRecommendedGoalCostPerAction historicalCpaEstimate = cpaEstimateByGoalId.get(goalId);
        if (historicalCpaEstimate != null && historicalCpaEstimate.getCostPerAction() != null) {
            return new GdCpaEstimate()
                    .withGoalId(goalId)
                    .withGoalType(goal.getGoalType())
                    .withCpa(historicalCpaEstimate.getCostPerAction())
                    .withConversionValue(historicalCpaEstimate.getCostPerAction());
        }

        MetrikaCounterGoalType goalType = GdMetrikaCounterGoalType.toSource(goal.getGoalType());
        CpaEstimatesContainer categoricalCpaEstimate = cpaEstimateByGoalType.get(goalType);
        if (categoricalCpaEstimate != null) {
            return new GdCpaEstimate()
                    .withGoalId(goalId)
                    .withGoalType(goal.getGoalType())
                    .withCpa(categoricalCpaEstimate.getMedianCpa())
                    .withConversionValue(categoricalCpaEstimate.getMedianCpa())
                    .withMinCpa(categoricalCpaEstimate.getMinCpa())
                    .withMaxCpa(categoricalCpaEstimate.getMaxCpa());
        }

        return null;
    }

    private List<GdCpaEstimate> getCpaEstimatesWithValidConversionValue(CurrencyCode currencyCode,
                                                                        List<GdCpaEstimate> cpaEstimatesComputed) {
        return StreamEx.of(cpaEstimatesComputed)
                .filter(cpa -> cpa.getConversionValue() != null &&
                        cpa.getConversionValue().compareTo(currencyCode.getCurrency().getMinPrice()) >= 0)
                .toList();
    }

    private List<GdCpaEstimate> getCpaEstimatesFallbacked(CurrencyCode currencyCode,
                                                          Set<GdSuggestCpaGoalInfo> goals,
                                                          List<GdCpaEstimate> cpaEstimatesComputed) {
        Set<Long> processedGoalIds = listToSet(cpaEstimatesComputed, GdCpaEstimate::getGoalId);
        Set<GdSuggestCpaGoalInfo> unprocessedGoals = StreamEx.of(goals)
                .remove(goal -> processedGoalIds.contains(goal.getGoalId()))
                .toSet();

        BigDecimal fallback = getConversionValueFallback(currencyCode, cpaEstimatesComputed);

        return StreamEx.of(unprocessedGoals)
                .map(goal -> new GdCpaEstimate()
                        .withGoalId(goal.getGoalId())
                        .withGoalType(goal.getGoalType())
                        .withConversionValue(fallback))
                .toList();
    }

    private BigDecimal getConversionValueFallback(CurrencyCode currencyCode, List<GdCpaEstimate> cpaEstimatesComputed) {
        OptionalDouble fallbackFromComputed = StreamEx.of(cpaEstimatesComputed)
                .map(GdCpaEstimate::getCpa)
                .mapToDouble(BigDecimal::doubleValue)
                .max();

        return convertToCommonOptional(fallbackFromComputed)
                .map(BigDecimal::valueOf)
                .orElse(nvl(Currencies.getCurrency(currencyCode).getUcDefaultConversionValue(), BigDecimal.ZERO));
    }

    private Set<GdSuggestCpaGoalInfo> convertGoals(Set<GdGoal> goals) {
        return mapSet(goals, goal -> new GdSuggestCpaGoalInfo()
                .withGoalId(goal.getId())
                .withGoalType(goal.getMetrikaGoalType()));
    }

    private GdCalltrackingOnSiteSuggest getCalltrackingOnSiteForSuggest(
            ClientId clientId,
            Long operatorUid,
            String url
    ) {
        var shard = shardHelper.getShardByClientId(clientId);
        var settings = calltrackingSettingsService.getByUrl(shard, clientId, url);
        if (settings == null) {
            return new GdCalltrackingOnSiteSuggest()
                    .withCampaignIds(emptySet())
                    .withPhones(emptyList());
        }
        Long settingsId = settings.getId();
        var campaignIds = calltrackingSettingsService.getCampaignIds(shard, settingsId);
        var payload = calltrackingOnSiteDataService.getGdCalltrackingOnSitePayload(
                shard,
                clientId,
                operatorUid,
                List.of(settings),
                true
        ).getItems().get(0);
        return new GdCalltrackingOnSiteSuggest()
                .withCampaignIds(campaignIds)
                .withSettingsId(settingsId)
                .withCounterId(settings.getCounterId())
                .withPhones(payload.getCalltrackingPhones())
                .withHasExternalCalltracking(payload.getHasExternalCalltracking());
    }

    public GdSuggestMetrikaDataByUrlPayload getSuggestMetrikaDataByUrl(GridGraphQLContext context,
                                                                       GdSuggestMetrikaDataByUrl input) {
        User subjectUser = context.getSubjectUser();
        Long operatorUid = context.getOperator().getUid();
        String url = input.getUrl();
        Set<Long> additionalCounterIds = input.getAdditionalCounterIds();

        Set<MetrikaCounterForSuggest> countersForSuggest = getMetrikaCounterIdsForSuggest(operatorUid, subjectUser, url,
                additionalCounterIds);

        Set<MetrikaCounterForSuggest> countersToFetchGoals;
        if (CollectionUtils.isEmpty(additionalCounterIds)) {
            countersToFetchGoals = countersForSuggest;
        } else {
            Set<Long> inaccessibleAdditionalCounterIds = StreamEx.of(countersForSuggest)
                    .filter(counter -> additionalCounterIds.contains(counter.getId()))
                    .remove(MetrikaCounterForSuggest::getAccessible)
                    .map(MetrikaCounterForSuggest::getId)
                    .toSet();
            Set<Long> allowedInaccessibleAdditionalCounterIds =
                    campMetrikaCountersService.getAllowedInaccessibleCounterIds(inaccessibleAdditionalCounterIds);
            countersToFetchGoals = StreamEx.of(countersForSuggest)
                    .remove(counter -> inaccessibleAdditionalCounterIds.contains(counter.getId())
                            && !allowedInaccessibleAdditionalCounterIds.contains(counter.getId()))
                    .toSet();
        }
        Set<GdGoal> goalsForSuggest = new HashSet<>(
                getGoalsForSuggest(operatorUid, subjectUser, countersToFetchGoals));

        Set<Long> counterIds = new HashSet<>(mapSet(countersForSuggest, MetrikaCounterForSuggest::getId));
        Set<GdClientMetrikaCounterForSuggest> counters = new HashSet<>(mapSet(
                countersForSuggest, ClientDataConverter::toGdMetrikaCounterForSuggest));

        return new GdSuggestMetrikaDataByUrlPayload()
                .withCounterIds(counterIds)
                .withCounters(counters)
                .withGoals(goalsForSuggest);
    }

    public GdShowCpmUacPromoParams getShowCpmUacParams(GdClientInfo clientInfo) {
        var clientId = ClientId.fromLong(clientInfo.getId());
        var showCpmUacPromo = featureService.isEnabledForClientId(clientId, FeatureName.SHOW_CPM_UAC_PROMO);
        var showCpmUacPromoBonus = featureService.isEnabledForClientId(clientId, FeatureName.SHOW_CPM_UAC_PROMO_BONUS);
        var colors = ppcPropertiesSupport.get(PpcPropertyNames.CPM_UAC_PROMO_COLORS).get();
        if (colors == null || colors.size() < 2) {
            colors = DEFAULT_CPM_UAC_PROMO_COLORS;
        }
        return new GdShowCpmUacPromoParams()
                .withShowCpmUacPromo(showCpmUacPromo)
                .withShowCpmUacPromoBonus(showCpmUacPromoBonus)
                .withColorLeft(colors.get(0))
                .withColorRight(colors.get(1));
    }

    public boolean cpmUacNewCamp(GdClientInfo clientInfo) {
        var clientId = ClientId.fromLong(clientInfo.getId());
        if (featureService.isEnabledForClientId(clientId, FeatureName.CPM_UAC_NEW_CAMP)) {
            return true;
        }
        if (!featureService.isEnabledForClientId(clientId, FeatureName.CPM_UAC_NEW_CAMP_LAST)) {
            return false;
        }
        var shard = shardHelper.getShardByClientId(clientId);
        Long lastCid = campaignRepository.getLastCidForType(shard, clientId.asLong(), CampaignType.CPM_BANNER);
        if (lastCid == null) {
            //Если охватных РК не создавали и CPM_UAC_NEW_CAMP выключена,
            // а CPM_UAC_NEW_CAMP_LAST включена, то по умолчанию создаём в uac
            return true;
        }
        return campaignService.getCampaigns(clientId, List.of(lastCid))
                .stream()
                .map(Campaign::getSource)
                .anyMatch(it -> it == CampaignSource.UAC);
    }

    public List<Long> getNotArchivedConversionCampaignIds(int shard, ClientId clientId) {
        var campaigns = campaignTypedRepository.getSafelyNotArchivedCampaignsWithType(
                shard,
                clientId,
                Set.of(
                        CampaignType.DYNAMIC,
                        CampaignType.MOBILE_CONTENT,
                        CampaignType.PERFORMANCE,
                        CampaignType.TEXT
                ),
                CampaignWithStrategy.class);
        return filterAndMapList(campaigns,
                ClientDataService::shouldCleanBidModifiers,
                BaseCampaign::getId);
    }

    private static boolean shouldCleanBidModifiers(CampaignWithStrategy campaign) {
        return STRATEGY_TYPES_FOR_CLEANING_BID_MODIFIERS.contains(campaign.getStrategy().getStrategyName()) &&
                // Для РМП кампании цель = null когда выбираем 'Установки приложения'
                (AUTOBUDGET_AVG_CPI == campaign.getStrategy().getStrategyName() || campaign.getStrategy().getStrategyData().getGoalId() != null) &&
                //Не удаляем корректировки на мастер кампаниях
                !AvailableCampaignSources.INSTANCE.isUC(campaign.getSource());
    }

    @Nullable
    private GdSuggestSocialNetworkAccount extractSocialNetworkAccount(@Nullable String url) {
        var username = SocialNetworkAccountDomainUtils.extractInstagramUsername(url);
        if (username == null) {
            return null;
        }
        return new GdSuggestSocialNetworkAccount()
                .withUsername(username)
                .withType(GdSocialNetworkAccountType.INSTAGRAM);
    }

    private PhoneVerificationStatus getPhoneVerificationStatus(PhoneVerificationStatus phoneVerificationStatusFromBd,
                                                               ClientId clientId) {
        boolean isPhoneVerificationStatusEnabled = featureService.isEnabledForClientId(clientId,
                PHONE_VERIFICATION_STATUS_ENABLED);
        return isPhoneVerificationStatusEnabled ? phoneVerificationStatusFromBd : PhoneVerificationStatus.VERIFIED;
    }
}
