package ru.yandex.direct.grid.processing.service.client;

import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.container.CalloutSelection;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.service.CalloutService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.image.container.ImageFilterContainer;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.cliententity.GdCallout;
import ru.yandex.direct.grid.processing.model.cliententity.GdCalloutsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolanding;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolandingsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdTypedCreative;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImage;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageOrderBy;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContext;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdDeleteCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdDeleteCalloutsPayload;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCalloutsPayload;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcardsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcardsContext;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcards;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcardsPayload;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.client.container.CreativesCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.client.container.ImagesCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter;
import ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter;
import ru.yandex.direct.grid.processing.service.client.converter.VcardDataConverter;
import ru.yandex.direct.grid.processing.service.client.validation.ClientEntityValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Functions.identity;
import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.grid.processing.model.cliententity.image.GdImageOrderByField.CREATE_TIME;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.extractCallouts;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toCreativeFilterContainer;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toCreativesCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdCreativesContext;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdImagesContext;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toImageFilterContainer;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toImagesCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.client.converter.VcardDataConverter.toGdAddVcardsPayloadItems;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class ClientEntityDataService {

    private static final List<GdImageOrderBy> DEFAULT_IMAGES_ORDER_BY = singletonList(
            new GdImageOrderBy()
                    .withField(CREATE_TIME)
                    .withOrder(Order.DESC));
    static final int CLIENT_ENTITY_FETCH_LIMIT = 10_000;

    private final CalloutService calloutService;
    private final CreativeRepository creativeRepository;
    private final ImageDataRepository imageDataRepository;
    private final TurboLandingRepository turbolandingRepository;
    private final GridValidationService gridValidationService;
    private final ClientEntityValidationService clientEntityValidationService;
    private final GridCacheService gridCacheService;
    private final ClientEntityValidationService clientEntityCalloutValidationService;
    private final VcardService vcardService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public ClientEntityDataService(CalloutService calloutService,
                                   CreativeRepository creativeRepository,
                                   ImageDataRepository imageDataRepository,
                                   TurboLandingRepository turboLandingRepository,
                                   GridValidationService gridValidationService,
                                   ClientEntityValidationService clientEntityValidationService,
                                   GridCacheService gridCacheService,
                                   ClientEntityValidationService clientEntityCalloutValidationService,
                                   VcardService vcardService) {
        this.calloutService = calloutService;
        this.creativeRepository = creativeRepository;
        this.imageDataRepository = imageDataRepository;
        this.turbolandingRepository = turboLandingRepository;
        this.gridValidationService = gridValidationService;
        this.clientEntityValidationService = clientEntityValidationService;
        this.gridCacheService = gridCacheService;
        this.clientEntityCalloutValidationService = clientEntityCalloutValidationService;
        this.vcardService = vcardService;
    }

    List<GdCallout> getClientCallouts(GdClientInfo info, GdCalloutsContainer input) {
        List<Callout> callouts = calloutService
                .getCallouts(ClientId.fromLong(info.getId()),
                        new CalloutSelection().withDeleted(input.getFilter().getDeleted()), LimitOffset.maxLimited());
        return mapList(callouts, ClientDataConverter::toGdCallout);
    }

    List<GdTurbolanding> getClientTurbolandings(GdClientInfo info, GdTurbolandingsContainer input) {
        Set<Long> turbolandingIdIn = input.getFilter().getTurbolandingIdIn();

        List<TurboLanding> turboLandings = turbolandingRepository
                .getClientTurbolandings(info.getShard(), ClientId.fromLong(info.getId()), turbolandingIdIn);

        return mapList(turboLandings, ClientDataConverter::toGdTurbolanding);
    }


    public GdSaveCalloutsPayload saveCallouts(GdSaveCallouts input, ClientId clientId) {
        clientEntityValidationService.validateSaveCalloutsRequest(input);

        List<Callout> callouts = extractCallouts(input, clientId);
        MassResult<Long> saveCalloutsResult = calloutService.addCalloutsPartial(clientId, callouts);

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(saveCalloutsResult, path(field(GdSaveCallouts.SAVE_ITEMS)));

        return new GdSaveCalloutsPayload()
                .withCalloutIds(getSuccessfullyUpdatedIds(saveCalloutsResult, identity()))
                .withValidationResult(validationResult);
    }

    public GdDeleteCalloutsPayload deleteCallouts(GdDeleteCallouts input, ClientId clientId) {
        clientEntityValidationService.validateDeleteCalloutsRequest(input);
        boolean detachBeforeDelete = nvl(input.getDetachBeforeDelete(), false);
        MassResult<Long> deleteCalloutsResult = detachBeforeDelete ?
                calloutService.detachAndDeleteCallouts(input.getCalloutIds(), clientId) :
                calloutService.deleteCallouts(input.getCalloutIds(), clientId);

        GdValidationResult validationResult = gridValidationService
                .getValidationResult(deleteCalloutsResult, path(field(GdDeleteCallouts.CALLOUT_IDS)));

        return new GdDeleteCalloutsPayload()
                .withCalloutIds(getSuccessfullyUpdatedIds(deleteCalloutsResult, identity()))
                .withValidationResult(validationResult);
    }

    public GdImagesContext getClientImages(GdClientInfo gdClientInfo, GdImagesContainer input) {
        clientEntityValidationService.validateGdImagesContainer(input);
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        ClientId clientId = ClientId.fromLong(gdClientInfo.getId());
        List<GdImageOrderBy> ordersBy = isNotEmpty(input.getOrderBy()) ? input.getOrderBy() : DEFAULT_IMAGES_ORDER_BY;
        ImagesCacheRecordInfo recordInfo = toImagesCacheRecordInfo(clientId.asLong(), input, ordersBy);

        Optional<GdImagesContext> res = gridCacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        ImageFilterContainer filterContainer = toImageFilterContainer(input.getFilter());
        List<Image> images = imageDataRepository.getImages(gdClientInfo.getShard(), clientId, filterContainer);
        Comparator<Image> comparator = convertToComparator(ordersBy);
        images.sort(comparator);

        List<GdImage> rowsetFull = mapList(images, ClientEntityConverter::toGdImage);
        GdImagesContext gdImagesContext = toGdImagesContext(rowsetFull);

        return gridCacheService.getResultAndSaveToCache(recordInfo, gdImagesContext, rowsetFull, range);
    }

    private static Comparator<Image> convertToComparator(List<GdImageOrderBy> orderByItems) {
        Comparator<Image> comparator = null;
        for (var orderBy : orderByItems) {
            Comparator<Image> nextComparator;
            //Поменять этот if на switch, когда и если появятся другие типы полей сортировки
            if (orderBy.getField() == CREATE_TIME) {
                nextComparator = Comparator.comparing(Image::getCreateTime);
            } else {
                throw new UnsupportedOperationException(format("Unknown order field: '%s'", orderBy.getField()));
            }
            if (notEquals(orderBy.getOrder(), Order.ASC)) {
                nextComparator = nextComparator.reversed();
            }
            comparator = comparator != null ? comparator.thenComparing(nextComparator) : nextComparator;
        }
        return comparator;
    }

    GdCreativesContext getClientCreatives(GdClientInfo gdClientInfo, GdCreativesContainer input) {
        clientEntityValidationService.validateGdCreativesContainer(input);
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        ClientId clientId = ClientId.fromLong(gdClientInfo.getId());
        CreativesCacheRecordInfo recordInfo = toCreativesCacheRecordInfo(clientId.asLong(), input);

        Optional<GdCreativesContext> res = gridCacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        List<Creative> creatives = creativeRepository
                .getCreatives(gdClientInfo.getShard(), clientId, toCreativeFilterContainer(input.getFilter()));

        List<GdTypedCreative> rowsetFull = mapList(creatives, ClientEntityConverter::toGdCreativeImplementation);
        GdCreativesContext gdCreativesContext = toGdCreativesContext(rowsetFull);
        return gridCacheService.getResultAndSaveToCache(recordInfo, gdCreativesContext, rowsetFull, range);
    }

    GdVcardsContext getVcards(Long operatorUid, GdClientInfo info, GdVcardsContainer input) {
        List<Vcard> vcards = vcardService
                .getVcards(operatorUid, ClientId.fromLong(info.getId()), input.getFilter().getVcardIdIn(),
                        LimitOffset.limited(CLIENT_ENTITY_FETCH_LIMIT));

        return new GdVcardsContext()
                .withRowset(mapList(vcards, VcardDataConverter::toGdVcard));
    }

    GdAddVcardsPayload addVcards(Long operatorUid, ClientId clientId, GdAddVcards input) {
        clientEntityCalloutValidationService.validateAddVcardsRequest(input);

        List<Vcard> vcards = mapList(input.getVcardAddItems(), VcardDataConverter::toVcard);
        MassResult<Long> result = vcardService.addVcardsPartial(vcards, operatorUid, clientId);

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddVcards.VCARD_ADD_ITEMS)));
        return new GdAddVcardsPayload()
                .withAddedVcards(toGdAddVcardsPayloadItems(result))
                .withValidationResult(validationResult);
    }

}
