package ru.yandex.direct.grid.processing.service.client;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.cliententity.GdCallout;
import ru.yandex.direct.grid.processing.model.cliententity.GdCalloutsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolanding;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolandingsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContext;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdDeleteCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdDeleteCalloutsPayload;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCalloutsPayload;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcardsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcardsContext;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcards;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcardsPayload;
import ru.yandex.direct.grid.processing.service.client.validation.ClientEntityValidationService;

/**
 * Сервис, возвращающий данные об объектах привязанных к клиенту
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ClientEntityGraphQlService {

    private ClientEntityDataService clientEntityDataService;
    private ClientDataService clientDataService;
    private ClientEntityValidationService clientEntityValidationService;

    public ClientEntityGraphQlService(ClientEntityDataService clientObjectsDataService,
                                      ClientDataService clientDataService,
                                      ClientEntityValidationService clientEntityValidationService) {
        this.clientEntityDataService = clientObjectsDataService;
        this.clientDataService = clientDataService;
        this.clientEntityValidationService = clientEntityValidationService;
    }

    /**
     * GraphQL подзапрос. Получает информацию о всех неудаленных уточнениях клиента,
     * полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "callouts")
    public List<GdCallout> getCallouts(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCalloutsContainer input) {
        clientEntityValidationService.validateGdCalloutsContainer(input);

        GdClientInfo info = clientDataService.getClientInfo(context, input.getSearchBy());
        return clientEntityDataService.getClientCallouts(info, input);
    }


    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "saveCallouts")
    public GdSaveCalloutsPayload saveCallouts(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSaveCallouts input) {
        return clientEntityDataService.saveCallouts(input, context.getSubjectUser().getClientId());
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "deleteCallouts")
    public GdDeleteCalloutsPayload deleteCallouts(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDeleteCallouts input) {
        return clientEntityDataService.deleteCallouts(input, context.getSubjectUser().getClientId());
    }


    /**
     * GraphQL подзапрос. Получает информацию о всех турболендингах клиента,
     * полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "turbolandings")
    public List<GdTurbolanding> getTurbolandings(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdTurbolandingsContainer input) {
        clientEntityValidationService.validateGdTurbolandingsContainer(input);

        GdClientInfo info = clientDataService.getClientInfo(context, input.getSearchBy());
        return clientEntityDataService.getClientTurbolandings(info, input);
    }

    /**
     * GraphQL подзапрос. Получает информацию об изображениях клиента,
     * полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "images")
    public GdImagesContext getImages(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdImagesContainer input) {
        GdClientInfo info = clientDataService.getClientInfo(context, input.getSearchBy());

        return clientEntityDataService.getClientImages(info, input);
    }

    /**
     * GraphQL подзапрос. Получает информацию о креативах клиента,
     * полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "creatives")
    public GdCreativesContext getCreatives(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCreativesContainer input) {
        GdClientInfo info = clientDataService.getClientInfo(context, input.getSearchBy());
        return clientEntityDataService.getClientCreatives(info, input);
    }

    /**
     * GraphQL подзапрос. Получает информацию о виртуальных визитках клиента,
     * полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "vcards")
    public GdVcardsContext getVcards(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdVcardsContainer input) {
        clientEntityValidationService.validateGdVcardsContainer(input);

        GdClientInfo info = clientDataService.getClientInfo(context, input.getSearchBy());
        return clientEntityDataService.getVcards(context.getOperator().getUid(), info, input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addVcards")
    public GdAddVcardsPayload addVcards(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddVcards input) {
        //noinspection ConstantConditions
        return clientEntityDataService
                .addVcards(context.getOperator().getUid(), context.getSubjectUser().getClientId(), input);
    }

}
