package ru.yandex.direct.grid.processing.service.client;

import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType;
import ru.yandex.direct.grid.model.entity.campaign.converter.GdCampaignFeatureHelper;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorHelper;
import ru.yandex.direct.grid.processing.model.client.GdClientFeatures;
import ru.yandex.direct.grid.processing.util.AggregationStatsCalculator;
import ru.yandex.direct.utils.CommonUtils;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType.AVG_CPA;
import static ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType.AVG_CPA_PER_CAMP;
import static ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType.AVG_CPA_PER_FILTER;
import static ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType.OPTIMIZE_CONVERSIONS;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignType;

@ParametersAreNonnullByDefault
public class ClientFeatureCalculator {
    public static final Set<String> FEATURES_DEPENDS_ON_CAMPAIGNS = Set.of(
            GdClientFeatures.IS_WALLET_ENABLED.name(),
            GdClientFeatures.HAS_METRIKA_COUNTERS.name(),
            GdClientFeatures.HAS_UNIVERSAL_CAMPAIGN.name(),
            GdClientFeatures.HAS_NON_UNIVERSAL_CAMPAIGN.name(),
            GdClientFeatures.HAS_CAMPAIGNS_WITH_C_P_A_STRATEGY.name(),
            GdClientFeatures.HAS_ECOMMERCE.name(),
            GdClientFeatures.USED_CAMPAIGN_TYPES.name(),
            GdClientFeatures.CAMPAIGN_MANAGERS.name(),
            GdClientFeatures.HAS_CAMPAIGNS_WITH_STATS.name(),
            GdClientFeatures.HAS_CAMPAIGNS_WITH_SHOWS.name(),
            GdClientFeatures.HAS_CAMPAIGNS_WITH_COMPLETED_MEDIAPLAN.name(),
            GdClientFeatures.CAMPAIGN_ID_FOR_V_CARDS_MANAGEMENT.name());

    private static final Set<GdStrategyType> CPA_STRATEGIES =
            ImmutableSet.of(AVG_CPA, AVG_CPA_PER_CAMP, AVG_CPA_PER_FILTER, OPTIMIZE_CONVERSIONS);

    /**
     * Важно: при добавлении расчета нового поля, не забыть добавить его в {@link #FEATURES_DEPENDS_ON_CAMPAIGNS}
     */
    static final AggregationStatsCalculator<GdiBaseCampaign, GdClientFeatures> FEATURE_CALCULATOR =
            AggregationStatsCalculator.<GdiBaseCampaign, GdClientFeatures>builder(GdClientFeatures::new)
                    // Есть хотя бы одна кампания с кошельком
                    .hasValue(GdClientFeatures::setIsWalletEnabled, GdiBaseCampaign::getWalletId,
                            CommonUtils::isValidId)
                    // Есть хотя бы одна кампания со счетчиком метрики
                    .hasValue(GdClientFeatures::setHasMetrikaCounters, GdiBaseCampaign::getMetrikaCounters,
                            CollectionUtils::isNotEmpty)
                    // Есть хотя бы одна неуниверсальная кампания
                    .hasValue(GdClientFeatures::setHasNonUniversalCampaign, GdiBaseCampaign::getIsUniversal,
                            isUniversal -> !requireNonNull(isUniversal, "isUniversal must be set for all campaigns"))
                    // Есть хотя бы одна универсальная кампания
                    .hasValue(GdClientFeatures::setHasUniversalCampaign, GdiBaseCampaign::getSource,
                            source -> AvailableCampaignSources.INSTANCE.isUC(GdiCampaignSource.toSource(source)))
                    // Есть хотя бы одна кампания с CPA-стратегией
                    .hasValue(GdClientFeatures::setHasCampaignsWithCPAStrategy,
                            GdStrategyExtractorHelper::toStrategyType, CPA_STRATEGIES::contains)
                    // Есть хотя бы одна кампания с ecommerсe целью
                    .hasValue(GdClientFeatures::setHasEcommerce, GdiBaseCampaign::getHasEcommerce, f -> f != null && f)
                    // Список типов кампаний, которые есть у клиента
                    .valuesSet(GdClientFeatures::setUsedCampaignTypes,
                            gdiCampaign -> toGdCampaignType(gdiCampaign.getType()))
                    // Список менеджеров кампаний, которые есть у клиента
                    .valuesSet(GdClientFeatures::setCampaignManagers, GdiBaseCampaign::getManagerUserId, Objects::nonNull)
                    // Есть хотя бы одна кампания со статистикой
                    .hasValue(GdClientFeatures::setHasCampaignsWithStats, GdiBaseCampaign::getOrderId,
                            CommonUtils::isValidId)
                    // Есть хотя бы одна кампания c показами
                    .hasValue(GdClientFeatures::setHasCampaignsWithShows, GdiBaseCampaign::getShows,
                            s -> s != null && s > 0)
                    // Есть хотя бы одна кампания c готовым медиапланом
                    .hasValue(GdClientFeatures::setHasCampaignsWithCompletedMediaplan,
                            GdCampaignFeatureHelper::hasCompletedMediaplan)
                    // ID первой подходящей кампании для правки визиток
                    .aggregate(GdClientFeatures::setCampaignIdForVCardsManagement,
                            c -> !c.getArchived() && c.getType() == CampaignType.TEXT && c.getActions().getCanEdit() ?
                                    c.getId() : null,
                            (oi, ni) -> oi == null ? ni : oi,
                            Objects::nonNull)
                    .build();

    private ClientFeatureCalculator() {
    }
}
