package ru.yandex.direct.grid.processing.service.client;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLEnvironment;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import io.leangen.graphql.execution.ResolutionEnvironment;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.CampAdditionalData;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.repository.CampAdditionalDataRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.client.mcc.ClientMccService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdMobileContentSuggestInfo;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignNotification;
import ru.yandex.direct.grid.model.campaign.notification.GdDefaultCampaignNotification;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdAccountScoreInfo;
import ru.yandex.direct.grid.processing.model.client.GdAvailableFields;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientAccess;
import ru.yandex.direct.grid.processing.model.client.GdClientConstants;
import ru.yandex.direct.grid.processing.model.client.GdClientExternalDataPayload;
import ru.yandex.direct.grid.processing.model.client.GdClientExternalOrganizationDataEntry;
import ru.yandex.direct.grid.processing.model.client.GdClientFeatures;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientMccCommonInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientMeasurerSystem;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounter;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounters;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCountersWithAdditionalInformation;
import ru.yandex.direct.grid.processing.model.client.GdClientNotificationInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientSearchRequest;
import ru.yandex.direct.grid.processing.model.client.GdClientTopRegionInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientTopRegions;
import ru.yandex.direct.grid.processing.model.client.GdClientTopRegionsPayload;
import ru.yandex.direct.grid.processing.model.client.GdCoreFeatureWithDescription;
import ru.yandex.direct.grid.processing.model.client.GdCpaEstimate;
import ru.yandex.direct.grid.processing.model.client.GdInternalDontShowDomainsDataPayload;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersByDomain;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersByDomainPayload;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersFilter;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersFilterPayload;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersInfoContainer;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCountersInfoPayload;
import ru.yandex.direct.grid.processing.model.client.GdShowCpmUacPromoParams;
import ru.yandex.direct.grid.processing.model.client.GdSuggestCpa;
import ru.yandex.direct.grid.processing.model.client.GdSuggestCpaGoalInfo;
import ru.yandex.direct.grid.processing.model.client.GdSuggestCpaPayload;
import ru.yandex.direct.grid.processing.model.client.GdSuggestDataByUrl;
import ru.yandex.direct.grid.processing.model.client.GdSuggestDataByUrlPayload;
import ru.yandex.direct.grid.processing.model.client.GdSuggestMetrikaDataByUrl;
import ru.yandex.direct.grid.processing.model.client.GdSuggestMetrikaDataByUrlPayload;
import ru.yandex.direct.grid.processing.model.client.GdUserFeature;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.model.client.GdXivaSecretSign;
import ru.yandex.direct.grid.processing.model.cliententity.GdClientMeasurerAccountDetails;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.processor.util.ClientResolverFetchedFieldsUtil;
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFields;
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFieldsHolder;
import ru.yandex.direct.grid.processing.service.banner.BannerDataService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.client.converter.ToGdClientExternalDataEntryConverters;
import ru.yandex.direct.grid.processing.service.operator.OperatorAccessService;
import ru.yandex.direct.grid.processing.service.organizations.OrganizationsDataService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.xiva.client.XivaClient;
import ru.yandex.direct.xiva.client.model.Signature;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isChiefRep;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий данные о клиентах и принадлежащих им объектах
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ClientGraphQlService {
    private static final Logger logger = LoggerFactory.getLogger(ClientGraphQlService.class);

    public static final String CLIENT_RESOLVER_NAME = "client";
    public static final String FEATURES_RESOLVER_NAME = "features";
    static final String ACCOUNT_SCORE_INFO_RESOLVER_NAME = "accountScoreInfo";
    static final String IS_OFFER_ACCEPTED_RESOLVER_NAME = "isOfferAccepted";
    static final String USER_FEATURES_RESOLVER_NAME = "userFeatures";
    static final String XIVA_SECRET_SIGN_RESOLVER_NAME = "xivaSecretSign";

    private static final String ORGANIZATIONS_EXTERNAL_DATA_LINKS_INFO_LIST_FIELD = "linksInfoList";
    private static final String EXTERNAL_DATA_ENTRY_ORGANIZATION_FIELD = "organization";

    static {
        checkState(ORGANIZATIONS_EXTERNAL_DATA_LINKS_INFO_LIST_FIELD.equals(GdClientExternalDataPayload.LINKS_INFO_LIST.name()));
        checkState(EXTERNAL_DATA_ENTRY_ORGANIZATION_FIELD.equals(GdClientExternalOrganizationDataEntry.ORGANIZATION.name()));
    }


    private final PpcRbac ppcRbac;
    private final RbacService rbacService;
    private final ClientDataService clientDataService;
    private final CampaignInfoService campaignsInfoService;
    private final OperatorAccessService operatorAccessService;
    private final GridValidationService gridValidationService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final OrganizationsDataService organizationsDataService;
    private final GridCampaignService gridCampaignService;
    private final CampAdditionalDataRepository campAdditionalDataRepository;
    private final XivaClient xivaClient;
    private final BannerDataService bannerDataService;
    private final ClientMccService clientMccService;

    @Autowired
    public ClientGraphQlService(PpcRbac ppcRbac, ClientDataService clientDataService,
                                CampaignInfoService campaignsInfoService,
                                OperatorAccessService operatorAccessService,
                                GridValidationService gridValidationService,
                                CampMetrikaCountersService campMetrikaCountersService,
                                OrganizationsDataService organizationsDataService,
                                GridCampaignService gridCampaignService,
                                CampAdditionalDataRepository campAdditionalDataRepository,
                                XivaClient xivaClient,
                                BannerDataService bannerDataService,
                                ClientMccService clientMccService,
                                RbacService rbacService) {
        this.ppcRbac = ppcRbac;
        this.rbacService = rbacService;
        this.clientDataService = clientDataService;
        this.campaignsInfoService = campaignsInfoService;
        this.operatorAccessService = operatorAccessService;
        this.gridValidationService = gridValidationService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.organizationsDataService = organizationsDataService;
        this.gridCampaignService = gridCampaignService;
        this.campAdditionalDataRepository = campAdditionalDataRepository;
        this.xivaClient = xivaClient;
        this.bannerDataService = bannerDataService;
        this.clientMccService = clientMccService;
    }

    /**
     * GraphQL ручка. Получает информацию о клиенте, полученном по переданным параметрам поиска
     *
     * @param searchRequest параметры поиска
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CLIENT_RESOLVER_NAME)
    public GdClient getClient(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "searchBy") GdClientSearchRequest searchRequest,
            @GraphQLEnvironment ResolutionEnvironment resolutionEnvironment) {
        gridValidationService.validateClientSearchRequest(searchRequest);
        String login = searchRequest.getLogin();
        GdClientInfo info = null;
        Map<String, GdClientInfo> clientInfoByLogin = context.getClientInfoByLogin();
        // если понадобится кэширование по uid или ClientID, можно добавить
        if (login != null && clientInfoByLogin.containsKey(login)) {
            info = clientInfoByLogin.get(login);
        }
        if (info == null) {
            info = clientDataService.getClientInfo(context, searchRequest);
            info.withClientMccCommonInfo(getClientMccCommonInfo(context.getOperator(), info.getChiefUser()));
            if (login != null) {
                clientInfoByLogin.put(login, info);
            }
        }
        // Кладем, чтобы дальше было понятно с каким клиентом работаем
        context.setQueriedClient(info);

        String chiefLogin = ppcRbac.getChiefLoginClientId(info.getId());

        context.setFetchedFieldsReslover(ClientResolverFetchedFieldsUtil.resolve(resolutionEnvironment));

        return new GdClient()
                .withChiefLogin(chiefLogin)
                .withInfo(info)
                .withClientConstants(new GdClientConstants());
    }

    private GdClientMccCommonInfo getClientMccCommonInfo(User operator, GdUserInfo chiefUser) {
        var hasControlRequests = clientMccService.hasControlRequests(ClientId.fromLong(chiefUser.getClientId()));
        var canManageControlRequests = hasControlRequests && (isClient(operator) ? isChiefRep(operator)
                : rbacService.canWrite(operator.getUid(), chiefUser.getUserId()));
        return new GdClientMccCommonInfo()
                .withHasManagedClients(clientMccService.hasManagedClients(chiefUser.getUserId()))
                .withHasControlRequests(hasControlRequests)
                .withCanManageControlRequests(canManageControlRequests)
                .withCanUseClientMccCommonInfo(false);
    }

    /**
     * GraphQL ручка. Получает информацию о используемых клиентом возможностях
     */
    @GraphQLNonNull
    @GraphQLQuery(name = FEATURES_RESOLVER_NAME)
    public GdClientFeatures getClientFeatures(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLEnvironment ResolutionEnvironment resolutionEnvironment) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());

        boolean anyFeatureDependsOnCampaignsRequested =
                ClientResolverFetchedFieldsUtil.isAnyFeatureDependsOnCampaignsRequested(resolutionEnvironment);
        if (anyFeatureDependsOnCampaignsRequested) {
            List<GdiBaseCampaign> campaignsWithoutWallets =
                    campaignsInfoService.getAllBaseCampaignsWithoutWallets(clientId);
            return clientDataService.getClientFeatures(clientId, context.getOperator(), campaignsWithoutWallets);
        }
        return clientDataService.getClientFeaturesIndependentOfCampaigns(clientId, context.getOperator());
    }

    /**
     * Резолвер для GdClientFeatures. Получает информацию о публичных фичах клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "publicCoreFeatures")
    public Set<GdCoreFeatureWithDescription> getPublicCoreFeatures(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClientFeatures gdClientFeatures) {
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());

        return clientDataService.getClientPublicCoreFeatures(clientId);
    }

    /**
     * GraphQL ручка. Получает информацию о правах оператора на клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "access")
    public GdClientAccess getClientAccess(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        GdClientInfo info = client.getInfo();
        return operatorAccessService.getAccess(context.getOperator(), info, context.getSubjectUser(),
                context.getInstant());
    }

    /**
     * GraphQL запрос. Получить данные о разрешенных для клиента полях.
     * Удалится в https://st.yandex-team.ru/DIRECT-142815
     */
    @Deprecated(forRemoval = true)
    @GraphQLNonNull
    @GraphQLQuery(name = "availableFields")
    public GdAvailableFields getAvailableFieldsOld(@GraphQLRootContext GridGraphQLContext context,
                                                   @GraphQLContext GdClientAccess access) {
        return clientDataService.getAvailableFields(context.getSubjectUser().getClientId());
    }

    /**
     * GraphQL запрос. Получить данные о разрешенных для клиента полях
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "availableFields")
    public GdAvailableFields getAvailableFields(@GraphQLRootContext GridGraphQLContext context,
                                                @GraphQLContext GdClient client) {
        return clientDataService.getAvailableFields(context.getSubjectUser().getClientId());
    }

    /**
     * GraphQL ручка. Получает информацию о настройках SMS и email-уведомлений клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "notification")
    public GdClientNotificationInfo getSmsPhone(@GraphQLContext GdClient client) {
        return clientDataService.getNotificationInfo(client.getInfo().getChiefUserId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "metrikaCounters")
    public GdClientMetrikaCounters getMetrikaCounters(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        try {
            Set<Integer> availableCounterIds =
                    listToSet(campMetrikaCountersService.getAvailableCounterIdsByClientId(clientId),
                            Math::toIntExact);
            return new GdClientMetrikaCounters()
                    .withCounters(availableCounterIds)
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdClientMetrikaCounters()
                    .withCounters(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "metrikaCountersWithAdditionInformation")
    public GdClientMetrikaCountersWithAdditionalInformation getMetrikaCountersWithAdditionInformation(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        User operator = context.getOperator();

        return clientDataService.getMetrikaCountersWithAdditionInformation(operator.getUid(), clientId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "metrikaCountersFilter")
    public GdMetrikaCountersFilterPayload getMetrikaCountersSearch(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "input") GdMetrikaCountersFilter input) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        User operator = context.getOperator();

        return clientDataService.getMetrikaCountersFilter(operator.getUid(), clientId, input.getCounterIds(),
                input.getCampaignIds(), input.getNameOrId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "metrikaCountersByDomain")
    public GdMetrikaCountersByDomainPayload getMetrikaCountersByDomain(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "input") GdMetrikaCountersByDomain input) {
        User subjectUser = context.getSubjectUser();
        String url = input.getUrl();
        Long operatorUid = context.getOperator().getUid();

        return clientDataService.getMetrikaCountersByUrl(operatorUid, subjectUser, url, false);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "metrikaCountersInfo")
    public GdMetrikaCountersInfoPayload getMetrikaCountersInfo(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMetrikaCountersInfoContainer input) {
        Long uid = checkNotNull(context.getSubjectUser()).getUid();
        Set<Long> counterIds = input.getCounterIds();

        return clientDataService.getMetrikaCountersInfo(uid, mapList(counterIds, identity()));
    }

    @GraphQLQuery(name = "previousAdUrl")
    public String getPreviousAdUrl(
            @GraphQLContext GdClient client
    ) {
        return bannerDataService.getPreviousAdUrl(ClientId.fromLong(client.getInfo().getId()));
    }

    @GraphQLQuery(name = "previousMobileCampaignContent")
    public GdMobileContentSuggestInfo getPreviousMobileCampaignContent(
            @GraphQLContext GdClient client
    ) {
        return campaignsInfoService.getLatestMobileCampaignContentInfo(client.getInfo());
    }

    @GraphQLQuery(name = "defaultCidForVcardEdit")
    public Long getDefaultCidForVcardEdit(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        if (context.getOperator().getIsReadonlyRep()) {
            return null;
        }
        return campaignsInfoService.getFirstNotArchivedTextCampaignId(
                client.getInfo().getShard(), ClientId.fromLong(client.getInfo().getId()), context.getOperator().getUid()
        ).orElse(null);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "clientMetrikaExternalData")
    public GdClientExternalDataPayload getClientMetrikaExternalData(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        User operator = context.getOperator();

        CampMetrikaCountersService.CounterWithAdditionalInformationFilter counterFilter =
                CampMetrikaCountersService.CounterWithAdditionalInformationFilter
                        .defaultFilter()
                        .withAllowedSources(Set.of(MetrikaCounterSource.UNKNOWN));
        Set<GdClientMetrikaCounter> counters = nvl(
                clientDataService
                        .getMetrikaCountersWithAdditionInformation(operator.getUid(), clientId, counterFilter)
                        .getCounters(),
                Set.of());
        var resultList = counters
                .stream()
                .map(ToGdClientExternalDataEntryConverters::fromMetrikaData)
                .filter(t -> !StringUtils.isBlank(t.getUrl()) && StringConstraints.isValidHref(t.getUrl()))
                .collect(Collectors.toList());
        return new GdClientExternalDataPayload().withLinksInfoList(resultList);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "clientOrganizationsExternalData")
    public GdClientExternalDataPayload getClientOrganizationsExternalData(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GridRequestedFields(childFieldPath = {
                    ORGANIZATIONS_EXTERNAL_DATA_LINKS_INFO_LIST_FIELD, //"linksInfoList"
                    EXTERNAL_DATA_ENTRY_ORGANIZATION_FIELD //"organization"
            }) GridRequestedFieldsHolder requestedFieldsHolder) {
        Set<ModelProperty<?, ?>> propertySet =
                new HashSet<>(requestedFieldsHolder.getRequestedModelFields(GdOrganization.allModelProperties()));
        //добавим нужные для запроса проперти в список запрашиваемых
        propertySet.addAll(ToGdClientExternalDataEntryConverters.ORGANIZATION_PROPERTIES_FOR_CLIENT_BASED_SUGGEST);

        var resultList = organizationsDataService
                .getAllClientOrganizations(context.getSubjectUser(), propertySet)
                .stream()
                .filter(ToGdClientExternalDataEntryConverters::hasAnyValidUrl)
                .map(ToGdClientExternalDataEntryConverters::fromOrganizationData)
                .collect(Collectors.toList());
        return new GdClientExternalDataPayload().withLinksInfoList(resultList);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "clientCampaignsDataForSuggest")
    public GdClientExternalDataPayload clientCampaignsDataForSuggest(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {

        var resultList = campAdditionalDataRepository
                .getByClientId(client.getInfo().getShard(), ClientId.fromLong(client.getInfo().getId()))
                .stream()
                .sorted(Comparator.comparingLong(campData -> ((CampAdditionalData) campData).getCid()).reversed())
                .map(ToGdClientExternalDataEntryConverters::fromCampaignData)
                .filter(t -> t.getBusinessCategory() != null || t.getBusinessName() != null || t.getUrl() != null)
                .distinct()
                .collect(Collectors.toList());
        return new GdClientExternalDataPayload().withLinksInfoList(resultList);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "suggestDataByUrl")
    public GdSuggestDataByUrlPayload getSuggestDataByUrl(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSuggestDataByUrl input) {
        return clientDataService.getSuggestDataByUrl(context, input);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "suggestCpa")
    public GdSuggestCpaPayload getSuggestCpa(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSuggestCpa input) {
        User subjectUser = context.getSubjectUser();
        Long operatorUid = context.getOperator().getId();
        Long campaignId = input.getCampaignId();
        Set<GdSuggestCpaGoalInfo> goals = input.getGoals();
        List<Long> regionIds = input.getRegionIds();
        String businessCategory = input.getBusinessCategory();
        String url = input.getUrl();

        List<GdCpaEstimate> cpaEstimates = clientDataService.getCpaEstimates(subjectUser, operatorUid, campaignId,
                goals, regionIds, businessCategory, url);

        return new GdSuggestCpaPayload()
                .withCpaEstimates(cpaEstimates);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "suggestMetrikaDataByUrl")
    public GdSuggestMetrikaDataByUrlPayload getSuggestMetrikaDataByUrl(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSuggestMetrikaDataByUrl input) {
        return clientDataService.getSuggestMetrikaDataByUrl(context, input);
    }

    /**
     * GraphQL ручка. Получает информацию о числе кампаний клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "campaignsCount")
    public Integer getClientCampaignsCount(
            @GraphQLContext GdClient client) {
        Integer shard = client.getInfo().getShard();
        Long clientId = client.getInfo().getId();

        return gridCampaignService.getCampaignsCount(shard, ClientId.fromLong(clientId));
    }

    // TODO: перейти на новую ручку

    /**
     * GraphQL ручка. Отдает уведомления по умолчанию для создания кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "defaultCampaignNotification")
    public GdCampaignNotification getDefaultCampaignNotification(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient client) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        List<GdiBaseCampaign> clientGdiCampaigns = campaignsInfoService.getAllBaseCampaigns(clientId);

        boolean hasWallet = StreamEx.of(clientGdiCampaigns)
                .filterBy(GdiBaseCampaign::getType, CampaignType.WALLET)
                .findAny()
                .isPresent();

        return clientDataService.getDefaultCampaignNotificationForTextCampaign(context.getOperator(),
                context.getSubjectUser(), hasWallet);
    }

    /**
     * GraphQL ручка. Отдает уведомления по умолчанию для создания кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "defaultCampaignNotifications")
    public List<GdDefaultCampaignNotification> getDefaultCampaignNotifications(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient client,
            @GraphQLArgument(name = "campaignTypes") @Nullable List<@GraphQLNonNull GdCampaignType> campaignTypes) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        List<GdiBaseCampaign> clientGdiCampaigns = campaignsInfoService.getAllBaseCampaigns(clientId);

        boolean hasWallet = StreamEx.of(clientGdiCampaigns)
                .filterBy(GdiBaseCampaign::getType, CampaignType.WALLET)
                .findAny()
                .isPresent();

        return clientDataService.getDefaultCampaignNotificationsForCampaignTypes(context.getOperator(),
                context.getSubjectUser(), hasWallet, campaignTypes);
    }

    @GraphQLQuery(name = XIVA_SECRET_SIGN_RESOLVER_NAME)
    public GdXivaSecretSign getXivaSecretSign(@GraphQLContext GdClientInfo clientInfo) {
        Long clientId = clientInfo.getId();
        Signature signature = xivaClient.getSignature(clientId.toString());
        if (signature == null) {
            return null;
        }

        return new GdXivaSecretSign()
                .withSign(signature.getSign())
                .withTs(signature.getTs());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "measurerAccounts")
    public List<GdClientMeasurerSystem> measurerAccounts(@GraphQLContext GdClient client) {
        return clientDataService.getMeasurerAccounts(client.getInfo());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "measurerAccountDetails")
    public List<GdClientMeasurerAccountDetails> measurerAccountDetails(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        return clientDataService.getMeasurerAccountDetails(context.getOperator(), client.getInfo());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "internalDontShowDomainsData")
    public GdInternalDontShowDomainsDataPayload getInternalDontShowDomainsData(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        return clientDataService.getInternalDontShowDomainsData(client.getInfo());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "adGroupTopRegions")
    public GdClientTopRegionsPayload adGroupTopRegions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdClientTopRegions input) {
        gridValidationService.validateClientSearchRequest(input.getSearchRequest());
        gridValidationService.validateLimitOffset(input.getLimitOffset());

        GdClientInfo clientInfo = clientDataService.getClientInfo(context, input.getSearchRequest());

        List<GdClientTopRegionInfo> adGroupTopRegions = clientDataService.getAdGroupTopRegions(clientInfo, input);

        return new GdClientTopRegionsPayload()
                .withRegions(adGroupTopRegions);
    }

    /**
     * GraphQL подзапрос. Получает качество аккаунта
     */
    @GraphQLQuery(name = ACCOUNT_SCORE_INFO_RESOLVER_NAME)
    public CompletableFuture<GdAccountScoreInfo> getAccountScoreInfo(@GraphQLContext GdClientInfo clientInfo) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        return clientDataService.getAccountScoreInfo(clientId);
    }

    /**
     * GraphQL подзапрос. Возвращает, принята ли оферта текущим представителем.
     * Пока что работает только под фичёй MODERATION_OFFER_ENABLED_FOR_DNA
     */
    @GraphQLNonNull
    @GraphQLQuery(name = IS_OFFER_ACCEPTED_RESOLVER_NAME)
    public Boolean isOfferAccepted(@GraphQLRootContext GridGraphQLContext context) {
        return clientDataService.isOfferAccepted(context);
    }

    /**
     * GraphQL подзапрос. Получает
     */
    @GraphQLNonNull
    @GraphQLQuery(name = USER_FEATURES_RESOLVER_NAME)
    public Set<GdUserFeature> getUserFeatures(
            @GraphQLRootContext GridGraphQLContext context) {
        User subjectUser = context.getSubjectUser();
        return clientDataService.getUserFeatures(subjectUser);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "showCpmUacPromoParams")
    public GdShowCpmUacPromoParams showCpmUacPromoParams(@GraphQLContext GdClient client) {
        return clientDataService.getShowCpmUacParams(client.getInfo());
    }

    /**
     * GraphQL подзапрос. По умолчанию создавать новую охватную кампанию в UAC или нет
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "cpmUacNewCamp")
    public Boolean cpmUacNewCamp(@GraphQLContext GdClient client) {
        return clientDataService.cpmUacNewCamp(client.getInfo());
    }

}
