package ru.yandex.direct.grid.processing.service.client

import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.communication.service.CommunicationEventService
import ru.yandex.direct.core.entity.client.Constants.CLIENT_MCC_BIG_MANAGED_CLIENTS_COUNT_LIMIT
import ru.yandex.direct.core.entity.client.Constants.CLIENT_MCC_MANAGED_CLIENTS_COUNT_LIMIT
import ru.yandex.direct.core.entity.client.Constants.CLIENT_MCC_REQUESTS_COUNT_LIMIT
import ru.yandex.direct.core.entity.client.mcc.ClientMccService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.core.entity.user.service.validation.UserDefects.userNotFound
import ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles
import ru.yandex.direct.core.entity.user.utils.UserUtil.isChiefRep
import ru.yandex.direct.core.entity.user.utils.UserUtil.isClient
import ru.yandex.direct.core.security.AccessDeniedException
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.client.GdClientMccRequestsAccess
import ru.yandex.direct.grid.processing.model.client.GdClientMccRequestsPayload
import ru.yandex.direct.grid.processing.model.client.GdMccClients
import ru.yandex.direct.grid.processing.model.client.GdMccClientsAccess
import ru.yandex.direct.grid.processing.model.client_mcc.mutation.GdAddClientMccRequest
import ru.yandex.direct.grid.processing.model.client_mcc.mutation.GdClientMccPayload
import ru.yandex.direct.grid.processing.service.client.converter.toGdMccClient
import ru.yandex.direct.grid.processing.service.client.converter.toGdMccRequestShort
import ru.yandex.direct.grid.processing.service.client.validation.clientAlreadyLinked
import ru.yandex.direct.grid.processing.service.client.validation.clientNotAllowed
import ru.yandex.direct.grid.processing.service.client.validation.managedClientsLimitExceeded
import ru.yandex.direct.grid.processing.service.client.validation.mccLinkNotFound
import ru.yandex.direct.grid.processing.service.client.validation.mccRequestsLimitExceeded
import ru.yandex.direct.grid.processing.service.client.validation.requestNotFound
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService
import ru.yandex.direct.rbac.RbacClientsRelations
import ru.yandex.direct.rbac.RbacRole
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.utils.PassportUtils.normalizeLogin
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.PathNode
import ru.yandex.direct.validation.result.ValidationResult

/**
 * Сервис для работы с клиентским МСС
 */
@GridGraphQLService
class ClientMccGraphQlService(
    private val clientMccService: ClientMccService,
    private val rbacClientsRelations: RbacClientsRelations,
    private val userService: UserService,
    private val rbacService: RbacService,
    private val validationResultConversionService: GridValidationResultConversionService,
    private val featureService: FeatureService,
    private val communicationEventService: CommunicationEventService,
) {
    private val LOGIN_FIELD_NAME = "login"

    @PreAuthorizeRead
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addClientMccRequest")
    fun addClientMccRequest(
        @GraphQLRootContext context: GridGraphQLContext, @GraphQLArgument(name = "input") input: GdAddClientMccRequest
    ): GdClientMccPayload {
        checkOperatorForMutation(context)
        val fieldName = GdAddClientMccRequest.LOGIN.name()

        val managedClientChiefUser = getQueriedUser(context)
        val managedClientId = managedClientChiefUser.clientId
        if (!featureService.isEnabledForClientId(managedClientId, FeatureName.CLIENT_MCC)) {
            throw AccessDeniedException("Оператор не допускается к изменению данных клиента")
        }

        val managedClientRequestsCount = clientMccService.getOwnRequests(managedClientId).size
        val managedClientControlCount = clientMccService.getControlClients(managedClientId).size
        if (managedClientRequestsCount + managedClientControlCount >= CLIENT_MCC_REQUESTS_COUNT_LIMIT) {
            return payloadWithError(mccRequestsLimitExceeded(CLIENT_MCC_REQUESTS_COUNT_LIMIT))
        }

        val controlClientUser = userService.getUserByLogin(normalizeLogin(input.login))
        if (controlClientUser != null) {
            val controlClientId = controlClientUser.clientId
            val isClientMccEnabled = featureService.isEnabledForClientId(controlClientId, FeatureName.CLIENT_MCC)
            if (!isClientMccEnabled || controlClientId == managedClientId || !isClient(controlClientUser) || controlClientUser.agencyClientId != null) {
                return payloadWithErrorForField(fieldName, input.login, clientNotAllowed())
            }
            if (!isChiefRep(controlClientUser)) {
                return payloadWithErrorForField(fieldName, input.login, clientNotAllowed())
            }
            val controlClientManagedClientsCount = rbacClientsRelations.getMangedMccClientIds(controlClientId).size
            val controlClientControlRequestsCount = clientMccService.getControlRequests(controlClientId).size
            val managedClientsLimit =
                if (featureService.isEnabledForClientId(controlClientId, FeatureName.CLIENT_MCC_BIG)) {
                    CLIENT_MCC_BIG_MANAGED_CLIENTS_COUNT_LIMIT
                } else {
                    CLIENT_MCC_MANAGED_CLIENTS_COUNT_LIMIT
                }
            if (controlClientManagedClientsCount + controlClientControlRequestsCount >= managedClientsLimit) {
                return payloadWithErrorForField(
                    fieldName, input.login, managedClientsLimitExceeded(managedClientsLimit)
                )
            }

            val relation = rbacClientsRelations.getClientRelation(managedClientId, controlClientId)
            if (relation.relationType == null) {
                clientMccService.addRequest(controlClientId, managedClientId)
                communicationEventService.processOnClientMccRequest(controlClientId, managedClientId)
            } else {
                return payloadWithErrorForField(fieldName, input.login, clientAlreadyLinked())
            }
        } else {
            return payloadWithErrorForField(fieldName, input.login, userNotFound())
        }
        return GdClientMccPayload()
    }

    private fun payloadWithError(error: Defect<*>): GdClientMccPayload {
        return GdClientMccPayload().withValidationResult(
            validationResultConversionService.buildGridValidationResult(ValidationResult.failed(null, error))
        )
    }

    private fun payloadWithErrorForField(field: String, value: String, error: Defect<*>): GdClientMccPayload {
        val vr = ValidationResult<GdAddClientMccRequest, Defect<*>>(null, Defect::class.java).addSubResult(
            PathNode.Field(field), ValidationResult.failed(value, error)
        )
        return GdClientMccPayload().withValidationResult(
            validationResultConversionService.buildGridValidationResult(vr)
        )
    }

    @PreAuthorizeRead
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "deleteClientMccRequest")
    fun deleteClientMccRequest(
        @GraphQLRootContext context: GridGraphQLContext, @GraphQLArgument(name = "login") login: String
    ): GdClientMccPayload {
        checkOperatorForMutation(context)
        val managedClientChiefUser = getQueriedUser(context)
        val managedClientId = managedClientChiefUser.clientId
        val controlClientId = userService.getClientIdByLogin(login)
        var isDeleted = false
        if (controlClientId != null) {
            isDeleted = clientMccService.deleteRequest(ClientId.fromLong(controlClientId), managedClientId)
        }
        if (!isDeleted) {
            return payloadWithErrorForField(LOGIN_FIELD_NAME, login, requestNotFound())
        }
        return GdClientMccPayload()
    }

    fun isOperatorSubjectUser(context: GridGraphQLContext): Boolean {
        return context.operator.clientId == context.subjectUser!!.clientId
    }

    fun checkOperatorForMutation(context: GridGraphQLContext) {
        val operator = context.operator

        if (hasOneOfRoles(context.operator, RbacRole.SUPER, RbacRole.SUPPORT, RbacRole.MANAGER) &&
            rbacService.canWrite(operator.uid, context.subjectUser!!.uid)
        ) {
            return
        }

        if (isClient(context.operator) && isOperatorSubjectUser(context) && isChiefRep(operator)) {
            return
        }

        throw AccessDeniedException("Оператор не допускается к изменению данных клиента")
    }

    fun checkOperatorAllowMccForMutation(context: GridGraphQLContext) {
        val operator = context.operator

        if (hasOneOfRoles(context.operator, RbacRole.SUPER, RbacRole.SUPPORT, RbacRole.MANAGER) &&
            rbacService.canWrite(operator.uid, context.subjectUser!!.uid)
        ) {
            return
        }

        if (isClient(context.operator) && isChiefRep(operator)) {
            return
        }

        throw AccessDeniedException("Оператор не допускается к изменению данных клиента")
    }

    private fun checkOperatorForQuery(context: GridGraphQLContext) {
        val operator = context.operator
        if (!isClient(operator) || isOperatorSubjectUser(context)) {
            return
        }

        throw AccessDeniedException("Операция недоступна")
    }

    @PreAuthorizeRead
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "approveClientMccRequest")
    fun approveClientMccRequest(
        @GraphQLRootContext context: GridGraphQLContext, @GraphQLArgument(name = "login") login: String
    ): GdClientMccPayload {
        checkOperatorAllowMccForMutation(context)
        val controlClientChiefUser = getQueriedUser(context)
        val controlClientId = controlClientChiefUser.clientId
        val managedClientId = userService.getClientIdByLogin(login)
        var isApproved = false
        if (managedClientId != null) {
            isApproved = clientMccService.approveRequest(controlClientId, ClientId.fromLong(managedClientId))
        }

        if (!isApproved) {
            return payloadWithErrorForField(LOGIN_FIELD_NAME, login, requestNotFound())
        }
        return GdClientMccPayload()
    }

    @PreAuthorizeRead
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "declineClientMccRequest")
    fun declineClientMccRequest(
        @GraphQLRootContext context: GridGraphQLContext, @GraphQLArgument(name = "login") login: String
    ): GdClientMccPayload {
        checkOperatorAllowMccForMutation(context)
        val controlClientChiefUser = getQueriedUser(context)
        val controlClientId = controlClientChiefUser.clientId
        val managedClientId = userService.getClientIdByLogin(login)
        var isDeclined = false
        if (managedClientId != null) {
            isDeclined = clientMccService.declineRequest(controlClientId, ClientId.fromLong(managedClientId))
        }

        if (!isDeclined) {
            return payloadWithErrorForField(LOGIN_FIELD_NAME, login, requestNotFound())
        }
        return GdClientMccPayload()
    }

    @PreAuthorizeRead
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "unlinkClientMccManagedClient")
    fun unlinkClientMccManagedClient(
        @GraphQLRootContext context: GridGraphQLContext, @GraphQLArgument(name = "login") login: String
    ): GdClientMccPayload {
        checkOperatorAllowMccForMutation(context)
        val controlClientChiefUser = getQueriedUser(context)
        val controlClientId = controlClientChiefUser.clientId
        val managedClientId = userService.getClientIdByLogin(login)
        var isUnlinked = false
        if (managedClientId != null) {
            isUnlinked = clientMccService.unlinkMcc(controlClientId, ClientId.fromLong(managedClientId))
        }

        if (!isUnlinked) {
            return payloadWithErrorForField(LOGIN_FIELD_NAME, login, mccLinkNotFound())
        }
        return GdClientMccPayload()
    }

    @PreAuthorizeRead
    @GraphQLQuery(name = "getMccClients")
    fun getMccClients(
        @GraphQLRootContext context: GridGraphQLContext
    ): GdMccClients {
        checkOperatorForQuery(context)
        val queriedUser = getQueriedUser(context)
        val ownRequests = clientMccService.getOwnRequests(queriedUser.clientId)
        var managedClientIds = clientMccService.getManagedClients(queriedUser.clientId)
        if (!managedClientIds.isEmpty()) {
            managedClientIds = managedClientIds.plus(queriedUser.clientId)
        }
        val controlClientIds = clientMccService.getControlClients(queriedUser.clientId)
        val clientIdsToFetch = controlClientIds.plus(ownRequests.map { it.controlClientId }.toSet())
            .plus(managedClientIds)
        val chiefUsers = userService.getChiefUserByClientIdMap(clientIdsToFetch)
        val canModifyAndDeleteRequests = canOperatorEditQueriedUser(context.operator, queriedUser)
        return GdMccClients()
            .withControlClients(
                controlClientIds
                    .map { toGdMccClient(chiefUsers[it]!!) }
                    .toSortedSet(compareBy({ it.info.chiefUser.name }, { it.info.chiefUser.login }))
            )
            .withManagedClients(
                managedClientIds
                    .map { toGdMccClient(chiefUsers[it]!!) }
                    .toSortedSet(compareBy({ it.info.chiefUser.name }, { it.info.chiefUser.login }))
            )
            .withRequests(
                ownRequests
                    .map { toGdMccRequestShort(chiefUsers[it.controlClientId], null) }
                    .toSortedSet(compareBy({ it.controlClient.chiefUser.name }, { it.controlClient.chiefUser.login }))
            ).withAccess(
                GdMccClientsAccess()
                    .withCanModify(canModifyAndDeleteRequests)
                    .withCanDeleteRequest(canModifyAndDeleteRequests)
            )
    }

    private fun getQueriedUser(context: GridGraphQLContext): User {
        if (isClient(context.operator)) {
            return context.operator
        } else {
            if (!isClient(context.subjectUser)) {
                throw AccessDeniedException("Операция недоступна")
            }
            return context.subjectUser!!
        }
    }

    @PreAuthorizeRead
    @GraphQLQuery(name = "getClientMccControlRequests")
    fun getClientMccControlRequests(
        @GraphQLRootContext context: GridGraphQLContext
    ): GdClientMccRequestsPayload {
        val queriedUser = getQueriedUser(context)
        val requests = clientMccService.getControlRequests(queriedUser.clientId)
        val managedClientIds = requests.map { it.managedClientId }.toSet()
        val managedClientsChiefUsers = userService.getChiefUserByClientIdMap(managedClientIds)
        val canApproveAndDecline = canOperatorEditQueriedUser(context.operator, queriedUser)
        return GdClientMccRequestsPayload()
            .withRequests(
                requests
                    .map { toGdMccRequestShort(null, managedClientsChiefUsers[it.managedClientId]) }
                    .toSortedSet(compareBy({ it.managedClient.chiefUser.name }, { it.managedClient.chiefUser.login }))
            ).withAccess(
                GdClientMccRequestsAccess()
                    .withCanApprove(canApproveAndDecline)
                    .withCanDecline(canApproveAndDecline)
            )
    }

    private fun canOperatorEditQueriedUser(operator: User, queriedUser: User): Boolean {
        if (isClient(operator)) {
            return isChiefRep(operator)
        } else {
            return rbacService.canWrite(operator.uid, queriedUser.uid)
        }
    }
}
