package ru.yandex.direct.grid.processing.service.client;

import javax.annotation.ParametersAreNonnullByDefault;

import com.github.shyiko.mysql.binlog.GtidSet;
import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.grid.core.entity.sync.model.GdiClientMutationState;
import ru.yandex.direct.grid.core.entity.sync.service.MysqlStateService;
import ru.yandex.direct.grid.core.entity.sync.service.YtSyncStateService;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.model.client.GdCheckClientMutationState;
import ru.yandex.direct.grid.processing.model.client.GdCheckClientMutationStatePayload;
import ru.yandex.direct.grid.processing.model.client.GdClientMutationState;
import ru.yandex.direct.grid.processing.model.client.GdGetClientMutationId;
import ru.yandex.direct.grid.processing.model.client.GdGetClientMutationIdPayload;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.utils.PassportUtils;

/**
 * Сервис, возвращающий данные о мутациях клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ClientMutationIdGraphQlService {

    private final GridValidationService gridValidationService;
    private final MysqlStateService mysqlStateService;
    private final YtSyncStateService ytSyncStateService;

    @Autowired
    public ClientMutationIdGraphQlService(
            GridValidationService gridValidationService,
            MysqlStateService mysqlStateService,
            YtSyncStateService ytSyncStateService) {
        this.gridValidationService = gridValidationService;
        this.mysqlStateService = mysqlStateService;
        this.ytSyncStateService = ytSyncStateService;
    }

    /**
     * Возвращает mutationId для клиента
     * Резолвер сделан мутацией, чтобы гарантировалось его выполнении после других мутаций.
     * GraphQl гарантирует последовательное выполнение мутаций в рамках одного запроса
     *
     * @param input входные параметры
     */
    @GraphQLNonNull
    @GraphQLMutation(name = "getClientMutationId")
    public GdGetClientMutationIdPayload getClientMutationId(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetClientMutationId input) {
        gridValidationService.validateGetClientMutationIdRequest(input);

        GtidSet.UUIDSet currentServerGtidSet =
                mysqlStateService.getCurrentServerGtidSet(PassportUtils.normalizeLogin(input.getLogin()));

        return new GdGetClientMutationIdPayload()
                .withMutationId(currentServerGtidSet.toString());
    }

    /**
     * Возвращает статус синхронизации YT с mysql для данных клиента,  измененных мутационной ручкой
     *
     * @param input входные параметры
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "checkClientMutationState")
    public GdCheckClientMutationStatePayload checkClientMutationState(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCheckClientMutationState input) {
        gridValidationService.validateCheckClientMutationStateRequest(input);

        GdiClientMutationState state = ytSyncStateService.checkClientMutationStateInYt(
                PassportUtils.normalizeLogin(input.getLogin()), input.getMutationId());

        return new GdCheckClientMutationStatePayload()
                .withState(GdClientMutationState.fromSource(state));
    }

}
