package ru.yandex.direct.grid.processing.service.client;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.antifraud.client.Verdict;
import ru.yandex.direct.antifraud.client.model.Action;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsRepository;
import ru.yandex.direct.core.entity.client.service.ClientMeasurerSettingsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.page.service.PageService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GdExceptions;
import ru.yandex.direct.grid.processing.exception.GridPublicException;
import ru.yandex.direct.grid.processing.model.client.GdClientMeasurerSystem;
import ru.yandex.direct.grid.processing.model.client.GdRequestChoiceFromConversionModifiersPopup;
import ru.yandex.direct.grid.processing.model.client.GdRequestMetrikaCountersAccessPayload;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdAddClientMeasurerAccount;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdChoiceFromConversionModifiersPopupPayload;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdInternalDontShowDomainsPopupAction;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdProcessInternalDontShowDomainsPopup;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdProcessInternalDontShowDomainsPopupPayload;
import ru.yandex.direct.grid.processing.service.client.validation.SaveChoiceFromConversionModifiersPopupValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus.NEED_VERIFICATION;
import static ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus.VERIFICATION_IN_PROGRESS;
import static ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus.VERIFIED;
import static ru.yandex.direct.feature.FeatureName.CONVERSION_MODIFIERS_POPUP_ENABLED;
import static ru.yandex.direct.feature.FeatureName.SHOW_INTERNAL_DONT_SHOW_DOMAINS_WARNING;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter.toClientMeasurerSystem;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;

@Service
@ParametersAreNonnullByDefault
public class ClientMutationService {
    private static final Logger logger = LoggerFactory.getLogger(ClientMutationService.class);

    private static final List<CampaignType> CAMPAIGN_WITH_PAY_FOR_CONVERSION_TYPES = List.of(CampaignType.TEXT,
            CampaignType.DYNAMIC,
            CampaignType.PERFORMANCE, CampaignType.MOBILE_CONTENT);

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final ClientOptionsRepository clientOptionsRepository;
    private final PageService pageService;
    private final ClientService clientService;
    private final ClientDataService clientDataService;
    private final FeatureService featureService;
    private final CampaignService campaignService;
    private final BidModifierService bidModifierService;
    private final FeatureManagingService featureManagingService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final ClientMeasurerSettingsService clientMeasurerSettingsService;
    private final GridValidationResultConversionService gridValidationResultConversionService;
    private final SaveChoiceFromConversionModifiersPopupValidationService saveChoiceFromConversionModifiersPopupValidationService;

    @Autowired
    public ClientMutationService(ShardHelper shardHelper,
                                 CampaignRepository campaignRepository,
                                 ClientOptionsRepository clientOptionsRepository,
                                 PageService pageService,
                                 ClientService clientService,
                                 FeatureService featureService,
                                 CampaignService campaignService,
                                 ClientDataService clientDataService,
                                 BidModifierService bidModifierService,
                                 FeatureManagingService featureManagingService,
                                 CampMetrikaCountersService campMetrikaCountersService,
                                 ClientMeasurerSettingsService clientMeasurerSettingsService,
                                 GridValidationResultConversionService gridValidationResultConversionService,
                                 SaveChoiceFromConversionModifiersPopupValidationService saveChoiceFromConversionModifiersPopupValidationService) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.clientOptionsRepository = clientOptionsRepository;
        this.pageService = pageService;
        this.clientService = clientService;
        this.clientDataService = clientDataService;
        this.featureService = featureService;
        this.campaignService = campaignService;
        this.bidModifierService = bidModifierService;
        this.featureManagingService = featureManagingService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.clientMeasurerSettingsService = clientMeasurerSettingsService;
        this.gridValidationResultConversionService = gridValidationResultConversionService;
        this.saveChoiceFromConversionModifiersPopupValidationService =
                saveChoiceFromConversionModifiersPopupValidationService;
    }

    public void fillMeasurerSettings(GdAddClientMeasurerAccount input) {
        var settings = clientMeasurerSettingsService.extendSettings(
                toClientMeasurerSystem(input.getMeasurerSystem()), input.getSettings());
        input.setSettings(settings);
    }

    public void addMeasurerAccount(User operator, ClientId clientId, GdAddClientMeasurerAccount input) {
        if (!clientDataService.canEditUserSettings(operator, clientId.asLong())) {
            throw new GridPublicException(GdExceptions.ACCESS_DENIED,
                    String.format("Operator with uid %s cannot access measurer settings of client with id %s",
                            operator.getUid(), clientId),
                    SecurityTranslations.INSTANCE.accessDenied());
        }

        var clientMeasurerSettings = new ClientMeasurerSettings()
                .withClientId(clientId.asLong())
                .withClientMeasurerSystem(toClientMeasurerSystem(input.getMeasurerSystem()))
                .withSettings(input.getSettings());

        clientMeasurerSettingsService.insertOrUpdate(clientId.asLong(), List.of(clientMeasurerSettings));
    }

    public void deleteMeasurerAccount(User operator, ClientId clientId, GdClientMeasurerSystem gdClientMeasurerSystem) {
        if (!clientDataService.canEditUserSettings(operator, clientId.asLong())) {
            throw new GridPublicException(GdExceptions.ACCESS_DENIED,
                    String.format("Operator with uid %s cannot access measurer settings of client with id %s",
                            operator.getUid(), clientId),
                    SecurityTranslations.INSTANCE.accessDenied());
        }

        clientMeasurerSettingsService.deleteByClientIdAndSystem(
                clientId.asLong(),
                toClientMeasurerSystem(gdClientMeasurerSystem));
    }

    public GdRequestMetrikaCountersAccessPayload requestMetrikaCountersAccess(String userLogin,
                                                                              Set<Long> counterIds) {
        try {
            campMetrikaCountersService.requestMetrikaCountersAccess(userLogin, counterIds);

            return new GdRequestMetrikaCountersAccessPayload()
                    .withSuccess(true)
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            return new GdRequestMetrikaCountersAccessPayload()
                    .withSuccess(false)
                    .withIsMetrikaAvailable(false);
        }
    }

    public GdProcessInternalDontShowDomainsPopupPayload processInternalDontShowDomainsPopup(GdProcessInternalDontShowDomainsPopup input,
                                                                                            GridGraphQLContext context) {
        ClientId clientId = context.getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        boolean popupFeatureEnabled = featureService.isEnabledForClientId(clientId,
                SHOW_INTERNAL_DONT_SHOW_DOMAINS_WARNING);

        logger.info("Processing internal domains popup: operator uid {}, client uid {}, action {}, is allowed {}",
                context.getOperator().getUid(), context.getSubjectUser().getUid(), input.getAction(),
                popupFeatureEnabled);

        if (!popupFeatureEnabled) {
            return new GdProcessInternalDontShowDomainsPopupPayload()
                    .withValidationResult(gridValidationResultConversionService.buildGridValidationResult(
                            ValidationResult.failed(input, inconsistentState())));
        }

        if (input.getAction() == GdInternalDontShowDomainsPopupAction.CLEAR_INTERNAL_DOMAINS) {
            Set<Long> campaignIds = campaignRepository.getNonArchivedNonDeletedCampaignIdsByClientIds(shard,
                    Set.of(clientId));

            List<AppliedChanges<Campaign>> appliedChanges = StreamEx.of(campaignService.getCampaigns(clientId,
                            campaignIds))
                    .map(this::clearInternalPageDomains)
                    .filter(AppliedChanges::hasActuallyChangedProps)
                    .toList();

            logger.info("Updated {} campaigns", appliedChanges.size());

            campaignRepository.updateCampaigns(shard, appliedChanges);
        }

        featureManagingService.disableFeatureForClient(clientId, SHOW_INTERNAL_DONT_SHOW_DOMAINS_WARNING);

        return new GdProcessInternalDontShowDomainsPopupPayload();
    }

    public GdChoiceFromConversionModifiersPopupPayload saveChoiceFromConversionModifiersPopup(
            GridGraphQLContext context,
            GdRequestChoiceFromConversionModifiersPopup requestChoiceFromConversionModifiersPopup) {
        ClientId clientId = context.getSubjectUser().getClientId();
        Client client = clientService.getClient(clientId);

        boolean isConversionMultipliersPopupEnabled = isConversionMultipliersPopupEnabled(clientId, client);
        saveChoiceFromConversionModifiersPopupValidationService.validateChoiceFromConversionModifiersPopupRequest(
                requestChoiceFromConversionModifiersPopup, isConversionMultipliersPopupEnabled);

        if (requestChoiceFromConversionModifiersPopup.getShouldCleanMultipliers()) {
            deleteClientConversionCampaignModifiers(context.getOperator().getUid(), clientId);
        }
        turnOfShowingConversionModifiersPopup(client);
        return new GdChoiceFromConversionModifiersPopupPayload();
    }

    public void updateClientPhoneVerificationStatus(Verdict verdict,
                                                    PhoneVerificationStatus curVerificationStatus,
                                                    ClientId clientId) {
        if (verdict.getStatus().equals(Action.DENY)) {
            return;
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        String challenge = verdict.getChallenge();

        if (challenge != null && curVerificationStatus == NEED_VERIFICATION) {
            clientOptionsRepository.updatePhoneVerificationStatus(shard, clientId, VERIFICATION_IN_PROGRESS);
        } else if (challenge == null) {
            clientOptionsRepository.updatePhoneVerificationStatus(shard, clientId, VERIFIED);
        }
    }

    @VisibleForTesting
    public void deleteClientConversionCampaignModifiers(Long operatorUid, ClientId clientId) {
        var shard = shardHelper.getShardByClientId(clientId);

        var campaignIdsToCleanBidModifiers = clientDataService.getNotArchivedConversionCampaignIds(shard, clientId);

        List<MassResult<Long>> resultsWithErrors = StreamEx.ofSubLists(campaignIdsToCleanBidModifiers, 100)
                .map(campaignIds -> bidModifierService.deleteCampaignModifiers(clientId, operatorUid, campaignIds))
                .remove(result -> result.getValidationResult().flattenErrors().isEmpty())
                .toList();

        logger.info("For clientId: " + clientId +
                "has errors on campaign ids: " + StreamEx.of(resultsWithErrors).joining(",") +
                "while remove bid modifiers");
    }

    @VisibleForTesting
    public void turnOfShowingConversionModifiersPopup(Client client) {
        AppliedChanges<Client> appliedChanges =
                ModelChanges.build(client, Client.IS_CONVERSION_MULTIPLIERS_POPUP_DISABLED, true)
                        .applyTo(client);
        clientService.update(appliedChanges);
    }

    private AppliedChanges<Campaign> clearInternalPageDomains(Campaign campaign) {
        var mc = new ModelChanges<>(campaign.getId(), Campaign.class);

        Set<String> disabledDomains = nvl(campaign.getDisabledDomains(), emptySet());
        Set<String> nonInternalDisabledDomains = filterToSet(disabledDomains,
                disabledDomain -> !pageService.isInternalPageDomain(disabledDomain));

        List<String> disabledVideoDomains = nvl(campaign.getDisabledVideoPlacements(), emptyList());
        List<String> nonInternalDisabledVideoDomains = filterList(disabledVideoDomains,
                disabledVideoDomain -> !pageService.isInternalPageDomain(disabledVideoDomain));

        if (disabledDomains.size() != nonInternalDisabledDomains.size()
                || disabledVideoDomains.size() != nonInternalDisabledVideoDomains.size()) {

            if (disabledDomains.size() != nonInternalDisabledDomains.size()) {
                mc.process(nonInternalDisabledDomains, Campaign.DISABLED_DOMAINS);
            }
            if (disabledVideoDomains.size() != nonInternalDisabledVideoDomains.size()) {
                mc.process(nonInternalDisabledVideoDomains, Campaign.DISABLED_VIDEO_PLACEMENTS);
            }

            mc.process(StatusBsSynced.NO, Campaign.STATUS_BS_SYNCED);
            mc.process(LocalDateTime.now(), Campaign.LAST_CHANGE);
        }

        return mc.applyTo(campaign);
    }

    private boolean isConversionMultipliersPopupEnabled(ClientId clientId, Client client) {
        boolean isFeatureForConversionMultipliersPopupEnabled = featureService.isEnabledForClientId(clientId,
                CONVERSION_MODIFIERS_POPUP_ENABLED);
        return isFeatureForConversionMultipliersPopupEnabled && !client.getIsConversionMultipliersPopupDisabled();
    }

}
