package ru.yandex.direct.grid.processing.service.client

import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles
import ru.yandex.direct.core.entity.user.utils.UserUtil.isClient
import ru.yandex.direct.core.security.AccessDeniedException
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.service.integration.balance.BalanceService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.client.GdClientRepresentative
import ru.yandex.direct.grid.processing.model.client.GdClientRepresentatives
import ru.yandex.direct.grid.processing.model.client.GdClientRepresentativesAccess
import ru.yandex.direct.grid.processing.model.client.GdDeletedClientRepresentative
import ru.yandex.direct.grid.processing.model.client.GdDeletedClientRepresentativeAccess
import ru.yandex.direct.grid.processing.service.client.converter.ClientDataConverter
import ru.yandex.direct.grid.processing.service.operator.UserDataConverter
import ru.yandex.direct.rbac.PpcRbac
import ru.yandex.direct.rbac.RbacRepType
import ru.yandex.direct.rbac.RbacRole

/**
 * Сервис для работы с представителями клиента и клиентским МСС
 */

@GridGraphQLService
class ClientRepresentativesGraphQlService(
    private val userService: UserService,
    private val clientService: ClientService,
    private val ppcRbac: PpcRbac,
    private val balanceService: BalanceService,
) {
    @PreAuthorizeRead
    @GraphQLQuery(name = "getClientRepresentatives")
    fun getClientRepresentatives(
        @GraphQLRootContext context: GridGraphQLContext
    ): GdClientRepresentatives {
        val clientId = context.subjectUser!!.clientId
        if (!hasOneOfRoles(
                context.operator,
                RbacRole.SUPER,
                RbacRole.SUPPORT,
                RbacRole.MANAGER,
                RbacRole.LIMITED_SUPPORT,
                RbacRole.SUPERREADER,
                RbacRole.CLIENT
            ) ||
            (isClient(context.operator) && !isOperatorSubjectUserAndChief(context.operator, clientId))
        ) {
            throw AccessDeniedException("Операция недоступна")
        }
        val clientChiefUid = context.subjectUser!!.chiefUid
        val canModifyRepList = canOperatorModifyClientRepresentativeList(context.operator, clientId)
        val clientUids = userService.massGetUidsByClientIds(listOf(clientId))[clientId]
        val usersInfo = userService.massGetUser(clientUids!!)
        val deletedReps = clientService.getClientDeletedReps(clientId)
        val deletedRepUids = deletedReps.map { it.uid }.toSet()
        val deletedRepsLinkedClients = deletedRepUids.associateWith { balanceService.findClientIdByUid(it).orElse(null) }
        val deletedRepsRoles = ppcRbac.getUidsRoles(deletedRepUids)
        return GdClientRepresentatives()
            .withChief(
                usersInfo
                    .filter { it.uid == clientChiefUid }
                    .map { GdClientRepresentative().withInfo(UserDataConverter.toGdUserInfo(it)) }
                    .first()
            )
            .withActive(
                usersInfo
                    .filter { it.uid != clientChiefUid }
                    .map { GdClientRepresentative().withInfo(UserDataConverter.toGdUserInfo(it)) }
                    .toSortedSet(compareBy({it.info.name}, {it.info.login}))
            )
            .withDeleted(
                deletedReps
                    .map {
                        GdDeletedClientRepresentative()
                            .withInfo(
                                ClientDataConverter.toGdDeletedClientRepresentativeInfo(it)
                            )
                            .withAccess(
                                GdDeletedClientRepresentativeAccess()
                                    .withCanRestore(canModifyRepList &&
                                        deletedRepsRoles[it.uid] == RbacRole.EMPTY &&
                                        deletedRepsLinkedClients[it.uid] == null)
                            )
                    }
                    .toSortedSet(compareBy({it.info.name}, {it.info.login}))
            )
            .withAccess(
                GdClientRepresentativesAccess()
                    .withCanAddRepresentative(canModifyRepList)
                    .withCanChangeChiefRepresentative(canModifyRepList)
            )
    }

    private fun isOperatorSubjectUserAndChief(operator: User, clientId: ClientId): Boolean {
        return operator.repType == RbacRepType.CHIEF && clientId == operator.clientId
    }

    private fun canOperatorModifyClientRepresentativeList(operator: User, clientId: ClientId): Boolean {
        if (isClient(operator) && isOperatorSubjectUserAndChief(operator, clientId)) {
            return true
        }
        if (hasOneOfRoles(operator, RbacRole.SUPER, RbacRole.SUPPORT, RbacRole.MANAGER)) {
            return true
        }

        return false
    }
}
