package ru.yandex.direct.grid.processing.service.client.converter;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.account.score.model.AccountScore;
import ru.yandex.direct.core.entity.account.score.model.AccountScoreFactors;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.model.CalloutsStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem;
import ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterPermission;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.user.model.AgencyLimRep;
import ru.yandex.direct.core.entity.user.model.DeletedClientRepresentative;
import ru.yandex.direct.grid.core.entity.model.client.GdiClientInfo;
import ru.yandex.direct.grid.processing.container.agency.GdAgencyInfo;
import ru.yandex.direct.grid.processing.model.client.GdAccountScoreInfo;
import ru.yandex.direct.grid.processing.model.client.GdAgencyLimRepInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientMeasurerSystem;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounter;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounterForSuggest;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounterTruncated;
import ru.yandex.direct.grid.processing.model.client.GdDeletedClientRepresentativeInfo;
import ru.yandex.direct.grid.processing.model.client.GdGeotreeType;
import ru.yandex.direct.grid.processing.model.client.GdMetrikaCounterSource;
import ru.yandex.direct.grid.processing.model.client.GdPhoneVerificationStatus;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.model.client.GdVerdicts;
import ru.yandex.direct.grid.processing.model.cliententity.GdCallout;
import ru.yandex.direct.grid.processing.model.cliententity.GdCalloutStatusModerate;
import ru.yandex.direct.grid.processing.model.cliententity.GdClientMeasurerAccountDetails;
import ru.yandex.direct.grid.processing.model.cliententity.GdMediascopeClientMeasurerAccountDetails;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolanding;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdClientMeasurerSettingsMediascope;
import ru.yandex.direct.grid.processing.service.client.model.MetrikaCounterForSuggest;
import ru.yandex.direct.regions.GeoTreeType;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService.COUNTER_PERMISSIONS_TREATED_AS_EDITABLE;
import static ru.yandex.direct.grid.processing.service.operator.UserDataConverter.getAvatarUrl;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.JsonUtils.fromJson;


@ParametersAreNonnullByDefault
public class ClientDataConverter {

    private final static Integer STOP_DAYS_STAT_LENGTH = 30;
    public static final String EMPTY_NAME = "";
    public static final String EMPTY_DOMAIN = "";

    @Nullable
    public static GdAgencyInfo toGdAgencyInfo(@Nullable Client client, @Nullable GdUserInfo gdAgencyUserInfo,
                                              @Nullable GdUserInfo gdAgencyRepresentativeUserInfo,
                                              @Nullable Collection<GdAgencyLimRepInfo> agencyLimRepsInfo,
                                              boolean showAgencyContacts) {
        if (client == null || gdAgencyUserInfo == null) {
            return null;
        }

        GdAgencyInfo gdAgencyInfo = new GdAgencyInfo()
                .withLogin(gdAgencyUserInfo.getLogin())
                .withRepresentativeLogin(gdAgencyRepresentativeUserInfo == null ? gdAgencyUserInfo.getLogin() :
                        gdAgencyRepresentativeUserInfo.getLogin())
                .withRepresentativeName(gdAgencyRepresentativeUserInfo == null ? gdAgencyUserInfo.getName() :
                        gdAgencyRepresentativeUserInfo.getName())
                .withShowAgencyContacts(showAgencyContacts)
                //существуют агентства без имени, чтобы фронту не пришлось вносить правки добавляем дефолтное
                // пустое значение
                .withName(nvl(client.getName(), ""));

        if (showAgencyContacts) {
            gdAgencyInfo.withUserId(gdAgencyUserInfo.getUserId())
                    .withClientId(gdAgencyUserInfo.getClientId())
                    .withEmail(gdAgencyUserInfo.getEmail())
                    .withPhone(gdAgencyUserInfo.getPhone())
                    .withAvatarUrl(nvl(gdAgencyUserInfo.getAvatarUrl(), getAvatarUrl(gdAgencyUserInfo.getUserId())));
        }

        if (agencyLimRepsInfo != null) {
            gdAgencyInfo.withLimitedRepresentatives(List.copyOf(agencyLimRepsInfo));
        } else {
            gdAgencyInfo.withLimitedRepresentatives(List.of());
        }

        return gdAgencyInfo;
    }

    public static GdAgencyLimRepInfo toGdAgencyLimRepInfo(AgencyLimRep agencyLimRep, GdUserInfo gdAgencyUserInfo,
                                                          boolean isShowContacts) {
        var info = new GdAgencyLimRepInfo()
                .withLimRepType(agencyLimRep.getRepType())
                .withLogin(gdAgencyUserInfo.getLogin())
                .withName(nvl(gdAgencyUserInfo.getName(), ""))
                .withShowContacts(isShowContacts);
        if (isShowContacts) {
            info.withUserId(gdAgencyUserInfo.getUserId())
                    .withEmail(gdAgencyUserInfo.getEmail())
                    .withPhone(gdAgencyUserInfo.getPhone());
        }
        return info;
    }

    //TODO: перенести в ClientEntityConverter
    @Nullable
    public static GdCallout toGdCallout(@Nullable Callout callout) {
        return ifNotNull(callout, data -> new GdCallout()
                .withId(data.getId())
                .withText(data.getText())
                .withClientId(data.getClientId())
                .withStatusModerate(toGdCalloutStatusModerate(data.getStatusModerate()))
                .withFlags(parseFlags(data.getFlags()))
        );
    }

    @Nullable
    public static Set<String> parseFlags(@Nullable String flags) {
        return ifNotNull(flags, value -> StreamEx.split(value, ",")
                .toSet());
    }

    private static GdCalloutStatusModerate toGdCalloutStatusModerate(CalloutsStatusModerate calloutsStatusModerate) {
        return GdCalloutStatusModerate.valueOf(calloutsStatusModerate.name());
    }

    @Nullable
    public static GdTurbolanding toGdTurbolanding(@Nullable TurboLanding turboLanding) {
        return ifNotNull(turboLanding, data -> new GdTurbolanding()
                .withId(data.getId())
                .withClientId(data.getClientId())
                .withName(data.getName())
                .withHref(data.getUrl())
                .withTurboSiteHref(data.getTurboSiteHref())
                .withPreviewHref(data.getPreviewHref())
        );
    }

    /**
     * Временный конвертер во внутреннюю модель клиента.
     * Удалить после полноценного перехода на внутреннюю модель клиента (https://st.yandex-team.ru/DIRECT-88384)
     *
     * @param clientInfo - внешняя модель GdClientInfo
     * @return GdiClientInfo сурогатная внутренняя модель.
     */
    public static GdiClientInfo toGdiClientInfo(GdClientInfo clientInfo) {
        return new GdiClientInfo()
                .withId(clientInfo.getId())
                .withShard(clientInfo.getShard())
                .withAgencyClientId(clientInfo.getAgencyClientId())
                .withAgencyUserId(clientInfo.getAgencyUserId())
                .withChiefUserId(clientInfo.getChiefUserId())
                .withCountryRegionId(clientInfo.getCountryRegionId())
                .withManagerUserId(clientInfo.getManagerUserId())
                .withWorkCurrency(clientInfo.getWorkCurrency())
                .withNonResident(clientInfo.getNonResident());
    }

    public static GdPhoneVerificationStatus toGdPhoneVerificationStatus(PhoneVerificationStatus phoneVerificationStatus) {
        return GdPhoneVerificationStatus.valueOf(phoneVerificationStatus.name());
    }

    public static GdGeotreeType toGdGeotreeType(GeoTreeType coreGeoTreeType) {
        return GdGeotreeType.valueOf(coreGeoTreeType.name());
    }

    public static GdClientMeasurerSystem toGdClientMeasurerSystem(ClientMeasurerSystem clientMeasurerSystem) {
        return GdClientMeasurerSystem.valueOf(clientMeasurerSystem.name());
    }

    public static ClientMeasurerSystem toClientMeasurerSystem(GdClientMeasurerSystem gdClientMeasurerSystem) {
        return ClientMeasurerSystem.valueOf(gdClientMeasurerSystem.name());
    }

    public static GdClientMeasurerAccountDetails toGdClientMeasurerAccountDetails(
            ClientMeasurerSettings clientMeasurerSettings
    ) {
        if (ClientMeasurerSystem.MEDIASCOPE.equals(clientMeasurerSettings.getClientMeasurerSystem())) {
            var settings = fromJson(clientMeasurerSettings.getSettings(), GdClientMeasurerSettingsMediascope.class);
            var active = settings.getExpiresAt() > Instant.now().getEpochSecond();

            return new GdMediascopeClientMeasurerAccountDetails()
                    .withMeasurerSystem(GdClientMeasurerSystem.MEDIASCOPE)
                    .withUsername(nvl(settings.getUsername(), ""))
                    .withActive(active);
        }

        return null;
    }

    @Nullable
    public static GdAccountScoreInfo toGdAccountScoreInfo(@Nullable List<AccountScore> accountScores) {
        if (CollectionUtils.isEmpty(accountScores)) {
            return null;
        }
        AccountScore accountScore = accountScores.get(0);
        BigDecimal prevScore = accountScores.size() > 1 ? accountScores.get(1).getScore() :
                BigDecimal.ZERO;
        Set<GdVerdicts> verdicts = getVerdicts(accountScore.getFactors());

        return new GdAccountScoreInfo()
                .withScore(accountScore.getScore())
                .withPrevScore(prevScore)
                .withProgress(accountScore.getScore().compareTo(prevScore))
                .withVerdicts(verdicts);
    }

    private static Set<GdVerdicts> getVerdicts(@Nullable AccountScoreFactors factors) {
        if (factors == null) {
            return emptySet();
        }
        var res = new HashSet<GdVerdicts>();
        if (factors.getAvgBannersCount() != null && factors.getAvgBannersCount() <= 2) {
            res.add(GdVerdicts.AVG_BANNERS);
        }
        if (factors.getStopDaysFine() != null) {
            if (factors.getStopDaysFine() > 0) {
                res.add(GdVerdicts.STOP_DAY);
            }
        } else if (factors.getShowsDaysCount() != null) {
            if (STOP_DAYS_STAT_LENGTH - factors.getShowsDaysCount() <= 21 && factors.getShowsDaysCount() > 0) {
                res.add(GdVerdicts.STOP_DAY);
            }
        }
        if (factors.getAvgPhrasesCount() != null && factors.getAvgPhrasesCount() > 10) {
            res.add(GdVerdicts.AVG_PHRASES);
        }
        if (factors.getVcardsCount() == null || factors.getVcardsCount() == 0) {
            res.add(GdVerdicts.VCARDS);
        }
        if (factors.getBroadmatchPercent() != null && factors.getBroadmatchPercent() < 20) {
            res.add(GdVerdicts.BROADMATCH);
        }
        if (factors.getSitelinksPercent() != null && factors.getSitelinksPercent() < 20) {
            res.add(GdVerdicts.SITELINKS);
        }
        if (factors.getImagesPercent() != null && factors.getImagesPercent() < 20) {
            res.add(GdVerdicts.IMAGES);
        }
        if (factors.getCtxPriceCoefPercent() != null && factors.getCtxPriceCoefPercent() > 0
                && factors.getCtxPriceCoefPercent() < 16) {
            res.add(GdVerdicts.CTX_PRICE_COEF);
        }
        if (factors.getSeparatePlaceCount() == null || factors.getSeparatePlaceCount() == 0) {
            res.add(GdVerdicts.SEPARATE_PLACE);
        }
        return res;
    }

    public static GdClientMetrikaCounterForSuggest toGdMetrikaCounterForSuggest(MetrikaCounterForSuggest counter) {
        return new GdClientMetrikaCounterForSuggest()
                .withId(counter.getId())
                .withIsEditableByOperator(counter.getEditableByOperator())
                .withIsCalltrackingOnSiteCompatible(counter.getCalltrackingOnSiteCompatible())
                .withIsAccessible(counter.getAccessible())
                .withIsAccessRequested(counter.getAccessRequested());
    }

    public static Set<GdClientMetrikaCounter> toGdClientMetrikaCountersWithAdditionalInformation(
            Collection<MetrikaCounterWithAdditionalInformation> counters) {
        return listToSet(counters, counter -> toGdClientMetrikaCounterWithAdditionalInformation(counter, emptyMap()));
    }

    public static Set<GdClientMetrikaCounter> toGdClientMetrikaCountersWithAdditionalInformation(
            Collection<MetrikaCounterWithAdditionalInformation> countersAvailableToClient,
            Map<Long, MetrikaCounterPermission> operatorPermissionByCounterId) {
        return listToSet(countersAvailableToClient,
                counter -> toGdClientMetrikaCounterWithAdditionalInformation(counter, operatorPermissionByCounterId));
    }

    private static GdClientMetrikaCounter toGdClientMetrikaCounterWithAdditionalInformation(
            MetrikaCounterWithAdditionalInformation counter, Map<Long, MetrikaCounterPermission> operatorPermission) {
        var counterId = counter.getId();
        return new GdClientMetrikaCounter()
                .withId(counterId)
                .withDomain(nvl(counter.getDomain(), EMPTY_DOMAIN))
                .withName(nvl(counter.getName(), EMPTY_NAME))
                .withIsEditableByOperator(isCounterPermissionEditable(operatorPermission.get(counterId)))
                .withSource(toGdMetrikaCounterSource(counter.getSource()))
                .withHasEcommerce(counter.getHasEcommerce());
    }

    public static GdClientMetrikaCounterTruncated toGdClientMetrikaCounterTruncated(
            MetrikaCounterWithAdditionalInformation counter) {
        return new GdClientMetrikaCounterTruncated()
                .withId(counter.getId())
                .withDomain(nvl(counter.getDomain(), EMPTY_DOMAIN))
                .withName(nvl(counter.getName(), EMPTY_NAME));
    }

    private static boolean isCounterPermissionEditable(@Nullable MetrikaCounterPermission counterPermission) {
        if (counterPermission == null) {
            return false;
        }
        return COUNTER_PERMISSIONS_TREATED_AS_EDITABLE.contains(counterPermission);
    }

    @Nullable
    public static GdMetrikaCounterSource toGdMetrikaCounterSource(@Nullable MetrikaCounterSource metrikaCounterSource) {
        if (metrikaCounterSource == null) {
            return null;
        }

        switch (metrikaCounterSource) {
            case SPRAV:
                return GdMetrikaCounterSource.SPRAV;
            case TURBO:
                return GdMetrikaCounterSource.TURBO_DIRECT;
            case SYSTEM:
                return GdMetrikaCounterSource.SYSTEM;
            case PARTNER:
                return GdMetrikaCounterSource.PARTNER;
            case MARKET:
                return GdMetrikaCounterSource.MARKET;
            case EDA:
                return GdMetrikaCounterSource.EDA;
            case UNKNOWN:
                return null;
            default:
                throw new IllegalStateException("No such value: " + metrikaCounterSource);
        }
    }

    public static GdDeletedClientRepresentativeInfo toGdDeletedClientRepresentativeInfo(DeletedClientRepresentative deletedClientRepresentative) {
        return new GdDeletedClientRepresentativeInfo()
                .withUserId(deletedClientRepresentative.getUid())
                .withLogin(deletedClientRepresentative.getLogin())
                .withName(nvl(deletedClientRepresentative.getFio(), ""))
                .withEmail(nvl(deletedClientRepresentative.getEmail(), ""))
                .withPhone(deletedClientRepresentative.getPhone());
    }
}
