package ru.yandex.direct.grid.processing.service.client.converter;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.creative.container.CreativeFilterContainer;
import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoAspect;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoText;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;
import ru.yandex.direct.core.entity.image.container.ImageFilterContainer;
import ru.yandex.direct.core.entity.image.model.AvatarHost;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.model.Image;
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta;
import ru.yandex.direct.core.entity.image.model.ImageSizeMeta;
import ru.yandex.direct.core.entity.image.model.ImageSmartCenter;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.cliententity.GdAdditionalData;
import ru.yandex.direct.grid.processing.model.cliententity.GdBannerStorageCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCanvasCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCpcVideoCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCpmAudioCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCpmIndoorCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCpmOutdoorCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCpmVideoCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeBusinessType;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeFilter;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeStatusModerate;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeType;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdHtml5Creative;
import ru.yandex.direct.grid.processing.model.cliententity.GdHtml5ExpandedCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdMdsHostedHtml5Creative;
import ru.yandex.direct.grid.processing.model.cliententity.GdModerationInfo;
import ru.yandex.direct.grid.processing.model.cliententity.GdModerationInfoAspect;
import ru.yandex.direct.grid.processing.model.cliententity.GdModerationInfoText;
import ru.yandex.direct.grid.processing.model.cliententity.GdOverlayCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdSmartCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdTypedCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdVideoAdditionCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdVideoFormat;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageAvatarsHost;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageNamespace;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageType;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImage;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageFilter;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageFormat;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageOrderBy;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageSize;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageSmartCenter;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageType;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContext;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCalloutsItem;
import ru.yandex.direct.grid.processing.service.client.container.CreativesCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.client.container.ImagesCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.cliententity.container.CreativesCacheFilterData;
import ru.yandex.direct.grid.processing.service.cliententity.container.ImagesCacheFilterData;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.creative.service.CreativeService.HTML_5_MEDIA_TYPES_FOR_GENERATED_CREATIVE;
import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_FORMATS_BY_IMAGE_TYPE;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.toDirectImagePath;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class ClientEntityConverter {
    public static List<Callout> extractCallouts(GdSaveCallouts input, ClientId clientId) {
        return mapList(input.getSaveItems(), item -> toCallout(item, clientId));
    }

    private static Callout toCallout(GdSaveCalloutsItem item, ClientId clientId) {
        return new Callout().withText(item.getText())
                .withClientId(clientId.asLong());
    }

    public static GdImage toGdImage(Image image) {
        return new GdImage()
                .withImageHash(image.getImageHash())
                .withAvatarsHost(toGdAdImageAvatarsHost(image.getAvatarsHost()))
                .withNamespace(toGdAdImageNamespace(image.getNamespace()))
                .withImageSize(toGdImageSize(image.getSize()))
                .withMdsGroupId(ifNotNull(image.getMdsGroupId(), Integer::longValue))
                .withFormats(toSupportedGdImageFormats(image.getImageType(), image.getMdsMetaWithUserSettings()))
                .withName(image.getName())
                .withCreateTime(image.getCreateTime())
                .withType(toGdAdImageType(image.getImageType()))
                .withImageHash(image.getImageHash());
    }

    @Nullable
    public static List<GdImageFormat> toSupportedGdImageFormats(ImageType imageType,
                                                                @Nullable ImageMdsMeta imageMdsMeta) {
        if (imageMdsMeta == null || imageMdsMeta.getSizes() == null) {
            return null;
        }

        return EntryStream.of(imageMdsMeta.getSizes())
                .filterKeys(SUPPORTED_FORMATS_BY_IMAGE_TYPE.get(imageType)::contains)
                .values()
                .map(ClientEntityConverter::toGdImageFormat)
                .toList();
    }

    static GdImageFormat toGdImageFormat(ImageSizeMeta imageSizeMeta) {
        return new GdImageFormat()
                .withPath(toDirectImagePath(imageSizeMeta.getPath()))
                .withImageSize(toGdImageSize(imageSizeMeta.getHeight(), imageSizeMeta.getWidth()))
                .withSmartCenters(toGdImageSmartCenters(imageSizeMeta.getSmartCenters()));
    }

    @Nullable
    private static List<GdImageSmartCenter> toGdImageSmartCenters(
            @Nullable Map<String, ImageSmartCenter> smartCenters) {
        if (smartCenters == null) {
            return null;
        }

        return mapList(smartCenters.entrySet(), entry -> toGdImageSmartCenter(entry.getKey(), entry.getValue()));
    }

    private static GdImageSmartCenter toGdImageSmartCenter(String ratio, ImageSmartCenter imageSmartCenter) {
        return new GdImageSmartCenter()
                .withRatio(ratio)
                .withHeight(imageSmartCenter.getHeight())
                .withWidth(imageSmartCenter.getWidth())
                .withX(imageSmartCenter.getX())
                .withY(imageSmartCenter.getY());
    }

    private static GdImageSize toGdImageSize(ImageSize imageSize) {
        return toGdImageSize(imageSize.getHeight(), imageSize.getWidth());
    }

    private static GdImageSize toGdImageSize(int height, int width) {
        return new GdImageSize()
                .withHeight(height)
                .withWidth(width);
    }

    private static GdHtml5Creative createGdHtml5Creative(Creative creative) {
        GdHtml5Creative typed;
        if (creative.getModerationInfo() != null
                && creative.getModerationInfo().getHtml() != null
                && creative.getModerationInfo().getHtml().getUrl() != null) {
            if (creative.getExpandedPreviewUrl() != null) {
                typed = new GdHtml5ExpandedCreative()
                        .withMdsUrl(creative.getModerationInfo().getHtml().getUrl())
                        .withMdsExtendedUrl(creative.getExpandedPreviewUrl());
            } else {
                typed = new GdMdsHostedHtml5Creative()
                        .withMdsUrl(creative.getModerationInfo().getHtml().getUrl());
            }
        } else {
            typed = new GdHtml5Creative();
        }
        var additionalData = creative.getAdditionalData();
        if (additionalData != null && additionalData.getOriginalWidth() != null && additionalData.getOriginalHeight() != null) {
            typed.setOriginalHeight(additionalData.getOriginalHeight());
            typed.setOriginalWidth(additionalData.getOriginalWidth());
        } else {
            typed.setOriginalHeight(creative.getHeight().intValue());
            typed.setOriginalWidth(creative.getWidth().intValue());
        }
        //Логика определения признака, что креатив явлется картинкой в перле
        //https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/Model
        // /CanvasHtml5Creative.pm#L95
        typed.withIsImage(HTML_5_MEDIA_TYPES_FOR_GENERATED_CREATIVE.contains(creative.getSourceMediaType()));
        return typed;
    }

    @Nullable
    public static GdTypedCreative toGdCreativeImplementation(@Nullable Creative creative) {
        if (creative == null) {
            return null;
        }
        GdTypedCreative typedAd;
        switch (creative.getType()) {
            case VIDEO_ADDITION_CREATIVE:
                typedAd = new GdVideoAdditionCreative()
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withDuration(creative.getDuration());
                break;
            case PERFORMANCE:
                typedAd = new GdSmartCreative()
                        .withBusinessType(GdCreativeBusinessType.fromSource(creative.getBusinessType()))
                        .withRegionIds(nvl(creative.getSumGeo(), emptyList()))
                        .withGroupName(creative.getGroupName())
                        .withThemeId(creative.getThemeId())
                        .withLayoutId(creative.getLayoutId())
                        .withGroupId(creative.getCreativeGroupId());
                break;
            case BANNERSTORAGE:
                typedAd = new GdBannerStorageCreative()
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withTemplateId(creative.getTemplateId())
                        .withDuration(creative.getDuration())
                        .withVersion(creative.getVersion());
                break;
            case CANVAS:
                typedAd = new GdCanvasCreative()
                        .withDuration(nvl(creative.getDuration(), 0L))
                        .withIsAdaptive(creative.getIsAdaptive())
                        .withBatchId(getBatchId(creative));
                break;
            case HTML5_CREATIVE:
                typedAd = createGdHtml5Creative(creative);
                break;
            case CPM_VIDEO_CREATIVE:
                typedAd = new GdCpmVideoCreative()
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withDuration(creative.getDuration());
                break;
            case CPM_AUDIO_CREATIVE:
                typedAd = new GdCpmAudioCreative()
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withDuration(creative.getDuration());
                break;
            case CPM_OVERLAY:
                typedAd = new GdOverlayCreative();
                break;
            case CPM_OUTDOOR_CREATIVE:
                typedAd = new GdCpmOutdoorCreative()
                        .withAdditionalData(toAdditionalData(creative.getAdditionalData()))
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withDuration(creative.getDuration());
                break;
            case CPM_INDOOR_CREATIVE:
                typedAd = new GdCpmIndoorCreative()
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withDuration(creative.getDuration());
                break;
            case CPC_VIDEO_CREATIVE:
                typedAd = new GdCpcVideoCreative()
                        .withLivePreviewUrl(creative.getLivePreviewUrl())
                        .withDuration(creative.getDuration());
                break;
            default:
                throw new IllegalArgumentException("Unsupported creative type: " + creative.getType());
        }

        return typedAd
                .withCreativeType(toGdCreativeType(creative.getType()))
                .withStatusModerate(GdCreativeStatusModerate.fromSource(creative.getStatusModerate()))
                .withCreativeId(creative.getId())
                .withHeight(creative.getHeight())
                .withWidth(creative.getWidth())
                .withName(creative.getName())
                .withModerationInfo(ifNotNull(creative.getModerationInfo(), ClientEntityConverter::toGdModerationInfo))
                .withPreviewUrl(creative.getPreviewUrl());
    }

    private static final Pattern OLD_CANVAS_CREATIVE_LIVE_PREVIEW_PATTERN =
            Pattern.compile("/creatives/([^/]+)/\\d+/preview");
    private static final Pattern CANVAS_CREATIVE_LIVE_PREVIEW_PATTERN =
            Pattern.compile("/creative-preview/image/([^/]+)/\\d+");

    @Nullable
    private static String getBatchId(Creative data) {
        if (data.getLivePreviewUrl() == null) {
            return null;
        }

        var matcher = CANVAS_CREATIVE_LIVE_PREVIEW_PATTERN.matcher(data.getLivePreviewUrl());
        if (matcher.find()) {
            return matcher.group(1);
        }

        var oldMatcher = OLD_CANVAS_CREATIVE_LIVE_PREVIEW_PATTERN.matcher(data.getLivePreviewUrl());
        if (oldMatcher.find()) {
            return oldMatcher.group(1);
        }

        return null;
    }

    private static GdAdditionalData toAdditionalData(@Nullable AdditionalData additionalData) {
        List<VideoFormat> videoFormats =
                Optional.ofNullable(additionalData)
                        .map(AdditionalData::getFormats)
                        .orElse(emptyList());
        List<GdVideoFormat> gdVideoFormats = filterAndMapList(videoFormats,
                vf -> vf.getHeight() != null && vf.getWidth() != null, ClientEntityConverter::toGdVideoFormat);
        return new GdAdditionalData()
                .withDuration(ifNotNull(additionalData, AdditionalData::getDuration))
                .withFormats(gdVideoFormats);
    }

    private static GdVideoFormat toGdVideoFormat(VideoFormat vf) {
        return new GdVideoFormat().withHeight(vf.getHeight()).withWidth(vf.getWidth()).withType(vf.getType()).withUrl(vf.getUrl());
    }

    private static GdModerationInfo toGdModerationInfo(ModerationInfo mi) {
        return new GdModerationInfo()
                .withTexts(ifNotNull(mi.getTexts(), texts -> mapList(texts,
                        ClientEntityConverter::toGdModerationInfoText)))
                .withAspects(ifNotNull(mi.getAspects(), aspects -> mapList(aspects,
                        ClientEntityConverter::toGdModerationInfoAspect)));
    }

    private static GdModerationInfoAspect toGdModerationInfoAspect(ModerationInfoAspect a) {
        return new GdModerationInfoAspect().withHeight(a.getHeight()).withWidth(a.getWidth());
    }

    private static GdModerationInfoText toGdModerationInfoText(ModerationInfoText text) {
        return new GdModerationInfoText().withType(text.getType()).withText(text.getText());
    }

    public static CreativeType toCreativeType(GdCreativeType type) {
        switch (type) {
            case VIDEO_ADDITION:
                return CreativeType.VIDEO_ADDITION_CREATIVE;
            case BANNERSTORAGE:
                return CreativeType.BANNERSTORAGE;
            case SMART:
                return CreativeType.PERFORMANCE;
            case CANVAS:
                return CreativeType.CANVAS;
            case HTML5_CREATIVE:
                return CreativeType.HTML5_CREATIVE;
            case CPM_VIDEO_CREATIVE:
                return CreativeType.CPM_VIDEO_CREATIVE;
            case CPM_AUDIO_CREATIVE:
                return CreativeType.CPM_AUDIO_CREATIVE;
            case CPM_OUTDOOR_CREATIVE:
                return CreativeType.CPM_OUTDOOR_CREATIVE;
            case CPM_INDOOR_CREATIVE:
                return CreativeType.CPM_INDOOR_CREATIVE;
            case CPC_VIDEO_CREATIVE:
                return CreativeType.CPC_VIDEO_CREATIVE;
            case OVERLAY_CREATIVE:
                return CreativeType.CPM_OVERLAY;
            default:
                throw new IllegalArgumentException("Unsupported creative type: " + type);
        }
    }

    public static GdCreativeType toGdCreativeType(CreativeType type) {
        switch (type) {
            case VIDEO_ADDITION_CREATIVE:
                return GdCreativeType.VIDEO_ADDITION;
            case PERFORMANCE:
                return GdCreativeType.SMART;
            case BANNERSTORAGE:
                return GdCreativeType.BANNERSTORAGE;
            case CANVAS:
                return GdCreativeType.CANVAS;
            case HTML5_CREATIVE:
                return GdCreativeType.HTML5_CREATIVE;
            case CPM_VIDEO_CREATIVE:
                return GdCreativeType.CPM_VIDEO_CREATIVE;
            case CPM_AUDIO_CREATIVE:
                return GdCreativeType.CPM_AUDIO_CREATIVE;
            case CPM_OUTDOOR_CREATIVE:
                return GdCreativeType.CPM_OUTDOOR_CREATIVE;
            case CPM_INDOOR_CREATIVE:
                return GdCreativeType.CPM_INDOOR_CREATIVE;
            case CPC_VIDEO_CREATIVE:
                return GdCreativeType.CPC_VIDEO_CREATIVE;
            case CPM_OVERLAY:
                return GdCreativeType.OVERLAY_CREATIVE;
            default:
                throw new IllegalArgumentException("Unsupported creative type: " + type);
        }
    }

    private static GdAdImageNamespace toGdAdImageNamespace(BannerImageFormatNamespace namespace) {
        return GdAdImageNamespace.valueOf(namespace.name());
    }

    private static GdAdImageAvatarsHost toGdAdImageAvatarsHost(AvatarHost avatarsHost) {
        return GdAdImageAvatarsHost.valueOf(avatarsHost.name());
    }

    private static GdAdImageType toGdAdImageType(ImageType imageType) {
        return GdAdImageType.valueOf(imageType.name());
    }

    public static ImageFilterContainer toImageFilterContainer(GdImageFilter filter) {
        return new ImageFilterContainer()
                .withBannerImageType(ifNotNull(filter.getImageType(), GdImageType::toSource))
                .withImageHashes(filter.getImageHashIn())
                .withNameContains(filter.getNameContains())
                .withMaxCreateTime(filter.getMaxCreatedDateTime())
                .withMinCreateTime(filter.getMinCreatedDateTime())
                .withWidth(filter.getWidth())
                .withHeight(filter.getHeight())
                .withSource(BannerImageSource.DIRECT);
    }

    public static CreativeFilterContainer toCreativeFilterContainer(GdCreativeFilter filter) {
        return new CreativeFilterContainer()
                .withCreativeIds(filter.getCreativeIdIn())
                .withCreativeTypes(mapSet(filter.getCreativeTypeIn(), ClientEntityConverter::toCreativeType));
    }

    public static CreativesCacheRecordInfo toCreativesCacheRecordInfo(long clientId, GdCreativesContainer input) {
        return new CreativesCacheRecordInfo(clientId, input.getCacheKey(),
                new CreativesCacheFilterData()
                        .withFilter(input.getFilter()));
    }

    public static ImagesCacheRecordInfo toImagesCacheRecordInfo(long clientId, GdImagesContainer input,
                                                                List<GdImageOrderBy> ordersBy) {
        return new ImagesCacheRecordInfo(clientId, input.getCacheKey(),
                new ImagesCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(ordersBy));
    }

    public static GdCreativesContext toGdCreativesContext(List<GdTypedCreative> rowsetFull) {
        return new GdCreativesContext()
                .withTotalCount(rowsetFull.size())
                .withCreativeIds(listToSet(rowsetFull, GdTypedCreative::getCreativeId));
    }

    public static GdImagesContext toGdImagesContext(List<GdImage> rowsetFull) {
        return new GdImagesContext()
                .withTotalCount(rowsetFull.size())
                .withImageHashes(listToSet(rowsetFull, GdImage::getImageHash));
    }

}
