package ru.yandex.direct.grid.processing.service.client.converter;

import java.util.List;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.campaign.model.CampAdditionalData;
import ru.yandex.direct.core.entity.domain.DomainUtils;
import ru.yandex.direct.grid.processing.model.client.GdClientExternalDataEntry;
import ru.yandex.direct.grid.processing.model.client.GdClientExternalDataType;
import ru.yandex.direct.grid.processing.model.client.GdClientExternalOrganizationDataEntry;
import ru.yandex.direct.grid.processing.model.client.GdClientExternalSiteDataEntry;
import ru.yandex.direct.grid.processing.model.client.GdClientMetrikaCounter;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.util.FaviconUtils;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.constraint.StringConstraints;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

public class ToGdClientExternalDataEntryConverters {
    private static final Logger logger = LoggerFactory.getLogger(ToGdClientExternalDataEntryConverters.class);

    public static final Set<ModelProperty<?, ?>> ORGANIZATION_PROPERTIES_FOR_CLIENT_BASED_SUGGEST =
            ImmutableSet.<ModelProperty<?, ?>>builder()
                    .add(GdOrganization.URLS)
                    .add(GdOrganization.PREVIEW_HREF)
                    .add(GdOrganization.RUBRIC)
                    .add(GdOrganization.COMPANY_NAME)
                    .build();

    public static GdClientExternalDataEntry fromOrganizationData(GdOrganization organization) {
        return new GdClientExternalOrganizationDataEntry()
                .withUrl(chooseOrganizationUrl(organization))
                .withPreviewHref(organization.getPreviewHref())
                .withOrganization(organization)
                .withType(GdClientExternalDataType.ORGANIZATION)
                .withBusinessCategory(organization.getRubric())
                .withBusinessName(organization.getCompanyName());
    }

    public static boolean hasAnyValidUrl(GdOrganization organization) {
        return CollectionUtils.isNotEmpty(organization.getUrls()) &&
                organization.getUrls().stream()
                        .filter(StringUtils::isNotBlank)
                        .anyMatch(StringConstraints::isValidHref);
    }

    private static String chooseOrganizationUrl(GdOrganization organization) {
        List<String> validUrls = filterList(organization.getUrls(),
                t -> StringUtils.isNotBlank(t) && StringConstraints.isValidHref(t));
        if (validUrls.isEmpty()) {
            logger.warn("organization without url somehow got into suggest");
            return null;
        }
        return validUrls.stream()
                .filter(t -> !DomainUtils.isSocialNetworkSite(t))
                .findFirst()
                .orElse(validUrls.get(0));
    }

    public static GdClientExternalDataEntry fromCampaignData(CampAdditionalData campAdditionalData) {
        return new GdClientExternalSiteDataEntry()
                .withUrl(campAdditionalData.getHref())
                .withType(GdClientExternalDataType.CAMPAIGN)
                .withBusinessCategory(campAdditionalData.getBusinessCategory())
                .withBusinessName(campAdditionalData.getCompanyName());
    }

    public static GdClientExternalDataEntry fromMetrikaData(GdClientMetrikaCounter metrikaCounter) {
        return new GdClientExternalSiteDataEntry()
                .withUrl(ifNotNull(metrikaCounter.getDomain(), t -> "https://" + t))
                .withPreviewHref(ifNotNull(metrikaCounter.getDomain(), t -> FaviconUtils.getFaviconLink(t)))
                .withMetrikaData(metrikaCounter)
                .withType(GdClientExternalDataType.SITE);
    }
}
