package ru.yandex.direct.grid.processing.service.client.converter;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.vcard.model.InstantMessenger;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.PointPrecision;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdAddress;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdCampaignVcard;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdInstantMessenger;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdInstantMessengerType;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPhone;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPointOnMap;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPointPrecision;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcard;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddAddress;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddInstantMessenger;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddPhone;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcard;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcardsPayloadItem;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Functions.identity;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class VcardDataConverter {

    //символ для обозначения корпуса в адресе
    static final char BUILDING_SYMBOL = 'к';

    private VcardDataConverter() {
    }

    @Nullable
    public static GdVcard toGdVcard(@Nullable Vcard vcard) {
        if (vcard == null) {
            return null;
        }
        return new GdVcard()
                .withId(vcard.getId())
                .withCampaignId(vcard.getCampaignId())
                .withCompanyName(vcard.getCompanyName())
                .withContactPerson(vcard.getContactPerson())
                .withEmail(vcard.getEmail())
                .withExtraMessage(vcard.getExtraMessage())
                .withOgrn(vcard.getOgrn())
                .withInstantMessenger(toGdInstantMessenger(vcard.getInstantMessenger()))
                .withWorkTimes(WorkTimeConverter.toGdWorkTimes(vcard.getWorkTime()))
                .withPhone(toGdPhone(vcard.getPhone()))
                .withAddress(toGdAddress(vcard));
    }

    @Nullable
    public static GdCampaignVcard toGdCampaignVcard(@Nullable Vcard vcard) {
        if (vcard == null) {
            return null;
        }
        return new GdCampaignVcard()
                .withId(vcard.getId())
                .withCampaignId(vcard.getCampaignId())
                .withCompanyName(vcard.getCompanyName())
                .withContactPerson(vcard.getContactPerson())
                .withEmail(vcard.getEmail())
                .withExtraMessage(vcard.getExtraMessage())
                .withOgrn(vcard.getOgrn())
                .withInstantMessenger(toGdInstantMessenger(vcard.getInstantMessenger()))
                .withWorkTimes(WorkTimeConverter.toGdWorkTimes(vcard.getWorkTime()))
                .withPhone(toGdPhone(vcard.getPhone()))
                .withAddress(toGdAddress(vcard));
    }

    private static GdAddress toGdAddress(Vcard vcard) {
        Integer geoId = ifNotNull(vcard.getGeoId(), Math::toIntExact);
        return new GdAddress()
                .withGeoId(geoId)
                .withCountry(vcard.getCountry())
                .withCity(vcard.getCity())
                .withStreet(vcard.getStreet())
                .withHouseWithBuilding(toHouseWithBuilding(vcard.getHouse(), vcard.getBuild()))
                .withApartment(vcard.getApart())
                .withMetroStationId(ifNotNull(vcard.getMetroId(), Math::toIntExact))
                .withMetroStationName(vcard.getMetroName())
                .withPointOnMap(toGdPointOnMap(vcard.getManualPoint()))
                .withPointPrecision(toGdPointPrecision(vcard.getPrecision()));
    }

    @Nullable
    static String toHouseWithBuilding(@Nullable String house, @Nullable String building) {
        if (house == null) {
            return null;
        }

        if (StringUtils.isBlank(building)) {
            return house;
        }

        //проверяем, надо ли добавлять букву к (корпус). Добавляем, если значение building начинается на цифру
        return Character.isDigit(building.charAt(0))
                ? house + BUILDING_SYMBOL + building //пример 2 + к + 2Щ = 2к2Щ
                : house + building; //пример 18 + А = 18А
    }

    @Nullable
    private static GdPointPrecision toGdPointPrecision(@Nullable PointPrecision precision) {
        return ifNotNull(precision, p -> GdPointPrecision.valueOf(p.name()));
    }

    @Nullable
    private static GdPointOnMap toGdPointOnMap(@Nullable PointOnMap manualPoint) {
        return ifNotNull(manualPoint, mP -> new GdPointOnMap()
                .withX(mP.getX())
                .withY(mP.getY())
                .withX1(mP.getX1())
                .withY1(mP.getY1())
                .withX2(mP.getX2())
                .withY2(mP.getY2()));
    }

    @Nullable
    public static PointOnMap toPointOnMap(@Nullable GdPointOnMap gdPointOnMap) {
        return ifNotNull(gdPointOnMap, mP -> new PointOnMap()
                .withX(mP.getX())
                .withY(mP.getY())
                .withX1(mP.getX1())
                .withY1(mP.getY1())
                .withX2(mP.getX2())
                .withY2(mP.getY2()));
    }

    private static GdPhone toGdPhone(Phone phone) {
        return new GdPhone()
                .withCountryCode(phone.getCountryCode())
                .withCityCode(phone.getCityCode())
                .withPhoneNumber(phone.getPhoneNumber())
                .withExtension(phone.getExtension());
    }

    public static Phone toPhone(GdAddPhone gdAddPhone) {
        return new Phone()
                .withCountryCode(gdAddPhone.getCountryCode())
                .withCityCode(gdAddPhone.getCityCode())
                .withPhoneNumber(gdAddPhone.getPhoneNumber())
                .withExtension(gdAddPhone.getExtension());
    }

    @Nullable
    private static GdInstantMessenger toGdInstantMessenger(@Nullable InstantMessenger instantMessenger) {
        return ifNotNull(instantMessenger, im -> new GdInstantMessenger()
                .withLogin(im.getLogin())
                .withType(toGdInstantMessengerType(im.getType())));
    }

    @Nullable
    private static GdInstantMessengerType toGdInstantMessengerType(@Nullable String type) {
        if (type == null) {
            return null;
        }
        return GdInstantMessengerType.valueOf(type.toUpperCase());
    }

    @Nullable
    public static InstantMessenger toInstantMessenger(@Nullable GdAddInstantMessenger gdAddInstantMessenger) {
        return ifNotNull(gdAddInstantMessenger, gd -> new InstantMessenger()
                .withLogin(gd.getLogin())
                .withType(toInstantMessengerType(gd.getType())));
    }

    private static String toInstantMessengerType(GdInstantMessengerType gdInstantMessengerType) {
        return gdInstantMessengerType.name().toLowerCase();
    }

    public static Vcard toVcard(GdAddVcard gdAddVcard) {
        GdAddAddress gdAddAddress = gdAddVcard.getAddress();

        return new Vcard()
                .withCampaignId(gdAddVcard.getCampaignId())
                .withCompanyName(gdAddVcard.getCompanyName())
                .withContactPerson(gdAddVcard.getContactPerson())
                .withEmail(gdAddVcard.getEmail())
                .withExtraMessage(gdAddVcard.getExtraMessage())
                .withOgrn(gdAddVcard.getOgrn())
                .withWorkTime(WorkTimeConverter.toWorkTime(gdAddVcard.getWorkTimes()))
                .withInstantMessenger(toInstantMessenger(gdAddVcard.getInstantMessenger()))
                .withPhone(toPhone(gdAddVcard.getPhone()))
                .withCountry(gdAddAddress.getCountry())
                .withCity(gdAddAddress.getCity())
                .withStreet(gdAddAddress.getStreet())
                .withHouse(gdAddAddress.getHouseWithBuilding())
                .withApart(gdAddAddress.getApartment())
                .withMetroId(ifNotNull(gdAddAddress.getMetroStationId(), Integer::longValue))
                .withManualPoint(toPointOnMap(gdAddAddress.getPointOnMap()));
    }

    public static List<GdAddVcardsPayloadItem> toGdAddVcardsPayloadItems(MassResult<Long> result) {
        List<Long> successfullyUpdatedIds = getSuccessfullyUpdatedIds(result, identity());
        return mapList(successfullyUpdatedIds, vcardId -> new GdAddVcardsPayloadItem().withId(vcardId));
    }

}
