package ru.yandex.direct.grid.processing.service.client.converter;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.IntStreamEx;

import ru.yandex.direct.core.entity.vcard.VcardWorktime;
import ru.yandex.direct.core.entity.vcard.VcardWorktimeUtils;
import ru.yandex.direct.grid.model.GdTime;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdWorkTime;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddWorkTime;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class WorkTimeConverter {

    private WorkTimeConverter() {
    }

    /**
     * Преобразовывает строку в список объектов {@link GdWorkTime}, которые содержат информацию о времени работы
     * Склеивает элементы списка, если время начало и конца работы совпадают. Нужно для удобство отображения на фронте
     */
    static List<GdWorkTime> toGdWorkTimes(String workTime) {
        List<VcardWorktime.DailySchedule> dailySchedules = VcardWorktimeUtils.getDailySchedules(workTime);
        List<GdWorkTime> gdWorkTimes = mapList(dailySchedules, WorkTimeConverter::toGdWorkTime);

        List<GdWorkTime> gluedGdWorkTimeList = new ArrayList<>();

        for (GdWorkTime gdWorkTime : gdWorkTimes) {

            for (GdWorkTime gluedGdWorkTime : gluedGdWorkTimeList) {
                if (canBeGlued(gdWorkTime, gluedGdWorkTime)) {
                    //склеиваем дни недели
                    gluedGdWorkTime.getDaysOfWeek().addAll(gdWorkTime.getDaysOfWeek());
                    //обозначаем, что объект склеен и не нужно добавлять в список gluedGdWorkTimeList
                    gdWorkTime.setDaysOfWeek(null);
                    break;
                }
            }

            if (gdWorkTime.getDaysOfWeek() != null) {
                gluedGdWorkTimeList.add(gdWorkTime);
            }
        }

        return gluedGdWorkTimeList;
    }

    private static boolean canBeGlued(GdWorkTime left, GdWorkTime right) {
        return left.getStartTime().equals(right.getStartTime())
                && left.getEndTime().equals(right.getEndTime());
    }

    private static GdWorkTime toGdWorkTime(VcardWorktime.DailySchedule dailySchedule) {
        List<Integer> daysOfWeek = IntStreamEx
                .rangeClosed(dailySchedule.getDaySince(), dailySchedule.getDayTill())
                .boxed()
                .toList();

        return new GdWorkTime()
                .withDaysOfWeek(daysOfWeek)
                .withStartTime(toGdTime(dailySchedule.getHourSince(), dailySchedule.getMinuteSince()))
                .withEndTime(toGdTime(dailySchedule.getHourTill(), dailySchedule.getMinuteTill()));
    }

    /**
     * Преобразовывает список {@link GdAddWorkTime} в объект {@link VcardWorktime}
     */
    private static VcardWorktime toVcardWorkTime(List<GdAddWorkTime> gdAddWorkTimeList) {
        List<VcardWorktime.DailySchedule> dailySchedules = new ArrayList<>();

        gdAddWorkTimeList.forEach(workTime -> {
            VcardWorktime.DailyScheduleBuilder dailyScheduleBuilder = VcardWorktime.DailyScheduleBuilder.builder()
                    .withHourSince(workTime.getStartTime().getHour())
                    .withMinuteSince(workTime.getStartTime().getMinute())
                    .withHourTill(workTime.getEndTime().getHour())
                    .withMinuteTill(workTime.getEndTime().getMinute());

            dailySchedules.addAll(mapList(workTime.getDaysOfWeek(), dayOfWeek -> dailyScheduleBuilder
                    .withDaySince(dayOfWeek)
                    .withDayTill(dayOfWeek)
                    .build())
            );
        });

        return VcardWorktime.fromDailySchedules(dailySchedules);
    }

    public static String toWorkTime(List<GdAddWorkTime> gdAddWorkTimes) {
        return toVcardWorkTime(gdAddWorkTimes).toEncodedString();
    }

    private static GdTime toGdTime(int hour, int minute) {
        return new GdTime()
                .withHour(hour)
                .withMinute(minute);
    }

}
