package ru.yandex.direct.grid.processing.service.client.validation;

import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.grid.processing.model.client.GdClientMeasurerSystem;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdAddClientMeasurerAccount;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdClientMeasurerSettingsMediascope;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.invalidClientMeasurer;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@ParametersAreNonnullByDefault
public class ClientDataValidationService {
    private static final Logger logger = LoggerFactory.getLogger(ClientDataValidationService.class);

    public static final Validator<GdAddClientMeasurerAccount, Defect> ADD_CLIENT_MEASURER_ACCOUNT_VALIDATOR =
            account -> {
                ModelItemValidationBuilder<GdAddClientMeasurerAccount> vb = ModelItemValidationBuilder.of(account);
                vb.item(GdAddClientMeasurerAccount.SETTINGS)
                        .check(validateSettings(account.getMeasurerSystem()));
                return vb.getResult();
            };

    private ClientDataValidationService() {
    }

    private static Constraint<String, Defect> validateSettings(GdClientMeasurerSystem measurerSystem) {
        return settings -> {
            var defect = invalidClientMeasurer(measurerSystem);
            if (Strings.isNullOrEmpty(settings)) {
                return defect;
            }

            if (measurerSystem == GdClientMeasurerSystem.MEDIASCOPE) {
                return validateMediascopeSettings(settings);
            }

            return null;
        };
    }

    private static Defect validateMediascopeSettings(String settings) {
        var defect = invalidClientMeasurer(GdClientMeasurerSystem.MEDIASCOPE);

        try {
            var mediascopeSettings = fromJson(settings, GdClientMeasurerSettingsMediascope.class);

            if (Strings.isNullOrEmpty(mediascopeSettings.getAccessToken())
                    || Strings.isNullOrEmpty(mediascopeSettings.getRefreshToken())
                    || Strings.isNullOrEmpty(mediascopeSettings.getTmsecPrefix())
                    || mediascopeSettings.getExpiresAt() == null
                    || mediascopeSettings.getExpiresAt() < Instant.now().getEpochSecond()) {
                return defect;
            }

            return null;
        } catch (IllegalArgumentException exc) {
            logger.warn("Invalid client measurer settings", exc);
            return defect;
        }
    }
}
