package ru.yandex.direct.grid.processing.service.client.validation;

import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import ru.yandex.direct.grid.processing.model.cliententity.GdCalloutsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeFilter;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolandingFilter;
import ru.yandex.direct.grid.processing.model.cliententity.GdTurbolandingsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImageFilter;
import ru.yandex.direct.grid.processing.model.cliententity.image.GdImagesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdDeleteCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCallouts;
import ru.yandex.direct.grid.processing.model.cliententity.mutation.GdSaveCalloutsItem;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcardFilter;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdVcardsContainer;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.mutation.GdAddVcards;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.StringDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.datesFromNotAfterTo;
import static ru.yandex.direct.validation.Predicates.each;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
@ParametersAreNonnullByDefault
public class ClientEntityValidationService {
    public static final int MAX_HASH_LENGTH = 22;

    private final GridValidationService gridValidationService;

    public ClientEntityValidationService(GridValidationService gridValidationService) {
        this.gridValidationService = gridValidationService;
    }

    private static final Validator<GdSaveCalloutsItem, Defect> SAVE_CALLOUT_ITEM_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSaveCalloutsItem> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdSaveCalloutsItem.TEXT)
                        .check(notNull())
                        .check(notBlank());
                return vb.getResult();
            };

    private static final Validator<GdSaveCallouts, Defect> SAVE_CALLOUTS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSaveCallouts> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdSaveCallouts.SAVE_ITEMS)
                        .check(notEmptyCollection())
                        .checkEach(notNull(), When.isValid())
                        .checkEachBy(SAVE_CALLOUT_ITEM_VALIDATOR, When.isValid());
                return vb.getResult();
            };

    private static final Validator<GdDeleteCallouts, Defect> DELETE_CALLOUTS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdDeleteCallouts> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdDeleteCallouts.CALLOUT_IDS)
                        .check(notNull())
                        .check(notEmptyCollection(), When.notNull())
                        .checkEach(validId(), When.isValid());
                return vb.getResult();
            };

    private static final Validator<GdCalloutsContainer, Defect> CALLOUTS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdCalloutsContainer> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdCalloutsContainer.SEARCH_BY)
                        .checkBy(GridValidationService.CLIENT_SEARCH_REQUEST_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdTurbolandingFilter, Defect> TURBOLANDINGS_CONTAINER_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdTurbolandingFilter> vb = ModelItemValidationBuilder.of(filter);
                vb.item(GdTurbolandingFilter.TURBOLANDING_ID_IN)
                        .checkBy(GridValidationService.IDS_COLLECTION_VALIDATOR, When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdTurbolandingsContainer, Defect> TURBOLANDINGS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdTurbolandingsContainer> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdTurbolandingsContainer.SEARCH_BY)
                        .checkBy(GridValidationService.CLIENT_SEARCH_REQUEST_VALIDATOR);
                vb.item(GdTurbolandingsContainer.FILTER)
                        .checkBy(TURBOLANDINGS_CONTAINER_FILTER_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<Set<String>, Defect> IMAGE_HASH_VALIDATOR = hashes -> {
        ItemValidationBuilder<Set<String>, Defect> vb = ItemValidationBuilder.of(hashes);
        vb.check(Constraint.fromPredicate(each(StringUtils::isNotBlank), StringDefects.notEmptyString()),
                When.notNull());
        vb.check(Constraint.fromPredicate(each(hash -> hash.length() <= MAX_HASH_LENGTH),
                CollectionDefects.maxStringLength(MAX_HASH_LENGTH)),
                When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdImageFilter, Defect> IMAGES_CONTAINER_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdImageFilter> vb = ModelItemValidationBuilder.of(filter);
                vb.item(GdImageFilter.IMAGE_HASH_IN)
                        .checkBy(IMAGE_HASH_VALIDATOR, When.notNull());
                vb.item(GdImageFilter.HEIGHT)
                        .check(greaterThan(0), When.notNull());
                vb.item(GdImageFilter.WIDTH)
                        .check(greaterThan(0), When.notNull());
                vb.check(minCreateTimeIsNotAfterMaxCreateTime(), When.isValid());
                return vb.getResult();
            };

    private static final Validator<GdImagesContainer, Defect> IMAGES_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdImagesContainer> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdImagesContainer.SEARCH_BY)
                        .checkBy(GridValidationService.CLIENT_SEARCH_REQUEST_VALIDATOR);
                vb.item(GdImagesContainer.FILTER)
                        .checkBy(IMAGES_CONTAINER_FILTER_VALIDATOR, When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdCreativeFilter, Defect> CREATIVES_CONTAINER_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdCreativeFilter> vb = ModelItemValidationBuilder.of(filter);
                vb.item(GdCreativeFilter.CREATIVE_ID_IN)
                        .checkBy(GridValidationService.IDS_COLLECTION_VALIDATOR, When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdCreativesContainer, Defect> CREATIVES_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdCreativesContainer> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdCreativesContainer.SEARCH_BY)
                        .checkBy(GridValidationService.CLIENT_SEARCH_REQUEST_VALIDATOR);
                vb.item(GdCreativesContainer.FILTER)
                        .checkBy(CREATIVES_CONTAINER_FILTER_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdVcardFilter, Defect> VCARD_FILTER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdVcardFilter> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdVcardFilter.VCARD_ID_IN)
                        .check(notEmptyCollection())
                        .checkBy(GridValidationService.IDS_COLLECTION_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdVcardsContainer, Defect> VCARDS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdVcardsContainer> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdVcardsContainer.SEARCH_BY)
                        .checkBy(GridValidationService.CLIENT_SEARCH_REQUEST_VALIDATOR);
                vb.item(GdVcardsContainer.FILTER)
                        .checkBy(VCARD_FILTER_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdAddVcards, Defect> ADD_VCARDS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdAddVcards> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdAddVcards.VCARD_ADD_ITEMS)
                        .check(notEmptyCollection());
                return vb.getResult();
            };

    public void validateSaveCalloutsRequest(GdSaveCallouts saveCallouts) {
        gridValidationService.applyValidator(SAVE_CALLOUTS_VALIDATOR, saveCallouts, false);
    }

    public void validateDeleteCalloutsRequest(GdDeleteCallouts deleteCallouts) {
        gridValidationService.applyValidator(DELETE_CALLOUTS_VALIDATOR, deleteCallouts, false);
    }

    public void validateGdCalloutsContainer(GdCalloutsContainer input) {
        gridValidationService.applyValidator(CALLOUTS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdTurbolandingsContainer(GdTurbolandingsContainer input) {
        gridValidationService.applyValidator(TURBOLANDINGS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdCreativesContainer(GdCreativesContainer input) {
        gridValidationService.applyValidator(CREATIVES_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdImagesContainer(GdImagesContainer input) {
        gridValidationService.applyValidator(IMAGES_CONTAINER_VALIDATOR, input, false);
    }

    public void validateGdVcardsContainer(GdVcardsContainer input) {
        gridValidationService.applyValidator(VCARDS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateAddVcardsRequest(GdAddVcards input) {
        gridValidationService.applyValidator(ADD_VCARDS_VALIDATOR, input, false);
    }

    private static Constraint<GdImageFilter, Defect> minCreateTimeIsNotAfterMaxCreateTime() {
        Predicate<GdImageFilter> predicate =
                filter -> filter.getMinCreatedDateTime() == null || filter.getMaxCreatedDateTime() == null ||
                        !filter.getMinCreatedDateTime().isAfter(filter.getMaxCreatedDateTime());
        return Constraint.fromPredicate(predicate, datesFromNotAfterTo());
    }
}
