package ru.yandex.direct.grid.processing.service.communication;

import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationActionInput;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationActionResult;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationInput;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageGroupList;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageList;
import ru.yandex.direct.grid.processing.model.communication.SlotName;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class CommunicationGraphQlService {
    public static final String COMMUNICATION_MESSAGE_LIST_RESOLVER_NAME = "communicationMessageList";
    public static final String COMMUNICATION_MESSAGE_GROUP_LIST_RESOLVER_NAME = "communicationMessageGroupList";
    public static final String COMMUNICATION_ACTION_RESOLVER_NAME = "communicationAction";

    public static final String AUTO_APPLIED_RECOMMENDATION_LIST_RESOLVER_NAME = "autoAppliedRecommendationList";

    private final CommunicationInfoService communicationInfoService;
    private final GridValidationService gridValidationService;

    @Autowired
    public CommunicationGraphQlService(CommunicationInfoService communicationInfoService, GridValidationService gridValidationService) {
        this.communicationInfoService = communicationInfoService;
        this.gridValidationService = gridValidationService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = COMMUNICATION_MESSAGE_LIST_RESOLVER_NAME)
    public GdCommunicationMessageList getCommunicationMessageList(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCommunicationInput input) {
        GdCommunicationMessageList communicationMessageList = communicationInfoService.getCommunicationMessageList(
                context.getSubjectUser().getClientId(),
                context.getOperator().getUid(),
                mergeSlotNames(input.getSlots(), input.getSlotNames()),
                input.getTargetObjects(),
                input.getInterfaceLanguage());
        return communicationMessageList;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = COMMUNICATION_MESSAGE_GROUP_LIST_RESOLVER_NAME)
    public GdCommunicationMessageGroupList getCommunicationMessageGroupList(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCommunicationInput input) {
        return communicationInfoService.getCommunicationMessageGroupList(
                context.getSubjectUser().getClientId(),
                context.getOperator().getUid(),
                mergeSlotNames(input.getSlots(), input.getSlotNames()),
                input.getTargetObjects(),
                input.getInterfaceLanguage()
        );
    }

    private List<String> mergeSlotNames(List<String> slots, List<SlotName> slotNames) {
        return StreamEx.of(slotNames == null ? Collections.emptyList() : slotNames)
                .map(SlotName::getTypedValue)
                .append(slots == null ? Collections.emptyList() : slots)
                .distinct()
                .toList();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = COMMUNICATION_ACTION_RESOLVER_NAME)
    @GraphQLMutation(name = COMMUNICATION_ACTION_RESOLVER_NAME)
    public GdCommunicationActionResult confirmAction(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCommunicationActionInput input
    ) {
        return communicationInfoService.confirmAction(
                context.getSubjectUser().getClientId(),
                context.getOperator().getUid(),
                input.getRequestId(),
                input.getButtonId(),
                input.getSlot() != null ? input.getSlot() : input.getSlotName().getTypedValue(),
                input.getTargets(),
                input.getInterfaceLanguage()
        );
    }

    @GraphQLNonNull
    @GraphQLQuery(name = AUTO_APPLIED_RECOMMENDATION_LIST_RESOLVER_NAME)
    public GdCommunicationMessageList getAutoAppliedRecommendationList(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCommunicationInput input) {
        GdCommunicationMessageList communicationMessageList = communicationInfoService.getAutoAppliedRecommendationList(
                context.getSubjectUser().getClientId(),
                context.getOperator().getUid(),
                mergeSlotNames(input.getSlots(), input.getSlotNames()),
                input.getTargetObjects(),
                input.getInterfaceLanguage());
        return communicationMessageList;
    }

}
